(*  Title: 	tactic
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Tactics and tacticals
Derived rules and other operations on theorems and theories

*)

infix RS RSN;
infix RES RESN;
infix THEN ORELSE APPEND THEN' ORELSE' APPEND';

signature TACTIC =
sig
  structure Thm : THM
  structure Stringtree : STRINGTREE
  datatype tactic = Tactic of Thm.thm -> Thm.thm Thm.Sequence.seq
  val ALLGOALS: (int -> tactic) -> tactic   
  val all_tac: tactic
  val APPEND: tactic * tactic -> tactic
  val APPEND': ('a -> tactic) * ('a -> tactic) -> 'a -> tactic
  val ares_tac: Thm.thm list -> int -> tactic   
  val assume_ax: Thm.theory -> string -> Thm.thm
  val assume_tac: int -> tactic   
  val BEST_FIRST: (Thm.thm -> bool) * (Thm.thm -> int) -> tactic -> tactic
  val biresolve_tac: (bool*Thm.thm)list -> int -> tactic
  val BREADTH_FIRST: (Thm.thm -> bool) -> tactic -> tactic
  val compat_resolve_tac: Thm.thm list -> int -> int -> tactic   
  val compat_thms: 'a Stringtree.tree * term -> 'a list   
  val compose_inst_tac: (string*string*typ)list -> (bool*Thm.thm*int) -> int -> tactic   
  val compose_tac: (bool * Thm.thm * int) -> int -> tactic 
  val COND: (Thm.thm -> bool) -> tactic -> tactic -> tactic   
  val cond_infer: (Thm.thm -> Thm.thm) -> Thm.thm -> Thm.thm   
  val cut_facts_tac: Thm.thm list -> int -> tactic
  val DEPTH_FIRST: (Thm.thm -> bool) -> tactic -> tactic
  val DEPTH_SOLVE_1: tactic -> tactic
  val DETERM: tactic -> tactic
  val eq_assume_tac: int -> tactic   
  val eresolve_tac: Thm.thm list -> int -> tactic
  val eres_inst_tac: (string*string*typ)list -> Thm.thm -> int -> tactic   
  val EVERY: tactic list -> tactic   
  val filter_thms: (term*term->bool) -> int*term*Thm.thm list -> Thm.thm list
  val filt_resolve_tac: Thm.thm list -> int -> int -> tactic
  val FIRST: tactic list -> tactic   
  val FIRSTGOAL: (int -> tactic) -> tactic
  val forall_intr_list: Thm.Sign.cterm list -> Thm.thm -> Thm.thm
  val forall_intr_frees: Thm.thm -> Thm.thm
  val forall_elim_var: int -> Thm.thm -> Thm.thm
  val forall_elim_vars: int -> Thm.thm -> Thm.thm
  val forwards_tac: Thm.thm -> int -> tactic   
  val has_fewer_prems: int -> Thm.thm -> bool   
  val head_string: term -> string list
  val insert_thm: Thm.thm * Thm.thm Stringtree.tree -> Thm.thm Stringtree.tree
  val lift_inst_rule: Thm.thm * int * (string*string*typ)list * Thm.thm -> Thm.thm
  val lift_inst_tac: (string*string*typ)list -> Thm.thm -> int -> tactic   
  val MAP_EVERY: ('a -> tactic) -> 'a list -> tactic
  val MAP_FIRST: ('a -> tactic) -> 'a list -> tactic
  val metacut_tac: Thm.thm -> int -> tactic   
  val no_tac: tactic
  val ORELSE: tactic * tactic -> tactic
  val ORELSE': ('a -> tactic) * ('a -> tactic) -> 'a -> tactic
  val prepare_proof: Thm.theory -> string -> Thm.thm list * Thm.thm * (bool*Thm.thm -> Thm.thm)   
  val PRIMITIVE: (Thm.thm -> Thm.thm) -> tactic  
  val print_goal_thm: int -> Thm.thm -> unit
  val print_thm: Thm.thm -> unit
  val prth: Thm.thm -> unit
  val prths: Thm.thm list -> unit
  val prove_goal: Thm.theory -> string -> (Thm.thm list -> tactic list) -> Thm.thm
  val read_instantiate: Thm.Sign.sg -> (string*string*typ)list -> Thm.thm -> Thm.thm
  val REPEAT1: tactic -> tactic
  val REPEAT: tactic -> tactic
  val REPEAT_FIRST: (int -> tactic) -> tactic
  val REPEAT_SOME: (int -> tactic) -> tactic
  val RES: Thm.thm * Thm.thm -> Thm.thm
  val RESN: Thm.thm * (int * Thm.thm list) -> Thm.thm
  val reslist: Thm.thm list * int * Thm.thm -> Thm.thm list
  val resolve_tac: Thm.thm list -> int -> tactic
  val res_inst_tac: (string*string*typ)list -> Thm.thm -> int -> tactic   
  val RS: Thm.thm * Thm.thm -> Thm.thm
  val RSN: Thm.thm list * (int * Thm.thm) -> Thm.thm
  val rv_assume: Thm.Sign.sg -> string -> Thm.thm
  val SELECT_GOAL: tactic -> int -> tactic
  val smash_all_ff_tac: tactic
  val SOMEGOAL: (int -> tactic) -> tactic   
  val standard: Thm.thm -> Thm.thm
  val STATE: (Thm.thm -> tactic) -> tactic
  val SUBGOAL: ((term*int) -> tactic) -> int -> tactic
  val subgoal_tac: string -> int -> tactic
  val tapply: tactic * Thm.thm -> Thm.thm Thm.Sequence.seq
  val THEN: tactic * tactic -> tactic
  val THEN': ('a -> tactic) * ('a -> tactic) -> 'a -> tactic
  val trace_tac: int -> tactic
  val TRY: tactic -> tactic
  val TRYALL: (int -> tactic) -> tactic   
  val zero_var_indexes: Thm.thm -> Thm.thm
end;


functor TacticFun (structure Logic: LOGIC and Thm: THM and
	Stringtree: STRINGTREE and Pretty: PRETTY) : TACTIC = 
struct
structure Thm = Thm;
structure Stringtree = Stringtree;
structure Sequence = Thm.Sequence;
structure Sign = Thm.Sign;
local open Thm
in


(**** More derived rules and operations on theorems ****)

(*Generalization over a list of variables, IGNORING bad ones*)
fun forall_intr_list [] th = th
  | forall_intr_list (y::ys) th =
	let val gth = forall_intr_list ys th
	in  forall_intr y gth   handle THM _ =>  gth  end;


(*Generalization over all suitable Free variables*)
fun forall_intr_frees th =
    let val {prop,sign,...} = rep_thm th
    in  forall_intr_list
         (map (Sign.cterm_of sign) 
	      (sort Logic.atless (Logic.add_term_frees (prop,[])))) th
    end;


(*Replace outermost quantified variable by Var of given index.
    Could clash with Vars already present.*)
fun forall_elim_var i th = 
    let val {prop,sign,...} = rep_thm th
    in case prop of
	  Const("all",_) $ Abs(a,T,_) =>
	      forall_elim (Sign.cterm_of sign (Var((a,i), T)))  th
	| _ => raise THM("forall_elim_var", i, [th])
    end;


(*Repeat forall_elim_var until all outer quantifiers are removed*)
fun forall_elim_vars i th = 
    forall_elim_vars i (forall_elim_var i th)
	handle THM _ => th;


(*maps  (bs,v)  to   v'::bs    this reverses the identifiers bs*)
fun add_new_id (bs, ((a,_),_)) : string list =  variant bs a  ::  bs;


(*Pair variables with their new names, restoring types*)
fun varpairs sign ([],[]) = []
  | varpairs sign ((var as Var(_,T)) :: vars, b::bs) =
	(Sign.cterm_of sign var, Sign.cterm_of sign (Var((b,0),T))) :: 
        varpairs sign (vars,bs)
  | varpairs sign _ = raise TERM_ERROR("varpairs", []);


(*Reset Var indexes to zero, renaming to preserve distinctness*)
fun zero_var_indexes th = 
    let val {prop,sign,...} = rep_thm th;
        val vars = Logic.add_term_vars(prop,[])
        val bs = itlist_left add_new_id ([], map dest_Var vars)
    in  instantiate (varpairs sign (vars, rev bs)) th  end;


(*Standard form of object-rule: no hypotheses, Frees, or outer quantifiers;
    all generality expressed by Vars having index 0.*)
fun standard th =
    let val {maxidx,...} = rep_thm th
    in  zero_var_indexes (forall_elim_vars(maxidx+1) 
                         (forall_intr_frees(implies_intr_hyps th)))
    end;


(*Assume a new formula, read following the same conventions as axioms. 
  Generalizes over Free variables,
  creates the assumption, and then strips quantifiers.
  Example is [| ALL x:?A. ?P(x) |] ==> [| ?P(?a) |]
	     [ !(A,P,a)[| ALL x:A. P(x) |] ==> [| P(a) |] ]    *)
fun assume_ax thy sP =
    let val sign = sign_of thy;
	val prop = Logic.close_form (Sign.term_of (Sign.read_cterm sign
			 (sP, Aprop)))
    in forall_elim_vars 0 (assume (Sign.cterm_of sign prop))  end;


(*Resolution: exactly one resolvent must be produced.  NEW*) 
fun ths RSN (i,thb) = 
  case Sequence.chop (2, resolution (thb,i,ths)) of
      ([th],_) => th
    |      _   => raise THM("RSN", i, thb::ths);


(*resolution: P==>Q, Q==>R gives P==>R.  (NEW: args reversed from RES!) *)
fun tha RS thb = [tha] RSN (1,thb);


(*Resolution: exactly one resolvent must be produced.  OBSOLETE FORM*) 
fun th2 RESN (i,ths) = 
  case Sequence.chop (2, resolution (th2,i,ths)) of
      ([th],_) => th
    |      _   => raise THM("RESN", i, th2::ths);


(*resolution of first premise with a single theorem.  OBSOLETE FORM*)
fun th2 RES th1 = th2 RESN (1,[th1]);


(*List of all resolvents of ths with thb at i*)
fun reslist (ths, i,thb) = 
  Sequence.list_of_s (resolution (thb,i,ths));


(*Instantiate rule, reading terms from strings under signature "sign".
  Resulting rule will belong to union of the two signatures. *)
fun read_instantiate sign sinsts th =
  instantiate (Sign.read_insts sign sinsts) th;


(*Assume a term read from a string, strips outer quantifiers giving Vars.
  For assuming an object-rule during a proof.  *)
fun rv_assume sign sP =
  forall_elim_vars 0 (assume (Sign.read_cterm sign (sP,Aprop)));


(*If condth has the form A==>B then recurs on B assuming A.
  It finally discharges these assumptions from result of applying mrule.*)
fun cond_infer mrule condth =
  case #prop(rep_thm condth) of
	imp$A$B => 
	  if imp=implies then  
	    let val cA = Sign.cterm_of (#sign(rep_thm condth)) A
	    in  implies_intr cA
		  (cond_infer mrule (implies_elim condth (assume cA)))
	    end
	  else mrule condth
    | _ => mrule condth;


(** Printing of theorems **)

(*Print a meta-theorem.  To be called by other printing operations. *)
fun print_thm th : unit =
  let val {sign, hyps, prop,...} = rep_thm th
  in  Pretty.bg 0;  Sign.print_term sign prop;  
      if null hyps then ()
      else  (Pretty.brk(2,0);
	     Pretty.list ("[ ", " ]", Sign.print_term sign) hyps);
      Pretty.en()
  end;


(*Print a meta-theorem.  Top-level command. *)
fun prth th = (Pretty.init();  print_thm th;  Pretty.nl());

(*Print a list of theorems, separated by blank lines*)
val prths = print_list_ln prth;


(*Print thm A1,...,An/B in "goal style" -- premises as numbered subgoals*)
fun print_goal_thm maxgoals th : unit =
  let val {sign, hyps, prop,...} = rep_thm th;
      fun printgoals (_, []) = ()
        | printgoals (n, A::As) =
             (Pretty.st (" "  ^ string_of_int n  ^  ". ");  
              Sign.print_term sign A;  Pretty.nl();
              printgoals (n+1,As));
      fun printpair (t,u) =
	(Sign.print_term sign t;  Pretty.st" =";  Pretty.brk(1,0);
	 Sign.print_term sign u);
      fun printff [] = ()
        | printff tpairs = 
	    (prs"\nFlex-flex pairs:\n";  
	     Pretty.list("", "", printpair) tpairs;  Pretty.nl());
      val (tpairs,As,B) = Logic.strip_horn(prop);
      val ngoals = length As
  in Pretty.init();  Sign.print_term sign B;  Pretty.nl();
     if ngoals=0  then prs"No subgoals!\n"
       else if ngoals>maxgoals 
       then (printgoals (1, front(maxgoals,As));
	     prs("A total of " ^ string_of_int ngoals ^ " subgoals...\n"))
       else printgoals (1, As);
     printff tpairs
  end;


(**** Tactics ****)


(*A tactic maps a proof tree to a sequence of proof trees:
    if length of sequence = 0 then the tactic does not apply;
    if length > 1 then backtracking on the alternatives can occur.*)

datatype tactic = Tactic of thm -> thm Sequence.seq;


fun tapply(Tactic tf, state) = tf (state);


(*** LCF-style tacticals ***)

(*the tactical THEN performs one tactic followed by another*)
fun Tactic tf1  THEN  Tactic tf2 = 
  Tactic (fn state => Sequence.flats (Sequence.maps tf2 (tf1 state)));


(*The tactical ORELSE uses the first tactic that returns a nonempty sequence.
  Like in LCF, ORELSE commits to either tac1 or tac2 immediately.
  Does not backtrack to tac2 if tac1 was initially chosen. *)
fun Tactic tf1  ORELSE  Tactic tf2 = 
  Tactic (fn state =>  
    case Sequence.pull(tf1 state) of
	None       => tf2 state
      | sequencecell => Sequence.seqof(fn()=> sequencecell));


(*The tactical APPEND combines the results of two tactics.
  Like ORELSE, but allows backtracking on both tac1 and tac2.
  The tactic tac2 is not applied until needed.*)
fun Tactic tf1  APPEND  Tactic tf2 = 
  Tactic (fn state =>  Sequence.append(tf1 state,
                          Sequence.seqof(fn()=> Sequence.pull (tf2 state))));


(*Versions for combining tactic-valued functions, as in
     SOMEGOAL (resolve_tac rls THEN' assume_tac) *)
fun tac1 THEN' tac2 = fn x => tac1 x THEN tac2 x;
fun tac1 ORELSE' tac2 = fn x => tac1 x ORELSE tac2 x;
fun tac1 APPEND' tac2 = fn x => tac1 x APPEND tac2 x;

(*passes all proofs through unchanged;  identity of THEN*)
val all_tac = Tactic (fn state => Sequence.single state);

(*passes no proofs through;  identity of ORELSE and APPEND*)
val no_tac  = Tactic (fn state => Sequence.null);


(*Make a tactic deterministic by chopping the tail of the proof sequence*)
fun DETERM (Tactic tf) = Tactic (fn state => 
      case Sequence.pull (tf state) of
	      None => Sequence.null
            | Some(x,_) => Sequence.cons(x, Sequence.null));


(*Conditional tactical: testfun controls which tactic to use next.
  Beware: due to eager evaluation, both thentac and elsetac are evaluated.*)
fun COND testfun (Tactic thenf) (Tactic elsef) = Tactic (fn prf =>
    if testfun prf then  thenf prf   else  elsef prf);

(*Do the tactic or else do nothing*)
fun TRY tac = tac ORELSE all_tac;


(*Performs no backtracking: quits when it gets stuck
  The abstraction over state is essential: else it would always loop!!*)
fun REPEAT tac = Tactic (fn state =>
  tapply((tac THEN REPEAT tac) ORELSE all_tac, state));

(*Repeat 1 or more times*)
fun REPEAT1 tac = Tactic (fn state => 
  tapply(tac THEN (REPEAT1 tac ORELSE all_tac), state));


(* EVERY [tac1,...,tacn]   equals    tac1 THEN ... THEN tacn   *)
fun EVERY tacs = itlist_right (op THEN) (tacs, all_tac);


(* FIRST [tac1,...,tacn]   equals    tac1 ORELSE ... ORELSE tacn   *)
fun FIRST tacs = itlist_right (op ORELSE) (tacs, no_tac);

(*Tacticals that apply tacf to all elements of xs*)
fun MAP_EVERY tacf xs = EVERY (map tacf xs);
fun MAP_FIRST tacf xs = FIRST (map tacf xs);


(*** Tacticals based on subgoal numbering ***)


(*For n subgoals, performs tf(n) THEN ... THEN tf(1) 
  Essential to work backwards since tf(i) may add/delete subgoals at i. *)
fun ALLGOALS tf = 
  let fun tac 0 = all_tac
	| tac n = tf(n) THEN tac(n-1)
  in  Tactic(fn state => tapply(tac(length(prems_of state)), state))  end;


(*For n subgoals, performs tf(n) ORELSE ... ORELSE tf(1)  *)
fun SOMEGOAL tf = 
  let fun tac 0 = no_tac
	| tac n = tf(n) ORELSE tac(n-1)
  in  Tactic(fn state => tapply(tac(length(prems_of state)), state))  end;


(*For n subgoals, performs tf(1) ORELSE ... ORELSE tf(n).
  More appropriate than SOMEGOAL in some cases.*)
fun FIRSTGOAL tf = 
  let fun tac (i,n) = if i>n then no_tac else  tf(i) ORELSE tac (i+1,n)
  in  Tactic(fn state => tapply(tac(1, length(prems_of state)), state))  end;

(*Repeatedly solve some using tf. *)
fun REPEAT_SOME tf = REPEAT1 (SOMEGOAL (REPEAT1 o tf));

(*Repeatedly solve the first possible subgoal using tf. *)
fun REPEAT_FIRST tf = REPEAT1 (FIRSTGOAL (REPEAT1 o tf));

(*For n subgoals, tries to apply tf to n,...1  *)
fun TRYALL tf = ALLGOALS (TRY o tf);


(*Discover what goal is chosen:  SOMEGOAL(tac THEN' trace_tac) *)
fun trace_tac i = Tactic (fn state => 
   (prs("Subgoal " ^ string_of_int i ^ " selected\n"); 
    Sequence.single state));


(*Make a tactic for subgoal i, if there is one.  *)
fun SUBGOAL goalfun i = Tactic(fn state =>
  case nth_tail(i-1, prems_of state) of
      [] => Sequence.null
    | prem::_ => tapply(goalfun (prem,i), state));


(*** Basic tactics ***)


(*Makes a tactic whose action on a state is the mapping thmfun.*)
fun PRIMITIVE thmfun = Tactic (fn state => Sequence.single (thmfun state));


(*Makes a tactic from one that uses the components of the state.*)
fun STATE tacfun = Tactic (fn state => tapply(tacfun state, state));


(*** The following fail if the goal number is out of range:
     thus (REPEAT (resolve_tac rules i)) stops once subgoal i disappears. *)


(*The composition rule/state: no lifting or var renaming.
  The arg = (bires_flg, orule, m) ;  see bicompose for explanation.*)
fun compose_tac arg i = Tactic (fn state =>
    bicompose(state, i)arg  handle THM _ => Sequence.null);

(*Solve subgoal i by assumption*)
fun assume_tac i = Tactic (fn state => assumption(state,i)
	handle THM _ => Sequence.null);

(*Solve subgoal i by assumption, using no unification*)
fun eq_assume_tac i = Tactic
    (fn state => Sequence.single (eq_assumption(state,i))
	handle THM _ => Sequence.null);

(*Resolution: the simple case, works for introduction rules*)
fun resolve_tac rules i = Tactic (fn state => resolution(state,i,rules)
	handle THM _ => Sequence.null);

(*Attack subgoal i by resolution, using flags to indicate elimination rules*)
fun biresolve_tac brules i = Tactic (fn state=> biresolution(state,i,brules)
	handle THM _ => Sequence.null);

(*Resolution with elimination rules only*)
fun eresolve_tac rules = biresolve_tac (map (pair true) rules);


(*Lift and instantiate a rule wrt the given state and subgoal number *)
fun lift_inst_rule (state, i, sinsts, rule) =
  let val {maxidx,sign,...} = rep_thm state;
      fun cfun f = (Sign.cterm_of sign) o f o Sign.term_of;
      val (_, _, Bi, _) = dest_state(state,i);
      val params = rev (#2 (Logic.strip_assums Bi));   (*params of subgoal i*)
      fun liftvar (Var ((a,j), T)) = Var((a, j+maxidx+1), map snd params--->T)
	| liftvar t = raise TERM_ERROR("lift_inst: Var expected", [t]);
      fun liftterm t = list_abs_free (params, t);
      (*Lifts instantiation pair over params*)
      fun liftpair (cv,ct) = (cfun liftvar cv, cfun liftterm ct);
      val insts = Sign.read_insts (#sign(rep_thm state)) sinsts
  in  instantiate (map liftpair insts)  (lift_rule (state,i) rule)
  end;


(*** Resolve after lifting and instantation; may refer to parameters of the
     subgoal.  Fails if "i" is out of range.  WARNING: Variables in the given
     terms will be incremented!  CAN THIS BE PREVENTED?? ***)

(*compose version: arguments are as for bicompose.*)
fun compose_inst_tac sinsts (bires_flg, rule, nsubgoal) i =
  STATE (fn state => 
    compose_tac (bires_flg, lift_inst_rule (state, i, sinsts, rule),
		 nsubgoal) i
    handle TERM_ERROR _ => no_tac
   	 | THM _ => no_tac);

(*Resolve version*)
fun res_inst_tac sinsts rule i =
    compose_inst_tac sinsts (false, rule, length(prems_of rule)) i;

(*eresolve (elimination) version*)
fun eres_inst_tac sinsts rule i =
    compose_inst_tac sinsts (true, rule, length(prems_of rule)) i;

(*older name: for compatibility*)
val lift_inst_tac = res_inst_tac;



(*Used by forwards_tac and metacut_tac*)
fun bires_cut_tac arg i =
    resolve_tac [cut_rl] i  THEN  biresolve_tac arg (i+1) ;

(*Forwards reasoning. 
  Rule has the form A==>B, infers forwards from assumption A.*)
fun forwards_tac rule = bires_cut_tac [(true,rule)];

(*The conclusion of the rule gets assumed in subgoal i,
  while subgoal i+1,... are the premises of the rule.*)
fun metacut_tac rule = bires_cut_tac [(false,rule)];

(*Recognizes theorems that are not rules, but simple propositions*)
fun is_fact th =
    case prems_of th of
	[] => true  |  _::_ => false;

(*"Cut" all facts from theorem list into the goal as assumptions. *)
fun cut_facts_tac ths i =
    MAP_EVERY (fn th => metacut_tac th i) (filter is_fact ths);

(*Introduce the given proposition as a lemma and subgoal*)
fun subgoal_tac sprop = res_inst_tac [("psi", sprop, Aprop)] cut_rl;


(*Use an assumption or some rules ... A popular combination!*)
fun ares_tac rules = assume_tac  ORELSE'  resolve_tac rules;



(*Tactical for restricting the effect of a tactic to subgoal i.
  Works by making a new state from subgoal i, applying tf to it, and
  composing the resulting metathm with the original state.
  The "main goal" of the new state will not be atomic, some tactics may fail!
  DOES NOT work if tactic affects the main goal other than by instantiation.  *)
fun SELECT_GOAL (Tactic tf) i = Tactic (fn state => 
  case nth_tail(i-1, prems_of state) of
      [] => Sequence.null
    | prem::_ => 
	let val st0 = trivial (Sign.cterm_of (#sign(rep_thm state)) prem);
	    val nasms = length(Logic.strip_imp_prems prem);
	    fun next st = bicompose(state, i) (false, st,
				length(prems_of st) - nasms)
	in  Sequence.flats (Sequence.maps next (tf st0))
	end);

(*Smash all flex-flex disagreement pairs*)
val smash_all_ff_tac =
    Tactic(fn state => unify_instantiate (tpairs_of state) state);


(*** Setting up goal-directed proof ***)


(* maps [A1,...,An], B  to   A1==>...==>An==>B  *)
fun implies_intr_list [] thB = thB
  | implies_intr_list (cA::cAs) thB =
	implies_intr cA (implies_intr_list cAs thB);      


(*Common treatment of "goal" and "prove_goal":
  Return assumptions, initial proof state, and function to make result. *)
fun prepare_proof thy agoal =
  let val chorn = Sign.type_assign (Sign.read_cterm(sign_of thy)(agoal,Aprop));
      val {sign, t=horn,...} = Sign.rep_cterm chorn;
      val (_,As,B) = Logic.strip_horn(horn);
      val cAs = map (Sign.cterm_of sign) As;
      fun result_error msg = error
	("result: " ^ msg ^ "\nGoal was " ^ agoal ^ "\n");
      (*discharges assumptions from state in the order they appear in goal;
	checks (if requested) that resulting theorem is equivalent to goal. *)
      fun mkresult (check,state) =
        let val ngoals = length (prems_of state);
            val th = implies_intr_list cAs state;
            val {hyps,prop,...} = rep_thm th
        in  if ngoals>0 then result_error 
		(string_of_int ngoals ^ " unsolved goals!")
            else  case hyps of
      	        [] => if (not check) orelse (prop aconv Sign.term_of chorn)
		      then  standard th 
		      else  result_error "proved a different theorem"
	      | _::_ => result_error "additional hypotheses"
        end;
      val asms = map (forall_elim_vars 0  o  assume) cAs
      and st0 = trivial (Sign.cterm_of sign B)
  in  (asms, st0, mkresult)  end;


(*Prove theorem using the tactics in sequence; check it has the specified form.
  Augments signature with all type assignments of goal.
  Syntax is similar to "goal" command for easy keyboard use.*)
fun prove_goal thy agoal tacsf =
  let val (asms, st0, mkresult) = prepare_proof thy agoal;
      val tac = EVERY (tacsf asms);
      val state = (case Sequence.pull (tapply(tac,st0)) of Some(st,_) => st
		 | _ => error ("prove_goal: tactic failed\n" ^ agoal ^ "\n"))
  in  mkresult (true,state)  end
  handle ERROR => error (*from type_assign, etc via prepare_proof*)
	    ("The above error occurred for " ^ agoal ^ "\n")
       | _ => error ("prove_goal: exception was raised!\n " ^ agoal ^ "\n");


(*** Search tacticals ***)

(*Repeats again and again until "satpred" reports proof tree as satisfied*)
fun DEPTH_FIRST satpred tac = STATE (fn state => 
   if satpred state  then  all_tac 
                     else  tac THEN DEPTH_FIRST satpred tac);


(*Predicate: Does the rule have fewer than n premises?*)
fun has_fewer_prems n rule =  (length (prems_of rule) < n);


(*Tactical to reduce the number of premises by 1.
  If no subgoals then it must fail! *)
fun DEPTH_SOLVE_1 tac = STATE
 (fn state => 
    (case length(prems_of state) of
	0 => no_tac
      | n => DEPTH_FIRST (has_fewer_prems n) tac));


(*Two theorems are taken as equal if their propositions are a-convertible*)
fun eq_thm (th1,th2) = #prop(rep_thm th1) aconv #prop(rep_thm th2); 

(*Insertion into priority queue of states *)
fun insert (nth: int*thm, []) = [nth]
  | insert ((m,th), (n,th')::nths) = 
      if  n<m then (n,th') :: insert ((m,th), nths)
      else if  n=m andalso eq_thm(th,th')
              then (n,th')::nths
              else (m,th)::(n,th')::nths;

(*Best-first search for a state that satisfies satpred (incl initial state)
  Function sizef estimates size of problem remaining (smaller means better) *)
fun BEST_FIRST (satpred,sizef) (Tactic tf) = 
  let fun pairsize th = (sizef th, th);
      fun bfs (news,nprfs) =
	   (case  partition satpred news  of
		([],nonsats) => next(itlist_right insert
					(map pairsize nonsats, nprfs)) 
	      | (sats,_)  => sats)
      and next [] = []
        | next ((n,prf)::nprfs) =
	    (prs("size=" ^ string_of_int n ^ "\n");
	       bfs (Sequence.list_of_s (tf prf), nprfs))
  in Tactic (fn state => Sequence.s_of_list (bfs ([state],[]))) end;


(*Breadth-first search to satisfy satpred (including initial state) *)
fun BREADTH_FIRST satpred (Tactic tf) = 
  let val tacf = Sequence.list_of_s o tf;
      fun bfs prfs =
	 (case  partition satpred prfs  of
	      ([],[]) => []
	    | ([],nonsats) => 
		  (prs("breadth=" ^ string_of_int(length nonsats) ^ "\n");
		   bfs (flat (map tacf nonsats)))
	    | (sats,_)  => sats)
  in Tactic (fn state => Sequence.s_of_list (bfs [state])) end;


(*** Indexing and filtering of theorems ***)

(*string of symbols following first argument of combinations
  symbols are constants, parameters, bound vars
  example:  rec(succ(x),...,...)  has   [ "rec", "succ", "*param*" ]  *)
fun head_string t : string list =
  let val (head,args) = strip_comb t;
      fun tail_string [] = []
	| tail_string(arg::_) = head_string arg
  in  case head of
      Const(name,_) =>     name :: tail_string args
    | Free (name,_) => "*"^name :: tail_string args
    | Bound bno => "B."^chr(bno+ord"0") :: tail_string args
    | _ => []
  end;


(*insert a thm in a thm tree using its head_string*)
fun insert_thm (th,rtr) =
  Stringtree.insert((head_string (concl_of th), th), rtr);

(*returns the list of theorems in the tree that are compatible with prem*)
fun compat_thms (rtr,prem) =
   Stringtree.compat(rtr, head_string (Logic.strip_assums_concl prem));


(*Resolve subgoal i using the tree of rules, unless too flexible --
   which means: >maxr are compatible.      *)
fun rtr_resolve_tac rtr maxr = SUBGOAL(fn (prem,i) =>
    let val rls = compat_thms (rtr,prem)
    in  if length rls > maxr  then no_tac  else resolve_tac rls i
    end);

(*For efficiency, bind its result to a tactic.  It builds a stringtree.*)
fun compat_resolve_tac rules = 
  rtr_resolve_tac (itlist_right insert_thm (rules, Stringtree.null));
 

(*Returns the list of potentially resolvable theorems for the goal "prem",
	using the predicate  could(subgoal,concl).
  Resulting list is no longer than "limit"*)
fun filter_thms could (limit, prem, ths) =
  let val pb = Logic.strip_assums_concl prem;   (*delete assumptions*)
      fun filtr (limit, []) = []
	| filtr (limit, th::ths) =
	    if limit=0 then  []
	    else if could(pb, concl_of th)  then th :: filtr(limit-1, ths)
	    else filtr(limit,ths)
  in  filtr(limit,ths)  end;


(*Resolve subgoal i using the rules, unless >maxr are compatible. *)
fun filt_resolve_tac rules maxr = SUBGOAL(fn (prem,i) =>
  let val rls = filter_thms could_unify (maxr+1, prem, rules)
  in  if length rls > maxr  then  no_tac  else resolve_tac rls i
  end);

end;
end;

