(*  Title: 	library
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Basic library: lists, pairs, etc
*)


exception LIST of string;

(*discriminator and selectors for lists. *)
fun null   []   = true
  | null (_::_) = false;

fun hd   []   = raise LIST "hd"
  | hd (a::_) = a;

fun tl   []   = raise LIST "tl"
  | tl (_::l) = l;


(*functional for pairs*)
fun pair x y = (x,y);

fun fst(x,y) = x and snd(x,y) = y;

(*Handy combinators*)
fun curry f x y = f(x,y);
fun uncurry f(x,y) = f x y;
fun I x = x  and  K x y = x;

(*Combine two functions forming the union of their domains*)
infix orelf;
fun f orelf g = fn x => f x  handle Match=> g x;


(*Application of (infix) operator to its left or right argument*)
fun apl (x,f) y = f(x,y);
fun apr (f,y) x = f(x,y);


(*functional for pairs*)
fun pairself f (x,y) = (f x, f y);

(*Apply the function to a component of a pair*)
fun apfst f (x, y) = (f x, y);
fun apsnd f (x, y) = (x, f y);

fun square (n: int) = n*n;

fun fact 0 = 1
  | fact n = n * fact(n-1);


(*The following versions of itlist are designed to fit nicely with infixes.
 itlist_left is related to the old revitlist.*)

(*  (op @) (e, [x1,...,xn])  ======>   ((e @ x1) @ x2) ... @ xn
    for operators that associate to the left.  TAIL RECURSIVE*)
fun itlist_left (f: 'a * 'b -> 'a) : 'a * 'b list -> 'a =
  let fun itl (e, [ ])  = e
        | itl (e, a::l) = itl (f(e,a), l)
  in  itl end;

(*  (op @) ([x1,...,xn], e)  ======>   x1 @ (x2 ... @ (xn @ e))
    for operators that associate to the right.  TAIL RECURSIVE*)
fun itlist_right (f: 'a * 'b -> 'b) : 'a list * 'b -> 'b =
  let fun itl ([ ],  e) = e
        | itl (a::l, e) = itl (l, f(a,e))
  in  fn (l,e) => itl (rev l, e) end;


(*Length of a list.  Should unquestionably be a standard function*)
local fun length1 (n, [ ])  = n   (*TAIL RECURSIVE*)
        | length1 (n, x::l) = length1 (n+1, l)   
in  fun length l = length1 (0,l) end;


(*front (n, [e1, ..., en, e[n+1], ... , e[n+m]])   ======>   [e1, ..., en] 
 If n is negative then runs off end, raising exception.
 Raises EXCEPTION if list too short, perhaps should not.*)
fun front (0, _ ) = []
  | front (n, x::l) = x :: front(n-1, l)
  | front _  = raise LIST "front";

(*Chop n elements from the front of l.
 If n is negative then runs off end, returning [].
 Does NOT raise exception;  for style of use see nth_elem, next.*)
fun nth_tail (_,[]) = []
  | nth_tail (0, l) = l
  | nth_tail (n, _::l) = nth_tail (n-1, l);

(*Return nth element of l, where 0 designates the first element;
  raise EXCEPTION if list too short.*)
fun nth_elem NL = case (nth_tail NL) of
    [] => raise LIST "nth_elem" 
  | x::l =>  x;


(*make the list [from, from+1, ..., to]*)
infix upto;
fun from upto to =
    if from>to then []  else  from :: ((from+1) upto to);

(*make the list [from, from-1, ..., to]*)
infix downto;
fun from downto to =
    if from<to then []  else  from :: ((from-1) downto to);

(*Like Lisp's MAPC -- seq proc [x1,...,xn] evaluates 
  proc(x1); ... ; proc(xn) for side effects.*)
fun seq (proc: 'a -> unit) : 'a list -> unit = 
  let fun seqf []     = ()
        | seqf (x::l) = (proc x;  seqf l)
  in  seqf end;


(* rotate_left [1,2,3,4]   ======>  [2, 3, 4, 1] *)
fun rotate_left (a::l) = l @ [a]
  | rotate_left []     = raise LIST "rotate_left";

(* rotate_right [1,2,3,4]   ======>  [4, 1, 2, 3] *)
fun rotate_right l = case (rev l) of
    (ra::rl) => ra :: (rev rl)  
  |  []   => raise LIST "rotate_right";



(*Print error message and abort to top level*)
exception ERROR;
fun error (msg) = (prs msg;  raise ERROR);


(*print a list surrounded by the brackets lpar and rpar, with comma separator
  print nothing for empty list*)
fun print_list (lpar, rpar, pre: 'a -> unit)  (l : 'a list) = 
    let fun prec(x) = (prs",";  pre(x)) 
    in  case l of
	    [] => () 
	  | x::l =>  (prs lpar;  pre x;  seq prec l;  prs rpar)
    end;


(*print a list of items separated by newlines*)
fun print_list_ln (pre: 'a -> unit)  : 'a list -> unit = 
    seq (fn x => (pre x;  writeln""));


(*write a string repeatedly, 'count' times*)
fun writerep(count,str) =
    if count>0 then (prs str;  writerep(count-1, str))   else ();


fun is_letter ch =
  (ord"A" <= ord ch)  andalso  (ord ch <= ord"Z")   orelse
  (ord"a" <= ord ch)  andalso  (ord ch <= ord"z");

fun is_digit ch =
  (ord"0" <= ord ch)  andalso  (ord ch <= ord"9");

(*letter or _ or prime (') *)
fun is_quasi_letter "_" = true
  | is_quasi_letter "'" = true
  | is_quasi_letter ch  = is_letter ch;


(*white space: blanks, tabs, newlines*)
val is_blank : string -> bool = fn
     " " => true  |  "\t" => true  |  "\n" => true  |  _ => false;


(*Function exponentiation: f(...(f x)...) with n applications of f *)
fun funpow n f x =
  let fun rep (0,x) = x
        | rep (n,x) = rep (n-1, f x)
  in  rep (n,x)  end;

(*The old 'combine'.
  [x1,...,xn] ~~ [y1,...,yn]  ======>   [(x1,y1), ..., (xn,yn)] *)
infix ~~;
fun []   ~~  []   = []
  | x::xs ~~ y::ys = (x,y) :: (xs ~~ ys)
  |  _   ~~   _   = raise LIST "~~";

(*Inverse of ~~;  the old 'split'.
   [(x1,y1), ..., (xn,yn)]  ======>  ( [x1,...,xn] , [y1,...,yn] ) *)
fun split_list l = (map fst l, map snd l);


(*make the list [x; x; ...; x] of length n*)
fun replicate n (x: 'a) : 'a list =
  let fun rep (0,xs) = xs
        | rep (n,xs) = rep(n-1, x::xs) 
  in   if n<0 then raise LIST "replicate"
       else rep (n,[])
  end;

(*Flatten a list of lists to a list.*)
fun flat (ls: 'c list list) : 'c list = itlist_right (op @) (ls,[]);


(*POLYMORPHIC EQUALITY: SET OPERATIONS*)

(*membership in a list*)
infix mem;
fun x mem []  =  false
  | x mem (y::l)  =  (x=y) orelse (x mem l);


(*insertion into list if not already there*)
infix ins;
fun x ins xs = if x mem xs then  xs   else  x::xs;


(*union of sets represented as lists: no repetitions*)
infix union;
fun   []    union ys = ys
  | (x::xs) union ys = xs union (x ins ys);

infix subset;
fun   []    subset ys = true
  | (x::xs) subset ys = x mem ys   andalso  xs subset ys;


(*Returns the tail beginning with the first repeated element, or []. *)
fun findrep [] = []
  | findrep (x::xs) = if  x mem xs  then  x::xs   else   findrep xs;


fun distinct1 (seen, []) = rev seen
  | distinct1 (seen, x::xs) =
      if x mem seen then distinct1 (seen, xs)
    		    else distinct1 (x::seen, xs);

(*Makes a list of the distinct members of the input*)
fun distinct xs = distinct1([],xs);


datatype 'a option = None  |  Some of 'a;


(*Association list lookup.  USES POLYMORPHIC EQUALITY.*)
fun assoc ([], key) = None
  | assoc ((keyi,xi)::pairs, key) =
      if key=keyi then Some xi  else assoc (pairs,key);


fun mapfilter (f: 'a -> 'b option) ([]: 'a list) = [] : 'b list
  | mapfilter f (x::xs) =
      case (f x) of
	  None => mapfilter f xs
	| Some y => y :: mapfilter f xs;


(*Copy the list preserving elements that satisfy the predicate*)
fun filter (pred: 'a->bool) : 'a list -> 'a list = 
  let fun filt [] = []
        | filt (x::xs) =  if pred(x) then x :: filt xs  else  filt xs
  in  filt   end;


(*Partition list into elements that satisfy predicate and those that don't.
  Preserves order of elements in both lists. *)
fun partition (pred: 'a->bool) (ys: 'a list) : ('a list * 'a list) =
  let fun part ([], answer) = answer
        | part (x::xs, (ys, ns)) =
	    if pred(x) then  part (xs, (x::ys, ns))  
		       else  part (xs, (ys, x::ns)) 
  in  part (rev ys, ([],[]))  end;


(*Partition a list into buckets  [ bi, b(i+1),...,bj ]
   putting x in bk if p(k)(x) holds.  Preserve order of elements if possible.*)
fun partition_list p i j =
  let fun part k xs = 
            if k>j then 
              (case xs of [] => []
                         | _ => raise LIST "partition_list")
            else
	    let val (ns,rest) = partition (p k) xs;
	    in  ns :: part(k+1)rest  end
  in  part i end;


(*Use the keyfun to make a list of (x,key) pairs.*)
fun make_keylist (keyfun: 'a->'b) : 'a list -> ('a * 'b) list =
  let fun keypair x = (x, keyfun x) 
  in   map keypair  end;


(*Given a list of (x,key) pairs and a searchkey
  return the list of xs from each pair whose key equals searchkey*)
fun keyfilter [] searchkey = []
  | keyfilter ((x,key)::pairs) searchkey = 
	if key=searchkey then x :: keyfilter pairs searchkey
	else keyfilter pairs searchkey;


(*Insertion sort.  Stable (does not reorder equal elements)
  'less' is less-than test on type 'a. *)
fun sort (less: 'a*'a -> bool) = 
  let fun insert (x, []) = [x]
        | insert (x, y::ys) = 
              if less(y,x) then y :: insert (x,ys) else x::y::ys;
      fun sort1 [] = []
        | sort1 (x::xs) = insert (x, sort1 xs)
  in  sort1  end;




(*operators for combining predicates*)
infix orf; 
fun p orf q = fn x => p x orelse q x ;

infix andf; 
fun p andf q = fn x => p x andalso q x ;

fun notf p x = not (p x) ;


fun orl [] = false
  | orl (x::l) =  x  orelse  orl l;

fun andl [] = true
  | andl (x::l) =  x  andalso  andl l;


(*exists pred [x1,...,xn] ======>  pred(x1)  orelse  ...  orelse  pred(xn)*)
fun exists (pred: 'a -> bool) : 'a list -> bool = 
  let fun boolf [] = false
        | boolf (x::l) = (pred x) orelse boolf l
  in boolf end;


(*forall pred [x1,...,xn] ======>  pred(x1)  andalso  ...  andalso  pred(xn)*)
fun forall (pred: 'a -> bool) : 'a list -> bool = 
  let fun boolf [] = true
        | boolf (x::l) = (pred x) andalso (boolf l)
  in boolf end;

(*Expand the number in the given base 
 example: radixpand(2, 8)  gives   [1, 0, 0, 0] *)
fun radixpand (base,num) : int list =
  let fun radix (n,tail) =
  	if n<base then n :: tail
	else radix (n div base, (n mod base) :: tail)
  in  radix (num,[])  end;

(*Expands a number into a string of characters starting from "zerochar"
 example: radixstring(2,"0", 8)  gives  "1000" *)
fun radixstring (base,zerochar,num) =
  let val offset = ord(zerochar); 
      fun chrof n = chr(offset+n)
  in  implode (map chrof (radixpand (base,num)))  end;

fun string_of_int n = 
  if n < 0 then "~" ^ radixstring(10,"0",~n)  else radixstring(10,"0",n);


val print_int = prs o string_of_int;


(*Increment a list of letters like a reversed base 26 number.
  If head is "z",  bumps chars in tail.  For making variants of identifiers.*)
fun bump_list [] = ["a"]
  | bump_list (c::cs) =
	if ord(c) >= ord("z")  then  "a" :: bump_list cs
        else  chr(ord(c)+1) :: cs;

fun bump_string s : string = implode (rev (bump_list (rev (explode s))));


(** Operations on integer lists *)

fun sum [] = 0
  | sum (n::ns) = n + sum ns;

fun max[m : int]  = m
  | max(m::n::ns) = if m>n  then  max(m::ns)  else  max(n::ns)
  | max []        = raise LIST "max";

fun min[m : int]  = m
  | min(m::n::ns) = if m<n  then  min(m::ns)  else  min(n::ns)
  | min []        = raise LIST "min";


(* [x1,...,xi,...,xn]  --->  ([x1,...,x(i-1)], [xi,..., xn])
   where xi is the first element that does not satisfy the predicate*)
fun take_prefix (pred : 'a -> bool)  (xs: 'a list) : 'a list * 'a list =
  let fun take (rxs, []) = (rev rxs, [])
	| take (rxs, x::xs) =
	    if  pred x  then  take(x::rxs, xs)  else  (rev rxs, x::xs)
  in  take([],xs)  end;


(* [x1,...,xi,...,xn]  --->  [xi,..., xn]
   where xi is the first element that does not satisfy the predicate*)
fun strip_prefix (pred : 'a -> bool) : 'a list -> 'a list  =
  let fun strip [] = []
	| strip (x::xs) = if  pred x  then  strip xs  else  x::xs
  in  strip  end;

