(*  Title: 	LK/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Sequent Calculus for classical first-order logic.
    Adapted from Philippe de Groote's work.
*)


signature LK_SYNTAX =
sig
  structure Syntax : SYNTAX
  val Aexp: typ
  val Aform: typ
  val Asequ: typ
  val Asobj: typ
  val const_decs: (string list * typ) list
  val parse1: ((string*Syntax.lexsy)list -> term * (string*Syntax.lexsy) list) -> (string*Syntax.lexsy)list -> term * (string*Syntax.lexsy)list
  val parser: (string*Syntax.lexsy) list -> term * (string*Syntax.lexsy) list
  val print1: (Syntax.lexsy Syntax.Symtab.table -> int * int -> term -> unit)-> Syntax.lexsy Syntax.Symtab.table -> int * int -> term -> unit
  val printer: Syntax.lexsy Syntax.Symtab.table -> term -> unit
  val prin: term -> unit
  val read: string -> term
  val syn: Syntax.syntax
end;

functor LK_SyntaxFun (structure Syntax: SYNTAX and Pretty: PRETTY) 
      : LK_SYNTAX = 
struct
structure Syntax = Syntax;
structure Symtab = Syntax.Symtab;
local open Syntax
in


(*Types of expressions, objects (?), hypotheses, formulae*)
val Aexp  = Ground "exp";
val Asobj = Ground "sobj";
val Asequ = Asobj-->Asobj;
val Aform = Ground "form";

(*Constants and their types*)
val const_decs = 
  [ (*Representation of sequents*)
    (["True"],			[Asequ,Asequ]--->Aprop ),
    (["Seqof"],			Aform-->Asequ ), 

    (* ~P is primitive; contrast with the intuitionistic P-->False*)
    (["not"], 			Aform-->Aform ), 
    (["&", "|", "-->", "<->"],	[Aform,Aform]--->Aform ), 
    (["="],			[Aexp,Aexp]--->Aform ),

    (["The"],			(Aexp-->Aform) --> Aexp ), 
    (["Forall","Exists"],	(Aexp-->Aform) --> Aform ) ];

 
(*Get constants.*)
val get = Sign.get_const const_decs;

val True	= get "True"
and Seqof	= get "Seqof";

val Forall	= get "Forall"
and Exists	= get "Exists"
and The		= get "The"
and Not   	= get "not";


(*Abstract over "sobj" -- representation of a sequence of formulae *)
fun abs_sobj t = Abs("sobj", Asobj, t);

(*Representation of empty sequence*)
val Sempty =  abs_sobj (Bound 0);

(*Make a sequence from a list of formulae*)
fun make_sequ Ps = abs_sobj (itlist_right (op $) (Ps, Bound 0));


(*** Parsing.  Allows prefix notation: ~P for not(P), with highest precedence. 
     There are bugs involving negation and subtheories.  ***)

(*Functional called by set theory parsers; recparse handles recursive calls*)
fun parse1 recparse =
  let fun parse_quant quant lxs =
	    apfst (fn ((a,T), P) => quant $ absfree(a,T,P)) 
		  ((parse_ident thenkey"." thenp recparse) lxs)
      fun parse (("[|",DelimSy) :: lxs) =
	       (parse_thm1 recparse  thenkey"|]") lxs
        | parse (("ALL",DelimSy) :: lxs) = parse_quant Forall lxs
        | parse (("EXISTS",DelimSy) :: lxs) = parse_quant Exists lxs
        | parse (("THE",DelimSy) :: lxs) = parse_quant The lxs
        | parse (("~",DelimSy) :: lxs) =
	    apfst (fn P => Not$P)  (parse_functl recparse 100 lxs)
        | parse lxs = parserr("Illegal expression", lxs)
  in  parse_functl parse 0  end

(* sequent ::= item list "|-" item list
   item ::= $ sequence    |    formula  *)
and parse_thm1 recparse lxs = 
  let fun parse_entry (("$",DelimSy) :: lxs) =
		apfst (fn H => (constrain Asequ) $ H)  (recparse lxs)
        | parse_entry lxs =  apfst (fn P => Seqof$P)  (recparse lxs)

      (*Parsing of a non-empty sequence*)
      fun parse_seq lxs = apfst make_sequ  (parepin (",", parse_entry) lxs)
      
      (* A sequence is absent if "|-" or "|]" is first*)
      fun parse_seq_opt lxs = (case lxs of
		("|-", DelimSy)::_ => (Sempty, lxs)
	      | ("|]", DelimSy)::_ => (Sempty, lxs)
	      | _ => parse_seq lxs)
  in  apfst (fn (hyp,con) => True$hyp$con) 
	    ((parse_seq_opt thenkey "|-" thenp parse_seq_opt) lxs)
  end;



(*Main parser for LK -- calls itself*)
fun parser lxs = parse1 parser lxs;



(*** Printing. Special treatment of not, Exists, Forall ***)

(*Convert term having meta-type "sobj" into a list of terms  *)
fun entries_of_sobj (H $ u) = H :: entries_of_sobj u
  | entries_of_sobj (Bound 0) = []
  | entries_of_sobj t    = raise TERM_ERROR("Ill-formed sequent", [t]);


(*Functional called by set theory printer; recprint handles recursive calls*)
fun print1 recprint lextab (arg as (outleft,outright)) =
  let fun print_entry (Const("Seqof",_) $ P) = recprint lextab (0,0) P
	| print_entry P = (Pretty.st"$";  recprint lextab (0,0) P)
      fun print_quant (pre, (a,P)) =
	  (Pretty.bg 2;  Pretty.brks (pre ^ " " ^ a ^ ".") 1;  
           recprint lextab (0,0) P;  Pretty.en())
  in fn
      Const("True",_) $ (Abs(_,_,hypbody)) $ (Abs(_,_,conbody)) => 
        (Pretty.bg 0;
	 Pretty.list ("[| ", "",  print_entry) (entries_of_sobj hypbody);
	 Pretty.brk(1,0);
	 Pretty.list ("|- "," |]",print_entry) (entries_of_sobj conbody);
	 Pretty.en())
    | Const("Forall",_) $ Abs(a,T,P) => 
  	enclose_if (outright>0) print_quant ("ALL", variant_abs(a,T,P))
    | Const("Exists",_)  $ Abs(a,T,P) =>
	enclose_if (outright>0) print_quant ("EXISTS", variant_abs(a,T,P))
    | Const("The",_)  $ Abs(a,T,P) =>
	enclose_if (outright>0) print_quant ("THE", variant_abs(a,T,P))
    | Const("not",_)  $ P  =>    
	(Pretty.st"~ ";  recprint lextab (100,outright) P)  
    | t => print_functl (recprint lextab, lextab) arg t
  end;


(*Main printer for LK*)
fun printer lextab =
  let fun pr lextab = print1 pr lextab
  in  pr lextab (0,0)  end;
 

(*Infixes and their precedence*)
val infix_decs = 
    [ (["="],		"left", 6),
      (["&"],		"right", 5),
      (["|"],		"right", 4),
      (["-->","<->"],	"right", 3) ];

val syn = Syntax.extend pure
	(["$", "|-", ".", "~", "ALL", "EXISTS", "THE"],
		infix_decs, parser, printer); 

(*read/print functions*)
fun read a = Syntax.read syn "?" a;
fun prin t = (Pretty.init();  Syntax.prin syn t;  Pretty.nl());


end;
end;
