(*  Title: 	HOL/simp
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Classical Higher-order Logic:

Lemmas for simplification
*)

val equal_iff = prove_goal HOL_Rule.thy
    "[| [ a = a' : A ] |] ==> [| [ b = b' : A ] |] ==> \
\    [| [ a = b : A ] <-> [ a' = b' : A ] |]"
 (fn [asma,asmb]=>
  [ (REPEAT (ares_tac [iff_intr, asma, asmb, asma RS sym, asmb RS sym] 1  ORELSE
	eresolve_tac [box_equals] 1)) ]);


(*Basic inference = assumption or contradiction*)
val basic_tac = assume_tac ORELSE' contr_tac;

(*Use iff_elim on an assumption to solve at least 2 goals*)
fun iff_tac asms i =
    resolve_tac [iff_elim] i THEN
    resolve_tac asms i THEN
    (SOMEGOAL basic_tac) THEN
    (SOMEGOAL basic_tac);


(*A congruence rule for "form" in terms of <-> *)
val form_iff = prove_goal HOL_Rule.thy
    "[| [ p = q : bool ] |] ==> [| form(p) <-> form(q) |]"
 (fn asms=>
  [ (resolve_tac [iff_intr] 1),
    (REPEAT (eresolve_tac (reslist(asms, 1, subst)
		      @   reslist(asms, 1, sym RS subst)) 1)) ]);


(*A congruence rule for "term" from <-> *)
val iff_term = prove_goal HOL_Rule.thy
    "[| P <-> Q |] ==> [| [ term(P) = term(Q) : bool ] |]"
 (fn asms=>
  [ (REPEAT (iff_tac asms 1 ORELSE
	     resolve_tac [term_congr] 1)) ]);

(** congruence rules for the connectives in terms of <-> *)

val conj_iff = prove_goal HOL_Rule.thy
    "[| P<->P' |] ==> ([| P' |] ==> [| Q<->Q' |]) ==> [| (P&Q) <-> (P'&Q') |]"
 (fn asms=>
  [ (REPEAT (Pc.step_tac [] 1 ORELSE iff_tac asms 1)) ]);

val disj_iff = prove_goal HOL_Rule.thy
    "[| P<->P' |] ==> [| Q<->Q' |] ==> [| (P|Q) <-> (P'|Q') |]"
 (fn asms=>
  [ (REPEAT (Pc.step_tac [] 1 ORELSE iff_tac asms 1)) ]);

val imp_iff = prove_goal HOL_Rule.thy
    "[| P<->P' |] ==> ([| P' |] ==> [| Q<->Q' |]) ==> \
\    [| (P-->Q) <-> (P'-->Q') |]"
 (fn asms=>
  [ (REPEAT (Pc.step_tac [] 1 ORELSE iff_tac asms 1)) ]);

val iff_iff = prove_goal HOL_Rule.thy
    "[| P<->P' |] ==> [| Q<->Q' |] ==> [| (P<->Q) <-> (P'<->Q') |]"
 (fn asms=>
  [ (REPEAT (Pc.step_tac [] 1 ORELSE iff_tac asms 1)) ]);

val not_iff = prove_goal HOL_Rule.thy
    "[| P<->P' |] ==> [| ~P <-> ~P' |]"
 (fn asms=>
  [ (REPEAT (Pc.step_tac [] 1 ORELSE iff_tac asms 1)) ]);

val all_iff = prove_goal HOL_Rule.thy
    "(!(x)[| x: A |] ==> [| P(x)<->P'(x) |])  ==>  \
\    [| (ALL x:A.P(x)) <-> (ALL x:A.P'(x)) |]"
 (fn asms=>
  [ (REPEAT (Pc.step_tac [] 1 ORELSE assume_tac 1 ORELSE iff_tac asms 1)) ]);

val exists_iff = prove_goal HOL_Rule.thy
    "(!(x)[| x: A |] ==> [| P(x)<->P'(x) |])  ==>  \
\    [| (EXISTS x:A.P(x)) <-> (EXISTS x:A.P'(x)) |]"
 (fn asms=>
  [ (REPEAT (Pc.step_tac [] 1 ORELSE assume_tac 1 ORELSE iff_tac asms 1)) ]);


val iff_congr_rls = 
   [equal_iff, form_iff, iff_term, conj_iff, disj_iff, imp_iff,
    iff_iff, not_iff, all_iff, exists_iff];


(*Reflexivity of <->*)
val refl_iff = prove_goal HOL_Rule.thy
    "[| P<->P |]"
 (fn _=>
  [ (REPEAT (Pc.step_tac [] 1)) ]);

(*Permits simplification of a formula*)
val replace_iff = prove_goal HOL_Rule.thy
    "[| P<->Q |] ==> [| Q |] ==> [| P |]"
 (fn asms=>
  [ (REPEAT (eresolve_tac [contr_elim] 1  ORELSE  
	     ares_tac (asms@[iff_elim]) 1)) ]);


(** Substitution: 1, 2, 3, 4, 5 arguments *)


(*just for res_subst_tac*)
fun res1_tac th = resolve_tac [th RS sym RS subst] 1;

(*just for proving subst1-5: each assumption is applied once in correct order*)
fun res_subst_tac (th::ths) =
    MAP_EVERY res1_tac ths  THEN
    (REPEAT (ares_tac ([refl,th]@(reslist(ths,1,eq_type1))) 1));

val subst1 = prove_goal HOL_Rule.thy
    "(!(u)[| u: A |] ==> [| s(u) : S |]) ==> \
\    [| [ a = a' : A ] |] ==> \
\    [| [ s(a) = s(a') : S ] |]"  
 (fn asms => [ res_subst_tac asms ] );

val subst2 = prove_goal HOL_Rule.thy
    "(!(u,v)[| u: A |] ==> [| v: B(u) |] ==> [| s(u,v) : S |]) ==> \
\    [| [ a = a' : A ] |] ==> \
\    [| [ b = b' : B(a) ] |] ==> \
\    [| [ s(a,b) = s(a',b') : S ] |]"  
 (fn asms => [ res_subst_tac asms ] );

val subst3 = prove_goal HOL_Rule.thy
    "(!(u,v,w)[| u: A |] ==> [| v: B(u) |] ==> [| w: C(u,v) |] ==> \
\	[| s(u,v,w) : S |]) ==> \
\    [| [ a = a' : A ] |] ==> \
\    [| [ b = b' : B(a) ] |] ==> \
\    [| [ c = c' : C(a,b) ] |] ==> \
\    [| [ s(a,b,c) = s(a',b',c') : S ] |]"  
 (fn asms => [ res_subst_tac asms ] );

val subst4 = prove_goal HOL_Rule.thy
    "(!(u,v,w,x)[| u: A |] ==> [| v: B(u) |] ==> [| w: C(u,v) |] ==> \
\	[| x: D(u,v,w) |] ==> [| s(u,v,w,x) : S |]) ==> \
\    [| [ a = a' : A ] |] ==> \
\    [| [ b = b' : B(a) ] |] ==> \
\    [| [ c = c' : C(a,b) ] |] ==> \
\    [| [ d = d' : D(a,b,c) ] |] ==> \
\    [| [ s(a,b,c,d) = s(a',b',c',d') : S ] |]"  
 (fn asms => [ res_subst_tac asms ] );

val subst5 = prove_goal HOL_Rule.thy
    "(!(u,v,w,x,y)[| u: A |] ==> [| v: B(u) |] ==> [| w: C(u,v) |] ==> \
\     [| x: D(u,v,w) |] ==> [| y: E(u,v,w,x) |] ==> [| s(u,v,w,x,y): S |])==> \
\    [| [ a = a' : A ] |] ==> \
\    [| [ b = b' : B(a) ] |] ==> \
\    [| [ c = c' : C(a,b) ] |] ==> \
\    [| [ d = d' : D(a,b,c) ] |] ==> \
\    [| [ e = e' : E(a,b,c,d) ] |] ==> \
\    [| [ s(a,b,c,d,e) = s(a',b',c',d',e') : S ] |]"  
 (fn asms => [ res_subst_tac asms ] );


fun subst1_tac rs = res_inst_tac [("s",rs, Aterm-->Aterm)] subst1;

fun subst2_tac rs = res_inst_tac [("s",rs, [Aterm,Aterm]--->Aterm)] subst2;

fun subst3_tac rs =
    res_inst_tac [("s",rs, [Aterm,Aterm,Aterm]--->Aterm)] subst3;

fun subst4_tac rs =
    res_inst_tac [("s",rs, [Aterm,Aterm,Aterm,Aterm]--->Aterm)] subst4;

fun subst5_tac rs =
    res_inst_tac [("s",rs, [Aterm,Aterm,Aterm,Aterm,Aterm]--->Aterm)] subst5;


(*** Transformation of X_type rule into X_congr rule *)

val subst_rls = [subst1,subst2,subst3,subst4,subst5];

(*Generalize the theorem over the variable v in the premise [|v:A|]*)
fun gen_elem_var (Const("Elem",_) $ v $ _, th) = 
	forall_intr (Sign.cterm_of HOL_Rule.sign v) th
  | gen_elem_var (_, th) = th;

(*Make a congruence rule.  X_type must have 1 to 5 Elem premises only. *)
fun make_congr_rule X_type =
    let val prems = prems_of X_type;
	(*call lift_rule to increment var indexes
	  (hence the "wrong" premise number!) *)
	val gen_rl = lift_rule (subst1, 2)
	      (itlist_right gen_elem_var (prems, X_type));
	(*select appropriate subst rule by premise count*)
	val subst_rl = nth_elem (length prems - 1, subst_rls)
		handle LIST _ => raise THM
		    ("make_congr_rule: premises", 0, [X_type]);
	val thsq = bicompose (subst_rl, 1) (false, gen_rl, 0)
    in  case Sequence.chop (2, thsq) of
	   ([th],_) => standard th
	 |      _   => raise THM("make_congr_rule: ill-formed", 0, [X_type])
    end;

fun make_congr_rules [] = []
  | make_congr_rules (rl::rls) =
	([make_congr_rule rl]  handle THM _ => []) @ make_congr_rules rls;

