(*  Title: 	HOL/lemmas
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Classical Higher-order Logic:

Lemmas for equality and logical reasoning
*)


signature HOL_LEMMAS = 
sig
  structure Tactic : TACTIC
  val all_elim: Tactic.Thm.thm
  val all_elim2: Tactic.Thm.thm
  val all_elim3: Tactic.Thm.thm
  val all_elim4: Tactic.Thm.thm
  val all_elim5: Tactic.Thm.thm
  val box_equals: Tactic.Thm.thm
  val chain_tac: int -> Tactic.tactic
  val conjunct1: Tactic.Thm.thm
  val conjunct2: Tactic.Thm.thm
  val conj_elim: Tactic.Thm.thm 
  val conj_intr: Tactic.Thm.thm 
  val contr: Tactic.Thm.thm
  val contr_elim: Tactic.Thm.thm
  val contr_tac: int -> Tactic.tactic
  val determ_type_tac: Tactic.Thm.thm list -> int -> Tactic.tactic
  val disj_cintr: Tactic.Thm.thm
  val disj_elim: Tactic.Thm.thm 
  val disj_intr1: Tactic.Thm.thm 
  val disj_intr2: Tactic.Thm.thm 
  val eqterm_intr: Tactic.Thm.thm
  val equal_prop: Tactic.Thm.thm
  val equal_prop_iff: Tactic.Thm.thm
  val eq_mp_tac: int -> Tactic.tactic
  val exists_cintr: Tactic.Thm.thm
  val exists_elim: Tactic.Thm.thm 
  val exists_intr: Tactic.Thm.thm 
  val extensionality: Tactic.Thm.thm
  val ex_middle: Tactic.Thm.thm
  val False_elim: Tactic.Thm.thm
  val False_type: Tactic.Thm.thm
  val form_elim2: Tactic.Thm.thm
  val goal_kind: term -> string
  val iff_elim: Tactic.Thm.thm
  val iff_intr: Tactic.Thm.thm
  val IF_PROP: (int -> Tactic.tactic) -> int -> Tactic.tactic
  val IF_TYPE: (int -> Tactic.tactic) -> int -> Tactic.tactic
  val imp_celim: Tactic.Thm.thm
  val imp_elim: Tactic.Thm.thm
  val is_elem: term -> bool
  val is_rigid_elem: term -> bool
  val is_rigid_prop: term -> bool
  val is_typing_goal: term -> bool
  val joinrules: Tactic.Thm.thm list * Tactic.Thm.thm list -> (bool * Tactic.Thm.thm) list
  val lessb: (bool * Tactic.Thm.thm) * (bool * Tactic.Thm.thm) -> bool
  val mp_tac: int -> Tactic.tactic
  val not_intr: Tactic.Thm.thm
  val pair_conv: Tactic.Thm.thm
  val Pick_equals: Tactic.Thm.thm
  val Pick_iff_equals: Tactic.Thm.thm
  val rigid_assume_tac: int -> Tactic.tactic
  val simp_equals: Tactic.Thm.thm
  val size_of_state: Tactic.Thm.thm -> int
  val size_of_thm: Tactic.Thm.thm -> int
  val split_conv: Tactic.Thm.thm
  val split_elim: Tactic.Thm.thm
  val split_intr: Tactic.Thm.thm
  val split_type: Tactic.Thm.thm
  val subgoals_of_brl: bool * Tactic.Thm.thm -> int
  val swap: Tactic.Thm.thm
  val TCOND: (int -> Tactic.tactic) -> (int -> Tactic.tactic) -> int -> Tactic.tactic
  val trans: Tactic.Thm.thm
  val True_intr: Tactic.Thm.thm
  val True_type: Tactic.Thm.thm
  val type_rls: Tactic.Thm.thm list
end;


functor HOL_LemmasFun
    (structure HOL_Syntax: HOL_SYNTAX
	   and HOL_Rule: HOL_RULE
	   and Logic: LOGIC
	   and Conv: CONV
	sharing HOL_Syntax.Syntax = HOL_Rule.Thm.Sign.Syntax
	    and Conv.Tactic.Thm = HOL_Rule.Thm) : HOL_LEMMAS = 
struct
structure Tactic = Conv.Tactic;
local open Conv Tactic Thm HOL_Rule HOL_Syntax
in


(** Falsity *)

val False_type = prove_goal HOL_Rule.thy "[| False: bool |]"
 (fn _=>
  [ (rewrite_goals_tac [False_def]),
    (resolve_tac [term_type] 1) ]);

val False_elim = prove_goal HOL_Rule.thy 
    "[| form(False) |] ==> [| P |]"
 (fn [asm]=>
  [ (resolve_tac [form_elim] 1),
    (resolve_tac [rewrite_rule[False_def] asm RS form_elim RS spec] 1),
    (resolve_tac [term_type] 1) ]);

(** Truth *)

val True_type = prove_goal HOL_Rule.thy "[| True: bool |]"
 (fn _=>
  [ (rewrite_goals_tac [True_def]),
    (resolve_tac [term_type] 1) ]);


val True_intr = prove_goal HOL_Rule.thy "[| form(True) |]"
 (fn asms=>
  [ (rewrite_goals_tac [True_def]),
    (REPEAT (ares_tac [form_intr,imp_intr,all_intr] 1)) ]);


(** Negation *)


val not_intr = prove_goal HOL_Rule.thy 
    "([| P |] ==> [| form(False) |]) ==> [| ~P |]"
 (fn asms=>
  [ (rewrite_goals_tac [not_def]),
    (REPEAT (ares_tac (asms@[imp_intr]) 1)) ]);


val contr = prove_goal HOL_Rule.thy 
    "[| ~P |] ==> [| P |] ==> [| form(False) |]"
 (fn asms=>
  [ (resolve_tac [mp] 1),
    (REPEAT (resolve_tac (map (rewrite_rule[not_def]) asms) 1)) ]);


val contr_elim = prove_goal HOL_Rule.thy 
    "[| ~P |] ==> [| P |] ==> [| Q |]"
 (fn asms=>
  [ (REPEAT (ares_tac (asms@[contr RS False_elim]) 1)) ]);




val form_elim2 = prove_goal HOL_Rule.thy 
    "[| form(term(P)) |] ==> ([| P |] ==> [| R |]) ==> [| R |]"
 (fn asms=>
  [ (REPEAT (resolve_tac (asms@[form_elim]) 1)) ]);


val imp_elim = prove_goal HOL_Rule.thy 
    "[| P-->Q |] ==> [| P |] ==> ([| Q |] ==> [| R |]) ==> [| R |]"
 (fn asms=>
  [ (REPEAT (resolve_tac (asms@[mp]) 1)) ]);


(*Note that a:A is last subgoal: others are proved first and instantiate a!*)
val all_elim = prove_goal HOL_Rule.thy 
    "[| ALL x:A.P(x) |] ==> ([| P(a) |] ==> [| R |]) ==> [| a : A |]  ==>  \
\    [| R |]"
 (fn asms=>
  [ (REPEAT (resolve_tac (asms @ [ asms RSN (1,spec) ]) 1)) ]);

(*Multiple forall elimination.   Combine with chain_tac to achieve PROLOG
  execution. *)
val all_elim2 = prove_goal HOL_Rule.thy 
    "[| ALL x:A. ALL y:B. P(x,y) |] ==> \
\    ([| P(a,b) |] ==> [| R |]) ==> \
\    [| a : A |] ==> [| b : B |] ==> \
\    [| R |]"
 (fn asms=>
  [ (resolve_tac [all_elim] 1),
    (REPEAT (ares_tac asms 1  ORELSE  eresolve_tac [all_elim] 1)) ]);

val all_elim3 = prove_goal HOL_Rule.thy 
    "[| ALL x:A. ALL y:B. ALL z:C. P(x,y,z) |] ==> \
\    ([| P(a,b,c) |] ==> [| R |]) ==> \
\    [| a : A |] ==> [| b : B |] ==> [| c : C |] ==> \
\    [| R |]"
 (fn asms=>
  [ (resolve_tac [all_elim] 1),
    (REPEAT (ares_tac asms 1  ORELSE  eresolve_tac [all_elim] 1)) ]);

val all_elim4 = prove_goal HOL_Rule.thy 
    "[| ALL x:A. ALL y:B. ALL z:C. ALL u:D. P(x,y,z,u) |] ==> \
\    ([| P(a,b,c,d) |] ==> [| R |]) ==> \
\    [| a : A |] ==> [| b : B |] ==> [| c : C |] ==> \
\    [| d : D |] ==> \
\    [| R |]"
 (fn asms=>
  [ (resolve_tac [all_elim] 1),
    (REPEAT (ares_tac asms 1  ORELSE  eresolve_tac [all_elim] 1)) ]);

val all_elim5 = prove_goal HOL_Rule.thy 
    "[| ALL x:A. ALL y:B. ALL z:C. ALL u:D. ALL v:E. P(x,y,z,u,v) |] ==> \
\    ([| P(a,b,c,d,e) |] ==> [| R |]) ==> \
\    [| a : A |] ==> [| b : B |] ==> [| c : C |] ==> \
\    [| d : D |] ==> [| e : E |] ==> \
\    [| R |]"
 (fn asms=>
  [ (resolve_tac [all_elim] 1),
    (REPEAT (ares_tac asms 1  ORELSE  eresolve_tac [all_elim] 1)) ]);

(*** Derived rules for conjunction, disjunction, existential *)

val conn_defs = [conj_def,disj_def,exists_def];

(** Introduction rules *)

(*special tactic for proving them*)
fun prove_intr_tac asms =
    rewrite_goals_tac conn_defs  THEN
    (DEPTH_SOLVE_1 (ares_tac ([all_intr,imp_intr]@asms) 1  ORELSE  
	eresolve_tac [imp_elim, all_elim] 1));

val conj_intr = prove_goal HOL_Rule.thy 
    "[| P |] ==> [| Q |] ==> [| P&Q |]"
 (fn asms=>
  [ (prove_intr_tac asms) ]);

val disj_intr1 = prove_goal HOL_Rule.thy 
    "[| P |] ==> [| P|Q |]"
 (fn asms=>
  [ (prove_intr_tac asms) ]);
    
val disj_intr2 = prove_goal HOL_Rule.thy 
    "[| Q |] ==> [| P|Q |]"
 (fn asms=>
  [ (prove_intr_tac asms) ]);

val exists_intr = prove_goal HOL_Rule.thy 
    "[| P(a) |]  ==>  [| a: A |] ==>  [| EXISTS x:A.P(x) |]"
 (fn asms=>
  [ (prove_intr_tac asms) ]);


(** Elimination rules *)

(*special tactic for proving them*)
fun prove_elim_tac asms =
    resolve_tac [map (rewrite_rule conn_defs) asms RSN (1,all_elim)] 1  THEN
    (REPEAT (eresolve_tac (asms @ [imp_elim, all_elim, form_elim2]) 1  ORELSE  
	ares_tac [term_type,all_intr,imp_intr,form_intr] 1));

    
val conj_elim = prove_goal HOL_Rule.thy 
    "[| P&Q |] ==> ([| P |] ==> [| Q |] ==> [| R |])  ==>  [| R |]"
 (fn asms=>
  [ (prove_elim_tac asms) ]);

    
val disj_elim = prove_goal HOL_Rule.thy 
    "[| P|Q |]  ==>  ([| P |] ==> [| R |])  ==>  ([| Q |] ==> [| R |])  ==> \
\    [| R |]"
 (fn asms=>
  [ (prove_elim_tac asms) ]);

val exists_elim = prove_goal HOL_Rule.thy 
    "[| EXISTS x:A.P(x) |] ==> (!(y)[| y:A |] ==> [| P(y) |] ==> [| Q |]) ==> \
\    [| Q |]"
 (fn asms=>
  [ (prove_elim_tac asms) ]);



(** Classical intro rules for | and EXISTS *)

val disj_cintr = prove_goal HOL_Rule.thy 
   "([| ~Q |] ==> [| P |]) ==> [| P|Q |]"
 (fn asms=>
  [ (resolve_tac [classical] 1),
    (REPEAT (ares_tac (asms@[disj_intr1,not_intr]) 1)),
    (REPEAT (ares_tac (asms@[disj_intr2,contr]) 1)) ]);


(*Note that a:A is last subgoal: others are proved first and instantiate a!*)
val exists_cintr = prove_goal HOL_Rule.thy 
   "([| ALL x:A. ~P(x) |] ==> [| P(a) |])  ==>  \
\   [| a: A |] ==>  \
\   [| EXISTS x:A.P(x) |]"
 (fn asms=>
  [ (resolve_tac [classical] 1),
    (REPEAT (ares_tac (asms@[exists_intr,all_intr,not_intr,contr]) 1))  ]);


val ex_middle = prove_goal HOL_Rule.thy 
   "[| ~P | P |]"
 (fn _=>
  [ (REPEAT (ares_tac [disj_cintr] 1)) ]);


(*** Special elimination rules *)


(*Classical implies (-->) elimination. *)
val imp_celim = prove_goal HOL_Rule.thy 
    "[| P-->Q |] ==> ([| ~P |] ==> [| R |]) ==> ([| Q |] ==> [| R |]) ==> \
\    [| R |]"
 (fn asms=>
  [ (resolve_tac [disj_elim RES ex_middle] 1),
    (REPEAT (DEPTH_SOLVE_1 (ares_tac (asms @ [ mp RESN (1,asms) ]) 1))) ]);


val conjunct1 = prove_goal HOL_Rule.thy "[| P & Q |] ==> [| P |]"
 (fn asms=>
  [ (REPEAT (ares_tac (asms @ [conj_elim]) 1)) ]);

val conjunct2 = prove_goal HOL_Rule.thy "[| P & Q |] ==> [| Q |]"
 (fn asms=>
  [ (REPEAT (ares_tac (asms @ [conj_elim]) 1)) ]);


(*** Tactics for implication and contradiction ***)

(*Solve goal that assumes both P and ~P 
  Does not use imp_elim -- would be redundant in complete prover *)
val contr_tac = eresolve_tac [contr_elim]  THEN'  assume_tac;

(*Given assumption P-->Q, reduces subgoal Q to P [deletes the implication!] *)
fun chain_tac i =
    eresolve_tac [imp_elim] i  THEN  
    (assume_tac (i+1)  ORELSE  contr_tac (i+1));

(*Finds P-->Q and P in the assumptions, replaces implication by Q *)
val mp_tac =
    eresolve_tac [imp_elim, contr_elim]  THEN'  assume_tac;

(*Like mp_tac but instantiates no variables*)
val eq_mp_tac =
    eresolve_tac [imp_elim, contr_elim]  THEN'  eq_assume_tac;


(*** Bi-implication *)

val iff_intr = prove_goal HOL_Rule.thy 
   "([| P |] ==> [| Q |]) ==> ([| Q |] ==> [| P |]) ==> [| P<->Q |]"
 (fn asms=>
  [ (rewrite_goals_tac [iff_def]),
    (REPEAT (ares_tac (asms @ [conj_intr, imp_intr]) 1)) ]);


(*Classical <-> elimination.  Proof substitutes P=Q in 
    [| ~P |] ==> [| ~Q |]    and    [| P |] ==> [| Q |]  *)
val iff_elim = prove_goal HOL_Rule.thy 
    "[| P<->Q |] ==> \
\    ([| P |] ==> [| Q |] ==> [| R |]) ==> \
\    ([| ~P |] ==> [| ~Q |] ==> [| R |])  ==> [| R |]"
 (fn asms =>
  [ (resolve_tac [conj_elim] 1),
    (REPEAT (DEPTH_SOLVE_1 
	(eresolve_tac [imp_celim] 1  ORELSE 
	 mp_tac 1  ORELSE 
	 ares_tac (map (rewrite_rule [iff_def]) asms) 1))) ]);


(*Should be used as swap since ~P becomes redundant*)
val swap = prove_goal HOL_Rule.thy 
   "[| ~P |] ==> ([| ~Q |] ==> [| P |]) ==> [| Q |]"
 (fn asms=>
  [ (resolve_tac [classical] 1),
    (resolve_tac [asms RSN (1, contr RS False_elim)] 1),
    (*Looks strange but it works!*)
    (REPEAT (ares_tac asms 1)) ]);



(**** products: the type A*B  *)

val pair_conv = prove_goal HOL_Rule.thy 
  "[| p: A*B |] ==> [| [ <fst(p), snd(p)> = p: A*B ] |]"
 (fn asms=>
  [ (resolve_tac (reslist(asms,1,prod_elim)) 1),
    (resolve_tac [fst_conv RS subst] 1),
    (resolve_tac [snd_conv RS subst] 3),
    (REPEAT (ares_tac [pair_type,refl] 1)) ]);

val split_type = prove_goal HOL_Rule.thy 
  "[| p: A*B |] ==>  \
\  (!(x,y)[| x: A |] ==> [| y: B |] ==> [| c(x,y) : C |]) ==>    \
\  [| split(p,c) : C |]"
 (fn asms=>
  [ (rewrite_goals_tac [split_def]),
    (REPEAT (ares_tac (asms@[fst_type, snd_type]) 1)) ]);

val split_conv = prove_goal HOL_Rule.thy 
  "[| a: A |] ==> [| b: B |] ==>  \
\  (!(x,y)[| x: A |] ==> [| y: B |] ==> [| c(x,y) : C |]) ==>    \
\  [| [ split(<a,b>, c) = c(a,b) : C ] |]"
 (fn asms=>
  [ (rewrite_goals_tac [split_def]),
    (resolve_tac [fst_conv RS subst] 1),
    (resolve_tac [snd_conv RS subst] 3),
    (REPEAT (ares_tac (asms@[refl]) 1)) ]);

(*Introduction of split as 'logical constant'!*)
val split_intr = prove_goal HOL_Rule.thy 
  "[| form(c(a,b)) |] ==> \
\  [| a: A |] ==> [| b: B |] ==>  \
\  (!(x,y)[| x: A |] ==> [| y: B |] ==> [| c(x,y) : bool |]) ==>    \
\  [| form(split(<a,b>, c)) |]"
 (fn asms=>
  [ (resolve_tac [split_conv RS subst] 1),
    (REPEAT (ares_tac asms 1)) ]);

(*Elimination of split as 'logical constant'!*)
val split_elim = prove_goal HOL_Rule.thy 
  "[| form(split(<a,b>, c)) |] ==> \
\  [| a: A |] ==> [| b: B |] ==>  \
\  (!(x,y)[| x: A |] ==> [| y: B |] ==> [| c(x,y) : bool |]) ==>    \
\  ([| form(c(a,b)) |] ==> [| P |]) ==>    \
\  [| P |]"
 (fn asms=>
  [ (resolve_tac asms 1),
    (resolve_tac [sym RS subst] 1),
    (res_inst_tac [("c", "c", [Aterm,Aterm]--->Aterm)] split_conv 1),
    (REPEAT (ares_tac asms 1)) ]);


(** Basic typechecking rules *)

(*Types of constants.  Excludes subtype_intr -- to avoid blowing up*)
val type_rls = 
    [Lambda_type, apply_type, pair_type, fst_type, snd_type, split_type,
     Zero_type, Succ_type, rec_type, Pick_type, False_type, True_type,
     term_type];


(*** Equality *)

(*Transitivity*)
val trans = prove_goal HOL_Rule.thy
    "[| [ a = b : A ] |] ==> [| [ b = c : A ] |] ==> [| [ a = c : A ] |]"  
 (fn asms=>
  [ (res_inst_tac [("a", "a", Aterm)] subst 1),
    (REPEAT (resolve_tac asms 1)) ]);


(*A useful rule for proving equalties from other equalities.
	a = b
	|   |
	c = d	*)
val box_equals = prove_goal HOL_Rule.thy
    "[| [ a = b : A ] |] ==> [| [ a = c : A ] |] ==> [| [ b = d : A ] |] ==>  \
\    [| [ c = d : A ] |]"  
 (fn asms=>
  [ (resolve_tac [trans] 1),
    (resolve_tac [trans] 1),
    (resolve_tac [sym] 1),
    (REPEAT (resolve_tac asms 1)) ]);


(*Dual of box_equals: for proving equalities backwards*)
val simp_equals = prove_goal HOL_Rule.thy
    "[| [ a = c : A ] |] ==> [| [ b = d : A ] |] ==> [| [ c = d : A ] |] ==>  \
\    [| [ a = b : A ] |]"  
 (fn asms=>
  [ (resolve_tac [trans] 1),
    (resolve_tac [trans] 1),
    (REPEAT (resolve_tac (asms @ reslist(asms,1,sym)) 1)) ]);


(*Extensionality of functions*)
val extensionality = prove_goal HOL_Rule.thy
    "[| f: A->B |] ==> [| g: A->B |] ==> \
\    (!(x)[| x: A |] ==> [| [ f`x = g`x : B ] |]) ==> \
\    [| [ f = g : A->B ] |]"
 (fn asms =>
  [ (resolve_tac [box_equals] 1),
    (REPEAT (resolve_tac (asms@[eta_conv]) 2)),
    (REPEAT (ares_tac (asms@[Lambda_congr]) 1)) ]);



(*Equality preserves truth. 
  With "subst" implies subst, but this incurs redundant type checking*)
val equal_prop = prove_goal HOL_Rule.thy
    "[| [ term(P) = term(Q) : bool ] |] ==> [| Q |] ==> [| P |]"  
 (fn asms=>
  [ (resolve_tac [form_elim] 1),
    (resolve_tac [subst] 1),
    (REPEAT (resolve_tac (asms@[form_intr]) 1)) ]);


(*Converts an equality of terms to a biconditional*)
val equal_prop_iff = prove_goal HOL_Rule.thy
    "[| [ term(P) = term(Q) : bool ] |] ==> [| P<->Q |]"  
 (fn [asm]=>
  [ (resolve_tac [iff_intr] 1),
    (REPEAT (ares_tac [asm RS equal_prop, asm RS sym RS equal_prop] 1)) ]);


val eqterm_intr = prove_goal HOL_Rule.thy
    "([| form(p) |] ==> [| form(q) |]) ==>  \
\    ([| form(q) |] ==> [| form(p) |]) ==>   \
\    [| p: bool |] ==> [| q: bool |] ==>   \
\    [| [ p = q : bool ] |]"
 (fn asms=>
  [ (resolve_tac [box_equals] 1),
    (resolve_tac [term_congr] 1),
    (REPEAT (resolve_tac (asms@[term_conv]) 3)),
    (REPEAT (ares_tac asms 1)) ]);

(** Descriptions and equality **)

val Pick_equals = prove_goal HOL_Rule.thy
    "[| a : A |] ==> [| [ (PICK x: A. [ x = a : A ]) = a : A ] |]"
 (fn asms=>
  [ (resolve_tac [Pick_intr] 1), 
    (REPEAT (resolve_tac (asms@[exists_intr,refl]) 1)) ]); 

val Pick_iff_equals = prove_goal HOL_Rule.thy
    "[| a : A |] ==> \
\    (!(x)[| x: A |] ==> [| [ x = a : A ] <-> P(x) |]) ==> \
\    [| [ (PICK x: A. P(x)) = a : A ] |]"
 (fn asms=>
  [ (resolve_tac [trans] 1), 
    (resolve_tac [Pick_equals] 2), 
    (resolve_tac [Pick_congr RS sym] 1), 
    (REPEAT (ares_tac (asms@[exists_intr,refl]) 1)) ]); 



(*** Operations used in theorem provers 
     Better one copy here than multiple copies in Prover *)

val size_of_thm = size_of_term o #prop o rep_thm;

(*Return head constant of goal: Elem or Trueprop*)
fun goal_kind prem = 
    case head_of (Logic.strip_assums_concl prem) of
	Const(a,_) => a
      | _ => raise TERM_ERROR("goal_kind", [prem]);

fun is_typing_goal prem = (goal_kind prem = "Elem");


fun TCOND tytac prtac = SUBGOAL(fn (prem,i) =>
    if is_typing_goal prem then tytac i
    else prtac i);

fun IF_TYPE tf = TCOND tf (K no_tac);
fun IF_PROP tf = TCOND (K no_tac) tf;

fun size_of_state prf =
    sum (map size_of_term 
	     (filter (not o is_typing_goal) (prems_of prf)));

(*For use with biresolve_tac.  Combines intrs with swap to catch negated
  assumptions.  Also pairs elims with true. *)
fun joinrules (intrs,elims) =  
  (map (pair true) (elims @ reslist(intrs,2,swap))) @
  (map (pair false) intrs);


(*The number of new subgoals produced by the brule*)
fun subgoals_of_brl (true,rule) = length (prems_of rule) - 1
  | subgoals_of_brl (false,rule) = length (prems_of rule);

(*Less-than test: for sorting to minimize number of new subgoals*)
fun lessb (brl1,brl2) = subgoals_of_brl brl1 < subgoals_of_brl brl2;


(*Test for rigid typing goal*)
fun is_rigid_elem (Const("Elem",_) $ a $ _) = not (is_var (head_of a))
  | is_rigid_elem _ = false;

fun is_elem (Const("Elem",_) $ a $ _) = true
  | is_elem _ = false;

(*Test for rigid proposition (i.e., not just ?P or form(?p)) *)
fun is_rigid_prop (Const("Trueprop",_) $ (Const("form",_) $ p)) =
	not (is_var (head_of p))
  | is_rigid_prop (Const("Trueprop",_) $ P) = not (is_var (head_of P))
  | is_rigid_prop _ = false;

(*Try solving a:A by assumption provided a is rigid!*) 
val rigid_assume_tac = SUBGOAL(fn (prem,i) =>
    if is_rigid_elem (Logic.strip_assums_concl prem)
    then  assume_tac i  else  no_tac);

(*Attack subgoal i using deterministic type checking rules
  filt_resolve_tac makes certain there is no ambiguity *)
fun determ_type_tac type_ths = 
    (DETERM o REPEAT1 o 
	IF_TYPE(rigid_assume_tac ORELSE' filt_resolve_tac type_ths 1));

end;
end;
