(*  Title: 	envir
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1986  University of Cambridge
*)

(*Environments*)

infix xmem;
fun (x:indexname) xmem   []  =  false
  | (x:indexname) xmem (y::l) = (x=y) orelse (x xmem l);

(*USED???*)
fun xassoc ([], key: indexname) = None
  | xassoc ((keyi,xi)::pairs, key) =
      if key=keyi then Some xi  else xassoc (pairs,key);

(*a total, irreflexive ordering on index names*)
fun xless ((r,i), (s,j): indexname) = i<j  orelse  (i=j andalso r<s);


(*association lists with keys in ascending order, no repeated keys*)
abstype 'a xname_olist = Olist of (indexname * 'a) list
  with

    exception xname_olist : string * indexname;
     
    (*look up key in ordered list*)  
    fun xsearch (Olist[], key) = None
      | xsearch (Olist ((keyi,xi)::pairs), key) =
          if xless(keyi,key) then xsearch (Olist pairs, key)
          else if key=keyi then Some xi  
          else None;
    
    (*insert pair in ordered list, reject if key already present*)
    fun xinsert_new (newpr as (key, x), Olist al) =
      let fun ins [] = [newpr]
            | ins ((pair as (keyi,xi)) :: pairs) =
                if xless(keyi,key) then pair :: ins pairs
                else if key=keyi then raise xname_olist with ("xinsert_new",key)
                else newpr::pair::pairs
      in  Olist (ins al)  end;
    
    (*insert pair in ordered list, overwrite if key already present*)
    fun xinsert (newpr as (key, x), Olist al) =
      let fun ins [] = [newpr]
            | ins ((pair as (keyi,xi)) :: pairs) =
                if xless(keyi,key) then pair :: ins pairs
                else if key=keyi then newpr::pairs
                else newpr::pair::pairs
      in  Olist (ins al)  end;
 
    (*apply function to the contents part of each pair*)   
    fun xmap f (Olist pairs) =
      let fun xmp [] = []
            | xmp ((key,x)::pairs) = (key, f x) :: xmp pairs
      in Olist (xmp pairs)  end;
   
    (*increment the index in each key,
      simultaneously apply function to contents*)
    fun incr_indexes_olist (f,inc) (Olist pairs) =
      let fun xmp [] = []
            | xmp (((s,idx), x) ::pairs) = ((s,idx+inc), f x) :: xmp pairs
      in Olist (xmp pairs)  end;
    
    (*allows redefinition of a key by "join"ing the contents parts*)
    fun xmerge_olists join (Olist pairsa, Olist pairsb) =
      let fun xmrg ([],pairsb) = pairsb
            | xmrg (pairsa,[]) = pairsa
            | xmrg ((keya,x)::pairsa, (keyb,y)::pairsb) =
                if xless(keya,keyb) then 
                    (keya,x) :: xmrg (pairsa, (keyb,y)::pairsb)
                else if xless(keyb,keya) then
                    (keyb,y) :: xmrg ((keya,x)::pairsa, pairsb)
                else (*equal*)  (keya, join(x,y)) :: xmrg (pairsa, pairsb)
      in  Olist (xmrg (pairsa,pairsb))  end;
 
    val null_olist = Olist[];
    
    fun alist_of_olist (Olist pairs) = pairs;
    
    fun olist_of_alist pairs = itlist_right xinsert (pairs, Olist[])

  end (*abstype xname_olist*);
 


(*updating can destroy environment in 2 ways!!
   (1) variables out of range   (2) circular assignments
*)
datatype envir = Envir of
    {maxidx: int, 			(*maximum index of vars, params*)
     asol: (term) xname_olist,		(*table of assignments to vars*)
     depol: (term list) xname_olist};	(*table of dependencies of params*)


exception environ : string * envir * term list;

(*Generate a list of distinct variables.
  Increments index to make them distinct from ALL present variables. *)
fun genvars (Envir{maxidx, asol, depol}, arys) : envir * term list =
  let fun genvs (_, [] : arity list) : term list = []
        | genvs (n, ary::arys) =
            Var(("G"^ radixstring(26,"a",n), maxidx+1), ary) 
            :: genvs(n+1,arys) 
  in  (Envir{maxidx=maxidx+1, asol=asol, depol=depol}, genvs (0,arys))  end;


fun genvar (env,ary) : envir * term =
  let val (env',[v]) = genvars (env,[ary])  in  (env',v)  end;

fun elookup (Envir{asol,...}, xname) : term option = 
  xsearch (asol,xname);

fun eupdate ((xname, tm), Envir{maxidx, asol, depol}) =
  Envir{maxidx=maxidx, asol=xinsert_new ((xname,tm), asol), depol=depol};

val null_env = Envir{maxidx=0, asol=null_olist, depol=null_olist};

(*raised when norm has no effect on a term, 
  to encourage sharing instead of copying*)
exception same;

(*Beta normal form for terms (not eta normal form). 
  Chases variables in env;  Does not exploit sharing of variable bindings
  Does not check arities, so could loop. *)
fun norm_term1 (env,tm) : term =
  let fun norm (Var (uname,ary)) = 
	    (case elookup (env,uname) of
		Some u => norm_term1 (env, u)
	      | None   => raise same)
	| norm (Abs(name,ary,body)) =  Abs(name, ary, norm body)
	| norm (Abs(_,_,body) $ rand) =
	    norm_term1 (env, subst_bounds(~1, [rand], body))
	| norm (rator $ rand) =
	    (case norm rator of
	       Abs(_,_,body) =>
		 norm_term1 (env, subst_bounds(~1, [rand], body))
	     | nrator => nrator $ norm_term1 (env,rand))
            handle same => rator $ norm rand
	| norm _ =  raise same
  in  norm tm  handle same=> tm  end;


fun norm_term env tm : term = norm_term1 (env, tm);


