`include "malwa_defines.v"

module malwa_execute(
	input clk,
	input reset,
	input stall,
	input clear,
	
	input [31:0] instr,
	input [`CONTROL_WIDTH] control,
	input [31:0] rs,
	input [31:0] rt,
	input [31:0] branchout,

	input writeregenMduring,
	input [`REGNUM_WIDTH] writeregnum,
	
	output writeregenduring,
	output [`REGNUM_WIDTH] writeregnumduring,
	
	output reg [31:0] instrWB,
	output reg [`CONTROL_WIDTH] controlWB,
	output reg [31:0] executeoutWB,
	output reg [31:0] branchoutWB,
	output reg [1:0] bottomaddressWB,
	
	output stallreq,
	
	output memread,mem16,mem8,memzerofill,memwrite,
	output [31:0] memaddress,memwritedata
);

	assign writeregenduring=control[`CONTROL_REGWRITE];
	assign writeregnumduring=control[`CONTROL_WRITEREGNUM];

	wire [63:0] mult = {{32{rs[31] && !control[`CONTROL_ALUCONTROL_UNSIGNED]}}, rs}
						* {{32{rt[31] && !control[`CONTROL_ALUCONTROL_UNSIGNED]}}, rt};
	
	wire [31:0] divLO, divHI;
	malwa_div div(
		.clock(clk),
		.denom(rt),
		.numer(rs),
		.quotient(divLO),
		.remain(divHI)
	);
	
	wire [31:0] divuLO, divuHI;
	malwa_divu divu(
		.clock(clk),
		.denom(rt),
		.numer(rs),
		.quotient(divuLO),
		.remain(divuHI)
	);
	
	wire [31:0] aluout;
	malwa_alu alu(
		.srca(rs),
		.srcb(control[`CONTROL_USEIMM] ? {{16{instr[15] && !control[`CONTROL_ZEROFILL]}}, instr[15:0]} : rt),
		.alucontrol(control[`CONTROL_ALUCONTROL]),
		.aluout(aluout)
	);
	
	wire [31:0] shiftout;
	shifter shift(
		.src(rt),
		.amt(control[`CONTROL_ALUCONTROL_VARIABLE] ? rs[4:0] : instr[10:6]),
		.dir(control[`CONTROL_ALUCONTROL_RIGHT]),
		.alusigned(!control[`CONTROL_ALUCONTROL_UNSIGNED]),
		.shifted(shiftout)
	);
	
	wire [3:0] newcountdown1=reset ||
							 control[`CONTROL_ALUCONTROL] == `ALU_MTHI ||
							 control[`CONTROL_ALUCONTROL] == `ALU_MTLO ||
							 control[`CONTROL_ALUCONTROL] == `ALU_MULT ||
							 control[`CONTROL_ALUCONTROL] == (`ALU_MULT | `ALU_UNSIGNED)?	4'd0
							:control[`CONTROL_ALUCONTROL] == `ALU_DIV ||
							 control[`CONTROL_ALUCONTROL] == (`ALU_DIV | `ALU_UNSIGNED)	?	4'd11
							:countdown>0												?	countdown-4'd1
							:																4'd0;

	wire [3:0] newcountdown2=reset			?	4'd0
							:countdown>0	?	countdown-4'd1
							:					4'd0;
							
	wire [31:0] newlow1= reset											?	32'd0
						:control[`CONTROL_ALUCONTROL] == `ALU_MTLO		?	rs
						:control[`CONTROL_ALUCONTROL] == `ALU_MULT		?	mult[31:0]
						:countdown==1									?	(!source ? divLO : divuLO)
						:													low;

	wire [31:0] newlow2= reset											?	32'd0
						:countdown==1									?	(!source ? divLO : divuLO)
						:													low;

	wire [31:0] newhigh1=reset											?	32'd0
						:control[`CONTROL_ALUCONTROL] == `ALU_MTHI		?	rs
						:control[`CONTROL_ALUCONTROL] == `ALU_MULT		?	mult[63:32]
						:countdown==1									?	(!source ? divHI : divuHI)
						:													high;

	wire [31:0] newhigh2=reset											?	32'd0
						:countdown==1									?	(!source ? divHI : divuHI)
						:													high;

	reg [3:0] countdown;
	reg source;
	reg [31:0] high, low;
	always @(posedge clk)
	begin
		countdown <= !stall ? newcountdown1 : newcountdown2;
		low <= !stall ? newlow1 : newlow2;
		high <= !stall ? newhigh1 : newhigh2;

		if( reset )
		begin
			instrWB <= 0;
			controlWB <= 0;
			executeoutWB <= 0;
			branchoutWB <= 0;
		end
		else if( stall )
		begin
			//stall instrWB
			//stall controlWB
			//stall executeoutWB
			//stall branchoutWB
		end
		else if( clear )
		begin
			instrWB <= 0;
			controlWB <= 0;
			executeoutWB <= 0;
			branchoutWB <= 0;
		end
		else
		begin
			instrWB <= instr;
			controlWB <= control;
			executeoutWB <=  control[`CONTROL_ALUCONTROL] == `ALU_MUL		?	mult[31:0]
							:&control[`CONTROL_ALUCONTROL_SHIFT]			?	shiftout
							:control[`CONTROL_ALUCONTROL] == `ALU_MFHI		?	high
							:control[`CONTROL_ALUCONTROL] == `ALU_MFLO		?	low
							:control[`CONTROL_MEML]||control[`CONTROL_MEMR]	?	rt
							:													aluout;
			branchoutWB <= branchout;
			bottomaddressWB <= aluout[1:0];
			
			if( control[`CONTROL_ALUCONTROL] == `ALU_DIV )
			begin
				source <= 0;
			end
			else if( control[`CONTROL_ALUCONTROL] == (`ALU_DIV | `ALU_UNSIGNED) )
			begin
				source <= 1;
			end
		end
	end

	assign stallreq=
	(
		(control[`CONTROL_ALUCONTROL] == `ALU_MFHI || control[`CONTROL_ALUCONTROL] == `ALU_MFLO)
		&& countdown > 0
	)
	||
	(
		writeregenMduring && writeregnum!=5'd0 &&
		(
			(
				instr[25:21]==writeregnum && !control[`CONTROL_IRQRETURN]
				&& !control[`CONTROL_JUMP]
				&& !control[`CONTROL_REGJUMP]
				&& !control[`CONTROL_BRANCH]
				
			)
		||
			(
				instr[20:16]==writeregnum && !control[`CONTROL_IRQRETURN] && !control[`CONTROL_JUMP] 
				&& !control[`CONTROL_REGJUMP]
				&& !control[`CONTROL_BRANCH]
				&& (!control[`CONTROL_USEIMM] || control[`CONTROL_MEMWRITE] || control[`CONTROL_MEML] || control[`CONTROL_MEMR])
			)
		)
	);

	assign memread=		clear ? 0 : control[`CONTROL_MEMREAD];
	assign memwrite=	clear ? 0 : control[`CONTROL_MEMWRITE];
	assign mem16=		clear ? 0 : control[`CONTROL_MEM16];
	assign mem8=		clear ? 0 : control[`CONTROL_MEM8];
	assign memzerofill=	clear ? 0 : control[`CONTROL_ZEROFILL];
	assign memaddress=	clear ? 0 : (control[`CONTROL_MEML] || control[`CONTROL_MEMR]) ? {aluout[31:2],2'b00} : aluout;
	assign memwritedata=clear ? 0 : rt;
endmodule
