package reducer;

/*
 *   Copyright (C) A C Norman, April 2000
 *   Permission is given to use this code in any way you want for
 *   purposes directly concerned with the University of Cambridge taught
 *   courses in Computer Science and associated study. Anybody who
 *   wants to re-distribute this code or use it commercially should
 *   check with ACN (acn1@cam.ac.uk) first.
 */

import java.util.Hashtable;

// All programs and data will be represented in terms of the
// class "Graph" which is an abstract class that gets specialised
// to versions representing various sorts of leaf nodes and
// simple pairs of sub-graphs. A pair (f,x) is used to represent the
// function f applied to an argument x.

public abstract class Graph
{

// I put some support for printing (and in particular for keeping
// track of the number of characters displayed on a line) as static
// members of Graph just as a convenient place to put them.
  
public abstract void reduceAndPrint() throws TooManySteps;
public abstract void print() throws TooManySteps;

static int column = 0;
static public int lengthLeft;

public static boolean debugFlag;

static public long reductionSteps = 0;
static long reductionStepLimit = 10000000; // long-stop?

public void print(int lengthLimit)
{
    lengthLeft = lengthLimit;
    try
    {   this.print();
    }
    catch (TooManySteps e)
    {   text("...");
    }
}

public static void println(Graph x, int lengthLimit)
{
    lengthLeft = lengthLimit;      // avoid infinite output
    reductionSteps = 0;
    try
    {   if (x == null) text("<null>"); // to be cautious!
        else 
	{   x = makeHeadNormal(x);
	    x.reduceAndPrint();
	}
    }
    catch (TooManySteps e)
    {   text(" ...");
    }
    newline();
}

public static void text(String s)
{
    int l = s.length();
    if (column + l >= 75)
    {   System.out.println();
        column = 0;
    }
    System.out.print(s);
    column += l;
    lengthLeft -= l;
}

public static void newline()
{
    System.out.println();
    column = 0;
}


// The following table is used to set up various pre-defined
// constants, ie leaf items that act as particularly useful
// functions. I do not export (eg) +,- etc here since the parser
// provides normal infix notation for them and also the "op +" etc
// syntax.


public static void setup(boolean fg)
{
    debugFlag = fg;
    for (int i=0; i<constants.length; i++)
    {   values.put(constants[i][0], constants[i][1]);
// I associate the type with the name, but also with the name with
// a "@" prefixed. The first is for the user-visible predefined symbol,
// the second remains valid however much users redefine things.
        types.put(constants[i][0], constants[i][2]);
        types.put("@" + constants[i][0], constants[i][2]);
    }
}

static Object [][] constants =
{
    {"s",         new SCombinator(), Type.sType()},
    {"k",         new KCombinator(), Type.kType()},
// Only S and K are actually essential.
    {"i",         new ICombinator(), Type.iType()},
// I is "almost" essential, but (S K K) behaves as I
    {"y",         new YCombinator(), Type.yType()},
    {"b",         new BCombinator(), Type.bType()},
    {"c",         new CCombinator(), Type.cType()},
    {"b1",        new B1Combinator(), Type.b1Type()},
    {"c1",        new C1Combinator(), Type.c1Type()},
    {"s1",        new S1Combinator(), Type.s1Type()},
    {"p",         new PCombinator(), Type.pType()},
    {"u",         new UCombinator(), Type.uType()},
    {"if",        new IfCombinator(), Type.ifType()},
    {"true",      new Gboolean(true), new Type(Type.boolType)},
    {"false",     new Gboolean(false), new Type(Type.boolType)},
    {"+",         new PlusCombinator(), Type.binaryArithType()},
    {"-",         new DifferenceCombinator(), Type.binaryArithType()},
    {"*",         new TimesCombinator(), Type.binaryArithType()},
    {"/",         new QuotientCombinator(), Type.binaryArithType()},
    {"%",         new RemainderCombinator(), Type.binaryArithType()},
    {"<",         new LtCombinator(), Type.compareArithType()},
    {">",         new GtCombinator(), Type.compareArithType()},
    {"<=",        new LeCombinator(), Type.compareArithType()},
    {">=",        new GeCombinator(), Type.compareArithType()},
    {"=",         new EqCombinator(), Type.compareType()},
    {"!=",        new NeCombinator(), Type.compareType()},
    {"orelse",    new OrElseCombinator(), Type.boolopType()},
    {"andalso",   new AndAlsoCombinator(), Type.boolopType()}


};

public static Graph I  = new ICombinator();
public static Graph K  = new KCombinator();
public static Graph KI = new Application(K, I);

public static Graph gTrue  = new Gboolean(true);
public static Graph gFalse = new Gboolean(false);

static Hashtable values = new Hashtable();
static Hashtable types  = new Hashtable();

abstract boolean makeHeadNormal() throws TooManySteps;

// The main reduction process here avoids (Java-level) recursion
// by using the graph that is being traversed to track its own structure.
// Two important variables are used. "current" refers to the place in
// the graph where reduction is about to be attempted, while "trail"
// points back to a chain that makes it possible to reconstruct where
// we came from.

static Graph current;
static Application trail;

static public Graph makeHeadNormal(Graph x) throws TooManySteps
{
// See lecture notes for a discussion about Head Normal Form.
    Graph saveCurrent = current;
    Application saveTrail = trail; // saved to make recursive calls safe.
                                   // (I recurse when evaluating (+ 2 2) etc)
    current = x;
    trail = null;
    // The idea here is that if we have a function application as
    // current the situation will look like:
    //
    //      trail --------> somewhere
    //
    //      current ------> [ .   . ]
    //                       /     \
    //                     fn      arg
    //
    // and after one step this will be re-written to be
    //
    //                    somewhere
    //                        ^
    //                        |
    //      trail --------> [ .   . ]
    //                             \
    //      current -----> fn      arg
    //
    // while if the current item is some constant rewrites are done
    // that correspond to using it as a function. The makeHeadNormal()
    // method (withour any arguments) will do this, and returns true if
    // further reduction at current is required.  
    try  
    {   while (current != null &&
               current.makeHeadNormal())
        {   if (reductionSteps++ > reductionStepLimit)
                throw new TooManySteps();
        }
    }
// Observe use of "try .. finally" to guarantee that pointer-reversing
// is undone. Without this a "throw" from deep within could leave the
// graph structure seriously mangled.
    finally
    {   // When reduction can proceed no further makeHeadNormal() returns
        // false, and then the pointer-reversing illustrated above has to
        // be unwound to restore the graph to its original state.
        while (trail != null)
        {   Application w = (Application)trail.function;
            trail.function = current;
            current = trail;
            trail = w;
	}
        // All done!
        x = current;
        current = saveCurrent;
        trail = saveTrail;
    }
    if (x == null)
    {   System.out.println("null pointer created");
        throw new TooManySteps();
    }
    return x;
}

// The following are used to support conversion from lambda-form

boolean eqC(String name)
{   return false;
}

boolean eqCfn(String name)
{   return false;
}

boolean eqCfnfn(String name)
{   return false;
}

boolean eqV(String name)
{   return false;
}

Graph fn()
{   return null;
}

Graph arg()
{   return null;
}

Graph abs(String name)
{   return new Application(K, this);
}

public Graph substitute()
{
    return this;
}

public Type typeCheck(Environment e) throws NoType
{
    throw new NoType();
}

public Type typeCheck() throws NoType
{
    return typeCheck((Environment)null);
}

}


