(**************************************
Compiler Construction 2015
Computer Laboratory 
University of Cambridge 
Timothy G. Griffin (tgg22@cam.ac.uk) 
*****************************************) 
(* 
   This is a "stackified" 
   version of Interpreter 2. 
   That is, we see that continuations 
   are really stacks, and here we implement
   them with lists. 
   This is really a small change. 

   The other change made is to 
   merge interp_2.ml's mutually (tail-)recursive functions
   
   eval_dfc
   apply_closure
   apply_cnt   

   into a non-recursive single-step function (called step), 
   which is then iterated by a tail-recursive driver function (called driver). 

   This requires using a new state definition 

   type state = 
      | EVAL of expr * env * continuation 
      | APPLY of continuation * value 

   so that step : state -> state. 
*) 


open Ast 
open Common 

let verbose = ref false 

(* "semenatic" domain *) 

type value = 
     | CON of Common.constant 
     | PAIR of value * value 
     | INL of value 
     | INR of value 
     | FUN of closure 

and closure = Past.var * expr * env 

and continuation_item = 
  | C_UNARY of Past.unary_oper 
  | C_OPER of Past.oper * value 
  | C_OPER_FST of expr * env * Past.oper 
  | C_IF of expr * expr * env 
  | C_PAIR of value 
  | C_PAIR_FST of expr * env 
  | C_FST
  | C_SND
  | C_INL
  | C_INR
  | C_CASE of Past.var * expr * Past.var * expr * env 
  | C_APPLY of closure 
  | C_ARG of expr * env 

and continuation = continuation_item  list

and env = Past.var -> value 

type binding = Past.var * value

type bindings = binding list

(* update : (env * binding) -> env *) 
let update(env, (x, v)) = fun y -> if x = y then v else env y

let do_unary = function 
  | (Past.NOT, CON (BOOL m)) -> CON(BOOL (not m))
  | (Past.NEG, CON (INT m))  -> CON(INT (-m))
  | (Past.READ, CON(UNIT))   -> CON(INT (readint()))
  | (op, _) -> complain ("malformed unary operator: " ^ (Pp_past.string_of_unary_oper op))

let do_oper = function 
  | (Past.AND, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m && n))
  | (Past.OR,  CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m || n))
  | (Past.EQB, CON (BOOL m), CON (BOOL n)) -> CON(BOOL (m = n))
  | (Past.LT, CON (INT m), CON (INT n)) -> CON(BOOL (m < n))
  | (Past.EQI, CON (INT m), CON (INT n)) -> CON(BOOL (m = n))
  | (Past.ADD, CON (INT m), CON (INT n)) -> CON(INT (m + n))
  | (Past.SUB, CON (INT m), CON (INT n)) -> CON(INT (m - n))
  | (Past.MUL, CON (INT m), CON (INT n)) -> CON(INT (m * n))
  | (op, _, _)  -> complain ("malformed binary operator: " ^ (Pp_past.string_of_oper op))

type state = 
   | EVAL of expr * env * continuation 
   | APPLY of continuation * value 
 
let step = function 
 | EVAL(Unit, _,             k) -> APPLY(k, CON UNIT) 
 | EVAL(Var x,  env,         k) -> APPLY(k, env x)
 | EVAL(Integer n, _,        k) -> APPLY(k, CON(INT n))
 | EVAL(Boolean b, _,        k) -> APPLY(k, CON (BOOL b)) 
 | EVAL(UnaryOp(op, e), env, k) -> EVAL(e, env,  (C_UNARY op) :: k)
 | EVAL(Op(e1, op, e2), env, k) -> EVAL(e1, env, C_OPER_FST(e2, env, op) :: k)
 | EVAL(If(e1, e2, e3), env, k) -> EVAL(e1, env, C_IF(e2, e3, env) :: k)

 | EVAL(Pair(e1, e2), env,   k) -> EVAL(e1, env, C_PAIR_FST(e2, env) :: k)
 | EVAL(Fst e, env,          k) -> EVAL(e, env, C_FST :: k)
 | EVAL(Snd e, env,          k) -> EVAL(e, env, C_SND :: k) 

 | EVAL(Inl e, env,          k) -> EVAL(e, env, C_INL :: k) 
 | EVAL(Inr e, env,          k) -> EVAL(e, env, C_INR :: k) 
 | EVAL(Case(e, (x1, e1), (x2, e2)), env, k) -> EVAL(e, env, C_CASE(x1, e1, x1, e2, env) :: k)

 | EVAL(Lambda(x, body), env, k) -> APPLY(k, FUN(x, body, env))
 | EVAL(App(e1, e2), env,     k) -> EVAL(e1, env, C_ARG(e2, env) :: k)
 | EVAL(LetFun(f, (x, body), e), env, k) -> 
       let new_env = update(env, (f, FUN(x, body, env)))
       in EVAL(e, new_env, k) 
 | EVAL(LetRecFun(f, (x, body), e), env, k) -> 
       let rec new_env g = (* a recursive environment! *) 
           if g = f then FUN(x, body, new_env) else env g
       in EVAL(e, new_env, k) 
 | APPLY((C_UNARY op) :: k,                 v) -> APPLY(k ,(do_unary(op, v)))
 | APPLY(C_OPER(op, v1) :: k,              v2) -> APPLY(k, do_oper(op, v1, v2))
 | APPLY(C_OPER_FST (e2, env, op) :: k,    v1) -> EVAL(e2, env, C_OPER (op, v1) :: k)
 | APPLY((C_APPLY (x, body, env)) :: k,    v2) -> EVAL(body, update(env, (x, v2)), k)
 | APPLY(C_ARG (e2, env) :: k,         FUN cl) -> EVAL(e2, env, (C_APPLY cl) :: k)
 | APPLY((C_PAIR v1) :: k,                 v2) -> APPLY(k, PAIR(v1, v2))
 | APPLY(C_PAIR_FST (e2, env) :: k,        v1) -> EVAL(e2, env, (C_PAIR v1) :: k)
 | APPLY(C_FST :: k,               PAIR(v, _)) -> APPLY(k, v)
 | APPLY(C_SND :: k,               PAIR(_, v)) -> APPLY(k, v)
 | APPLY(C_INL :: k,                        v) -> APPLY(k, (INL v))
 | APPLY(C_INR :: k,                        v) -> APPLY(k, (INR v))
 | APPLY(C_CASE (x1, e1, x2, e2, env) :: k, INL v) -> EVAL(e1, update(env, (x1, v)), k) 
 | APPLY(C_CASE (x1, e1, x2, e2, env) :: k, INR v) -> EVAL(e2, update(env, (x2, v)), k)
 | APPLY(C_IF (e2, e3, env) :: k,  CON(BOOL true))  -> EVAL(e2, env, k)
 | APPLY(C_IF (e2, e3, env) :: k,  CON(BOOL false)) -> EVAL(e3, env, k)
 | state -> complain "step : malformed state" 

let rec driver = function 
  | APPLY([], v) -> v 
  | state -> driver (step state) 

let eval(e, env) = driver(EVAL(e, env, []))

(* env_empty : env *) 
let env_empty = fun x -> complain (x ^ " is not defined!\n")

(* interpret : expr -> value *) 
let interpret e = eval(e, env_empty)

let rec string_of_value = function 
     | CON c -> string_of_constant c
     | PAIR(v1, v2) -> "(" ^ (string_of_value v1) ^ ", " ^ (string_of_value v2) ^ ")"
     | INL v -> "inl(" ^ (string_of_value v) ^ ")"
     | INR  v -> "inr(" ^ (string_of_value v) ^ ")"
     | FUN _ -> "FUNCTION( ... )" 
    

      
    
    
