//
// SystemC TLM wrapper for fast ISS of OR1200 processor core.
// $Id: $
//
//

#include "systemc.h"
#include "OR1200F.h"
#include "llsc_extension.h"

extern int g_full_traces;

const int enable_asym = 5;

// Constructor
OR1200F::OR1200F(sc_core::sc_module_name names, u8_t pID, bool harvardf):
  OR1200IF(names, pID, harvardf, pID *enable_asym), // INTRODUCE CORE ASYM
  ins_fetcher(this),
  orsim(names)
{
#ifdef TLM_POWER3
  //  std::cout << "constructing OR1200F cpu_busaccess this=" << ((void *)&busaccess) << " pw_module_base=" << ((void *)dynamic_cast<pw_module_base *>(&busaccess)) << "\n";
  instruction_energy = pw_energy(200.0, PW_pJ);
  pw_power leakage = pw_power(10.0, PW_mW);
  set_static_power(leakage);
  set_fixed_area(pw_area(0.5, PW_sqmm)); // Without caches
#endif
  build_automaton();
  lt_i_delay = SC_ZERO_TIME;
  lt_d_delay = SC_ZERO_TIME;
  ext_interrupt=0;
  SC_THREAD(run);
  char txt[20];
  sprintf(txt, "OR1200F_CORE_%d", procID);
  where_last = 0;
  ext_interrupt = 0;
  reset(false);
  printf("Made %s %s\n", txt, name());
  //std::cout << "constructing 2/2 OR1200F cpu_busaccess this=" << ((void *)&busaccess) << " pw_module_base=" << ((void *)dynamic_cast<pw_module_base *>(&busaccess)) << "\n";
  if (g_full_traces) busaccess.start_trace();
}

void OR1200F::reset(bool selfstart)
{
  halted = !selfstart;
  busaccess.trace_msg("RESET");
  core_reset();
}

// Elaboration phase: connect the interrupt wire to this core using this call.
bool *OR1200F::connect_interrupt(bool *c)
{
  bool *ov = ext_interrupt;
  ext_interrupt = c;
  return ov;
}



// Instruction mini-cache for other half of a 64 bit word.
void OR1200F::ins_fetcher_t::fetch(u32_t adr, u32_t &i, sc_time &lt_delay)
{ 
  u32_t a1 = adr & ~7;
  if (a1 != cached_adr)
    {
      parent->busaccess.instr_fetch64(a1, data, 0, lt_delay);
      cached_adr = a1;
    }
  // Return big-endian order.
  i = (adr & 4) ? (data >> 0) : (data >> 32); 
}


void OR1200F::trace100(const char *msg) 
{
  busaccess.trace100(msg);
}

// Main model process loop
void OR1200F::run()
{
  over = false;
  while(!over)
    {
      if (m_qk.need_sync()) m_qk.sync();   // Keeper synchronize when quantum is reached
      if (ext_interrupt && *ext_interrupt) // Level-sensitive interrupt.
	{
	  if (busaccess.traceregions && busaccess.traceregions->check(0, TENOS_TRACE_CPU_INTERRUPTS))
	    printf("%s: %s: INTERRUPT %i\n", name(), kind(), procID); // This is an unmasked version of the interrupt signal.
	  hw_interrupt("orsim_sc");
	}
      if (halted)
	{
	  m_qk.sync();
	  wait(1, SC_US); // must poll for interrupt in halted TODO make an event
	  continue;
	}
      //busaccess.tick();
      sc_time ins_start = m_qk.get_current_time();
      lt_i_delay = SC_ZERO_TIME; //
      lt_d_delay = SC_ZERO_TIME; //
      if (countdown > 0) countdown -= 1;
      else 
	{
	  step();
	  POWER3(record_energy_use(instruction_energy));
	}

      sc_time d_end = lt_d_delay + sc_time_stamp(); // Bus cycle end time
      sc_time i_end = lt_i_delay + sc_time_stamp(); // Bus cycle end time
      sc_time ins_end = ins_start+core_period;  // One instruction per core_period.

      //cout << name () << " lt_busdelay was " << lt_busdelay << "\n";
      //cout << name () << " ext_end was " << d_end << " and " << i_end "\n";
      //cout << name () << " ins_end was " << ins_end << "\n";

      // Retire join: take maximum of internal and external delays at join.
#define TMAX(X, Y) ((X)>(Y)?(X):(Y))
      //cout << "kernel=" << sc_time_stamp() << " start=" << ins_start << " ext_end=" << ext_end << " ins_end=" << ins_end << "\n";
      m_qk.set(TMAX(ins_end, TMAX(d_end, i_end))-sc_time_stamp());


      oraddr_t pc = get_pc(); // peek_into_itlb(cpu_state.iqueue.insn_addr);
      if (busaccess.traceregions)
	{
	  int m = busaccess.traceregions->check(pc, TENOS_TRACE_CPU_KEY_REGS|TENOS_TRACE_CPU_ALL_REGS);
	  if (m) dumpreg(stdout, m & TENOS_TRACE_CPU_ALL_REGS);
	}
    }
  m_qk.sync();
}






uint32_t OR1200F::eval_insn(unsigned int memaddr, int *bp)
{
  uint32_t rdata;
  ins_fetcher.fetch(memaddr, rdata, harvardf ? lt_i_delay: lt_d_delay);  //Service instruction fetch charging delay to appropriate account.x
  if (busaccess.traceregions && busaccess.traceregions->check(memaddr, TENOS_TRACE_CPU_IFETCH))  
    {
      disassemble_insn(rdata);
      printf("%s:%s: Fetch insn a=%x ins=%x %s\n", name(), kind(), memaddr, rdata, disassembled);
    }
  where_last = memaddr;
  return rdata;
}


void OR1200F::sim_done(const char *msg)
{
  busaccess.sim_done(msg);
}


void OR1200F::corepause(int microseconds, u32_t addr)  // Pause CPU for this time interval: this makes it take less static and dynamic power for this interval.
{
  if (microseconds < 0 || microseconds > 2) microseconds = 2;
  sc_time pi = sc_time(microseconds, SC_US);
  u64_t new_delperiods = pi / core_period;

  if (busaccess.traceregions && busaccess.traceregions->check(busaccess.lastfetch, TENOS_TRACE_CPU_IFETCH))  
    {
      printf("%s:%s: Pause %i microseconds (%li core periods)\n", name(), kind(), microseconds, new_delperiods);
    }
  //printf("%s:%s: Pause %i microseconds (%i core periods)\n", name(), kind(), microseconds, new_delperiods);
  log_delperiods(new_delperiods, addr);
  // We need to implement the halt and interrupt material as well...
  m_qk.sync();
  wait(pi);
}




void OR1200F::atomic_prefix() // Prefix following load/store pair as atomic.
{
  //printf("%s: atomic\n", name());
  busaccess.atomic_prefix(); // Trigger atomic transaction state machine.
}



u32_t OR1200F::eval_mem32(oraddr_t memaddr, int*)
{
  u32_t rd;
  busaccess.eval_mem32(xlat32to64(memaddr), rd, 0, lt_d_delay);
  return rd;
}

 
u16_t OR1200F::eval_mem16(oraddr_t memaddr, int*)
{
  u16_t rd;
  busaccess.eval_mem16(xlat32to64(memaddr), rd, 0, lt_d_delay);
  return rd;
}

u8_t OR1200F::eval_mem8(oraddr_t memaddr, int*)
{
  u8_t rd;
  busaccess.eval_mem8(xlat32to64(memaddr), rd, 0, lt_d_delay);
  return rd;
}


void OR1200F::set_direct8(oraddr_t a, u8_t d, int, int)
{
  assert(0);
}

u8_t OR1200F::eval_direct8(oraddr_t a, int, int)
{
  assert(0);
}

void OR1200F::set_mem32(oraddr_t memaddr, u32_t wd, int*)
{
  busaccess.set_mem32(xlat32to64(memaddr), wd, 0, lt_d_delay);
}

void OR1200F::set_mem16(oraddr_t memaddr, u16_t wd, int*)
{
  busaccess.set_mem16(xlat32to64(memaddr), wd, 0, lt_d_delay);
}

void OR1200F::set_mem8(oraddr_t memaddr, uint8_t wd, int*)
{
  busaccess.set_mem8(xlat32to64(memaddr), wd, 0, lt_d_delay);

}


void OR1200F::stat_report(const char *msg, FILE *fd, bool resetf)
{
  const char *p = core_period.to_string().c_str();
  if (fd) fprintf(fd, "%s %s  period=%s \n", name(), msg, p);
  core_stat_report(msg, fd, resetf);
  busaccess.stat_report(msg, fd, resetf);
  if (resetf) stats.reset();
  
}

// TENOS_KIND_DEFINITION(OR1200F)

// eof
