// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_vcd_observer.cpp,v 1.2 2011/07/25 15:32:45 my294 Exp $

/** @file pw_vcd_observer.cpp
 * @brief Record the power variation for generating a VCD trace.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 *
 * This class is used by the VCD trace file for generating a VCD trace of the 
 * power variation of a SystemC component.
 */

#include <sstream>
#include "iostream"
#include <systemc>

#include "pw_vcd_observer.h"
#include "pw_vcd_trace.h"
#include "pw_stat_observer_base.h"
#include "pw_tracing_ids.h"
#include "pw_debug.h"
#include "pw_physical_operators.h"

using namespace std;

// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
// Constructor
  pw_vcd_observer::pw_vcd_observer(sc_core::sc_object& p_obj,
				   const std::string& p_name
				  ):
    pw_accounting_base(p_obj, p_name),
    a_utilization(SC_ZERO_TIME),
    a_utilization_value(0.)
  { }

  // ---------------------------------------------------------------------------

  //! Update transaction energy/power - read transaction event from pw_module base.
  void pw_vcd_observer::update_energy_infos(const pw_module_base& p_module)
  {
    pw_energy l_energy = p_module.get_transaction_energy();
    int l_acct = p_module.get_transaction_acct();
    auto_start_acct(l_acct);
    a_accts[l_acct].record_energy(l_energy);
    if (l_energy > PW_ZERO_ENERGY) a_monitors[l_acct].new_energy(get_energy(l_acct)); // unnecessary diff then integrate  TODO
  }
  
  void pw_vcd_observer::vcd_monitor::new_energy(pw_energy new_e)
  {
    pw_energy q = new_e - last_e;
    sc_time t = sc_core::sc_time_stamp();
    assert(trc);
    trc->log(t, q);
    last_e = new_e;
  }


  // ---------------------------------------------------------------------------

  void pw_vcd_observer::update(pw_subject& p_subject)
  {
    try
      {
        const pw_module_base& l_module =
	  dynamic_cast< const pw_module_base& >(p_subject);
	
        switch(l_module.get_modified_attribute())
	  {
	  case pw_module_base::INIT:
	    pw_debug << "Initialization of VCD observer <"
		     << get_obj_name() << ", "
		     << get_att_name() << ">" << pw_endl;
	    //                update_dynamic_power(l_module);
	    //                update_static_power(l_module);
	    break;
	    
	  case pw_module_base::PHASEMODE_POWER:
	    pw_debug << "Phasemode Power Update of VCD observer <"
		     << get_obj_name() << ", "
		     << get_att_name() << ">" << pw_endl;
	    update_energy_infos(l_module);
	    break;
	    
	  case pw_module_base::TRANSACTION:
	    pw_debug << "Update transaction energies of VCD observer <"
		     << get_obj_name() << ", "
		     << get_att_name() << ">" << pw_endl;
	    update_energy_infos(l_module);
	    break;

	  case pw_module_base::QUANTUM_UP:
	    pw_debug << "Housekeeping for last quantum of VCD observer <"
		     << get_obj_name() << ", "
		     << get_att_name() << ">" << pw_endl;
	    break;
	  case pw_module_base::LOG_EVENT:
	    pw_debug << "Housekeeping for last logging interval of VCD observer <"
		     << get_obj_name() << ", "
		     << get_att_name() << ">" << pw_endl;
	    update_log_values(l_module);
	    break;
	  default:
	    pw_assert(false
                          and "Incoherent call of update.");
	    break;
	  }
      }
    catch(std::bad_cast)
      {
        pw_assert(false
                  && "Casting problem with the \"Observer\" design pattern");
      }
  }

  // ---------------------------------------------------------------------------

  void pw_vcd_observer::trace(sc_pwr::pw_vcd_trace_file *p_file_pt, const std::string& p_str, plot_control_t p_plot_control) // p_str is an optional alias
  {
    // Install account names for the first n standard accounts (if none already started).
    start_default_accounts();
    int plot_column = (p_plot_control == PlotCombinedAccounts) ? p_file_pt->alloc_next_plot_column(p_str): -1;
    for (int a=0; a<PW_ACCT_MAX_NO_ACCOUNTS; a++) 
      if (pw_stat_observer_base::global_inuse(a)) 
	{
	  std::string l_name = pw_stat_observer_base::get_global_name(a);
	  int plot_column1 = (p_plot_control == PlotEachAccount) ? p_file_pt->alloc_next_plot_column(l_name) : plot_column;
	  a_monitors[a].trc = p_file_pt -> add_trace(a, PW_ZERO_ENERGY, p_str + "_energy_" + l_name, plot_column1);
	}
    
    // TODO add in p_file_pt -> add_trace(-1, a_utilization_value, p_str+".util");
  }

// ---------------------------------------------------------------------------
  const std::string pw_vcd_observer::to_string(void)
  {
    std::ostringstream l_os;
    
    l_os << "{ " << a_obj.name()
         << " ," << a_utilization_value
	 << " ," << get_power()
	 << " ," << get_energy()
	 << " }";
    
    return l_os.str();
  }
  
  
  // ---------------------------------------------------------------------------
  void pw_vcd_observer::update_log_values(const pw_module_base& p_module)
  {
    sc_time l_tnow = sc_time_stamp();
    for (int a=0; a<PW_ACCT_MAX_NO_ACCOUNTS; a++) if (a_accts[a].inuse()) a_accts[a].fold_power_to_energy(l_tnow);
    
    
    pw_debug << "{ energy = " << get_energy() 
	     << ", time = "  << l_tnow << "}" << pw_endl;
    
    sc_time l_util = a_utilization; // - a_logged_utilization;
    a_utilization_value = l_util.to_double(); // l_time.to_double();
  }
  
  
  // ---------------------------------------------------------------------------
  // Output stream operator of energy object.
  std::ostream& operator<<(std::ostream& p_os,
			   pw_vcd_observer& p_observer)
  {
    std::ostream::sentry l_init(p_os);
    if(!l_init)
      {
        SC_REPORT_ERROR(PW_TRACING_OSTREAM_ERROR_TYPE_,
                        PW_TRACING_OSTREAM_ERROR_MSG_);
        return p_os;
      }
    
    return (p_os << p_observer.to_string());
  }


} // namespace sc_pwr

// eof
