// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_energy.cpp,v 1.2 2011/07/25 15:32:45 my294 Exp $

/** @file pw_energy.cpp
 * @brief Define energy data type.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#include <ios>
#include <cmath>
#include <sstream>
#include <iomanip>
#include <iostream>
#include <algorithm>

#include "pw_energy.h"
#include "pw_kernel_ids.h"
#include "pw_debug.h"
#include "pw_common.h"

// ===========================================================================
namespace
{


// ---------------------------------------------------------------------------
using sc_dt::uint64;


// ---------------------------------------------------------------------------
//! Convertion table of energy unit into string.
const char* g_tab_energy_units[] =
{
    "fJ",
    "pJ",
    "nJ",
    "uJ",
    "mJ",
    "J"
};


// ---------------------------------------------------------------------------
const double g_tab_energy_scales[] =
{
    1,       // fJ
    1e3,     // pJ
    1e6,     // nJ
    1e9,     // uJ
    1e12,    // mJ
    1e15     // J
};


// ---------------------------------------------------------------------------
/**
 * @brief Test if an uint is a power of 10.
 * @param[in] p_val The uint to test.
 * @return true only if p_val is a energy of 10.
 */
static bool is_pow10(double p_val)
{
    double l_tmp;
    return std::modf(std::log10(p_val), &l_tmp) == 0.0;
}


// ---------------------------------------------------------------------------
//! Simple rinting function
inline double pw_rint(double p_val)                                                     
{
    return std::floor( p_val + 0.5 );
}

} // ANONYMOUS NAMESPACE (replace static declaration)


// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
using sc_dt::uint64;


// ---------------------------------------------------------------------------
//! Energy resolution in femto joule.
double pw_energy::a_resolution_value = g_tab_energy_scales[PW_pJ];
//! True if a pw_energy different of 0 have been constructed.
bool pw_energy::a_is_resolution_fixed = false;


// ---------------------------------------------------------------------------
//! ZERO energy constant
const pw_energy PW_ZERO_ENERGY;


// ---------------------------------------------------------------------------
pw_energy_unit& operator++(pw_energy_unit& p_unit)
{
    if(p_unit < PW_fJ)
    {
        p_unit = PW_fJ;
        SC_REPORT_WARNING(PW_ENERGY_UNIT_OVERFLOW_TYPE_,
                          PW_ENERGY_UNIT_OVERFLOW_MSG_);
    }
    if(p_unit >= PW_JOULE)
    {
        p_unit = PW_JOULE;
        SC_REPORT_WARNING(PW_ENERGY_UNIT_OVERFLOW_TYPE_,
                          PW_ENERGY_UNIT_OVERFLOW_MSG_);
        return p_unit;
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    p_unit = static_cast< pw_energy_unit >(++l_uint);

    pw_assert(p_unit > PW_fJ and p_unit <= PW_JOULE \
              and "Unexpected overflow!!!");

    return p_unit;
}


// ---------------------------------------------------------------------------
pw_energy_unit operator++(pw_energy_unit& p_unit,
                          int)
{
    if(p_unit < PW_fJ)
    {
        p_unit = PW_fJ;
        SC_REPORT_WARNING(PW_ENERGY_UNIT_UNDERFLOW_TYPE_,
                          PW_ENERGY_UNIT_UNDERFLOW_MSG_);
    }
    if(p_unit >= PW_JOULE)
    {
        p_unit = PW_JOULE;
        SC_REPORT_WARNING(PW_ENERGY_UNIT_OVERFLOW_TYPE_,
                          PW_ENERGY_UNIT_OVERFLOW_MSG_);
        return p_unit;
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    pw_energy_unit l_unit = static_cast< pw_energy_unit >(++l_uint);

    pw_assert(l_unit > PW_fJ and l_unit <= PW_JOULE \
              and "Unexpected overflow!!!");

    return l_unit;
}


// ---------------------------------------------------------------------------
pw_energy_unit& operator--(pw_energy_unit& p_unit)
{
    if(p_unit <= PW_fJ)
    {
        p_unit = PW_fJ;
        SC_REPORT_WARNING(PW_ENERGY_UNIT_UNDERFLOW_TYPE_,
                          PW_ENERGY_UNIT_UNDERFLOW_MSG_);
        return p_unit;
    }
    if(p_unit > PW_JOULE)
    {
        p_unit = PW_JOULE;
        SC_REPORT_WARNING(PW_ENERGY_UNIT_OVERFLOW_TYPE_,
                          PW_ENERGY_UNIT_OVERFLOW_MSG_);
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    p_unit = static_cast< pw_energy_unit >(--l_uint);

    pw_assert(p_unit >= PW_fJ and p_unit < PW_JOULE \
              and "Unexpected overflow!!!");

    return p_unit;
}


// ---------------------------------------------------------------------------
pw_energy_unit operator--(pw_energy_unit& p_unit,
                          int)
{
    if(p_unit <= PW_fJ)
    {
        p_unit = PW_fJ;
        SC_REPORT_WARNING(PW_ENERGY_UNIT_UNDERFLOW_TYPE_,
                          PW_ENERGY_UNIT_UNDERFLOW_MSG_);
        return p_unit;
    }
    if(p_unit > PW_JOULE)
    {
        p_unit = PW_JOULE;
        SC_REPORT_WARNING(PW_ENERGY_UNIT_OVERFLOW_TYPE_,
                          PW_ENERGY_UNIT_OVERFLOW_MSG_);
    }

    unsigned l_uint = static_cast< unsigned >(p_unit);
    pw_energy_unit l_unit = static_cast< pw_energy_unit >(--l_uint);

    pw_assert(l_unit >= PW_fJ and l_unit < PW_JOULE \
              and "Unexpected overflow!!!");

    return l_unit;
}


// ---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& p_os,
                         const pw_energy_unit& p_unit)
{
    std::ostream::sentry l_init(p_os);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_ENERGY_UNIT_OSTREAM_ERROR_TYPE_,
                        PW_ENERGY_UNIT_OSTREAM_ERROR_MSG_);
        return p_os;
    }

    if(p_unit < PW_fJ or p_unit > PW_JOULE)
    {
        SC_REPORT_ERROR(PW_ENERGY_UNIT_OVERFLOW_TYPE_,
                        PW_ENERGY_UNIT_OVERFLOW_MSG_);
        return (p_os << "undef");
    }

    return (p_os << g_tab_energy_units[p_unit]);
}


// ---------------------------------------------------------------------------
std::istream& operator>>(std::istream& p_is,
                         pw_energy_unit& p_unit)
{
    std::istream::sentry l_init(p_is,
                                false);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_ENERGY_UNIT_ISTREAM_ERROR_TYPE_,
                        PW_ENERGY_UNIT_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    std::string l_unit_idf;
    p_is >> l_unit_idf;
    if(not p_is)
    {
        SC_REPORT_ERROR(PW_ENERGY_UNIT_ISTREAM_ERROR_TYPE_,
                        PW_ENERGY_UNIT_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    unsigned l_pos = 0;
    const unsigned l_size =
        sizeof(g_tab_energy_units) / sizeof(const char*);
    while(l_pos < l_size
          and l_unit_idf != g_tab_energy_units[l_pos])
        ++l_pos;

    if(l_pos == l_size)
    {
        p_is.setstate(std::ios_base::failbit);
        SC_REPORT_ERROR(PW_ENERGY_UNIT_ISTREAM_ERROR_TYPE_,
                        PW_ENERGY_UNIT_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    p_unit = static_cast< pw_energy_unit >(l_pos);
    return p_is;
}


// ---------------------------------------------------------------------------
pw_energy::pw_energy(void):
    a_value(0)
{ }


// ---------------------------------------------------------------------------
pw_energy::pw_energy(double p_value,
                     pw_energy_unit p_unit):
    a_value(0)
{
    if(p_unit < PW_fJ or p_unit > PW_JOULE)
    {
        SC_REPORT_ERROR(PW_ENERGY_UNIT_OVERFLOW_TYPE_,
                        PW_ENERGY_UNIT_OVERFLOW_MSG_);
        return;
    }

    if(p_value < 0.0)
    {
        a_value = 0;
        SC_REPORT_ERROR(PW_ENERGY_NEGATIVE_TYPE_,
                        PW_ENERGY_NEGATIVE_MSG_);
        return;
    }

    if(p_value == 0) return;

    double l_scaling_factor =
        g_tab_energy_scales[p_unit] / a_resolution_value;
    double l_value = p_value * l_scaling_factor;
//       printf("p_value %f  l_scaling  %f \n", p_value, l_scaling_factor);

    /* XXX conversion problem when pw_rint(l_value) == PW_UINT64_MAX
     * We have to manage when these two values are equal.
     */
    if(pw_rint(l_value) >= (static_cast<double>(PW_UINT64_MAX)))
    {
        if(pw_rint(l_value) > (static_cast<double>(PW_UINT64_MAX)))
        {
            SC_REPORT_WARNING(PW_ENERGY_OVERFLOW_TYPE_,
                              PW_ENERGY_OVERFLOW_MSG_);
        }
        a_value = PW_UINT64_MAX;
        return;
    }

    /* XXX DEPRECATED XXX
     * Arithmetic operation with double are not enough accurate for checking 
     * the possible loss of information.
    double l_tmp;
    if(std::modf(l_value, &l_tmp) != 0.0)
    {
        SC_REPORT_WARNING(PW_ENERGY_OVERFLOW_TYPE_,
                          PW_ENERGY_OVERFLOW_MSG_);
    }
    XXX DEPRECATED XXX*/

    a_value = static_cast<uint64>(pw_rint(l_value));
    if(a_value != 0)
    {
        if(not a_is_resolution_fixed)
            pw_info << "Resolution Fixed to: " << a_resolution_value
                    << " Femto Joule" << pw_endl;
        a_is_resolution_fixed = true;
    }
}


// ---------------------------------------------------------------------------
pw_energy::pw_energy(const pw_energy& p_energy):
    a_value(p_energy.a_value)
{ }


// ---------------------------------------------------------------------------
pw_energy& pw_energy::operator=(const pw_energy& p_energy)
{
    a_value = p_energy.a_value;
    return *this;
}

// ---------------------------------------------------------------------------
pw_energy& pw_energy::operator+=(const pw_energy& p_energy)
{

    if(PW_UINT64_MAX - p_energy.a_value < a_value)
    {
      //printf("Ina tmp += p_energy %llx  a_value %llx \n", p_energy.a_value, a_value);
      //breakpoint();
        a_value = PW_UINT64_MAX;
        SC_REPORT_WARNING(PW_ENERGY_OVERFLOW_TYPE_,
                          PW_ENERGY_OVERFLOW_MSG_);
    }
    else
        a_value += p_energy.a_value;

    //std::cout << "pw_energy += " << a_value << " after add " <<  p_energy.a_value << "\n";
    return *this;
}


// ---------------------------------------------------------------------------
pw_energy& pw_energy::operator-=(const pw_energy& p_energy)
{
    if(p_energy.a_value > a_value)
    {
        a_value = 0;
        SC_REPORT_WARNING(PW_ENERGY_UNDERFLOW_TYPE_,
                          PW_ENERGY_UNDERFLOW_MSG_);
    }
    else
        a_value -= p_energy.a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_energy& pw_energy::operator*=(double p_value)
{
    if(p_value == 0 or a_value == 0)
        a_value = 0;
    else
    {
        double l_tmp = to_double() * p_value;
        a_value = static_cast<uint64>(pw_rint(l_tmp));
        if(std::fabs(l_tmp - static_cast< double >(a_value)) > 1.)
        {
            a_value = PW_UINT64_MAX;
            SC_REPORT_WARNING(PW_ENERGY_OVERFLOW_TYPE_,
                              PW_ENERGY_OVERFLOW_MSG_);
        }
    }

    return *this;
}


// ---------------------------------------------------------------------------
pw_energy& pw_energy::operator/=(double p_value)
{
    if(p_value == 0.)
    {
        a_value = PW_UINT64_MAX;
        SC_REPORT_ERROR(PW_ENERGY_DIVISION_BY_ZERO_TYPE_,
                        PW_ENERGY_DIVISION_BY_ZERO_MSG_);
        return *this;
    }

    double l_tmp = to_double() / static_cast< double >(p_value);
    a_value = static_cast< uint64 >(pw_rint(l_tmp));

    return *this;
}


// ---------------------------------------------------------------------------
pw_energy& pw_energy::operator++(void)
{
    if(PW_UINT64_MAX == a_value)
    {
        SC_REPORT_WARNING(PW_ENERGY_OVERFLOW_TYPE_,
                          PW_ENERGY_OVERFLOW_MSG_);
    }
    else
        ++a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_energy pw_energy::operator++(int)
{
    if(PW_UINT64_MAX == a_value)
    {
        SC_REPORT_WARNING(PW_ENERGY_OVERFLOW_TYPE_,
                          PW_ENERGY_OVERFLOW_MSG_);
        return *this;
    }

    pw_energy l_res;
    l_res.a_value = a_value;
    ++a_value;
    return l_res;
}


// ---------------------------------------------------------------------------
pw_energy& pw_energy::operator--(void)
{
    if(0 == a_value)
    {
        SC_REPORT_WARNING(PW_ENERGY_UNDERFLOW_TYPE_,
                          PW_ENERGY_UNDERFLOW_MSG_);
    }
    else
        --a_value;

    return *this;
}


// ---------------------------------------------------------------------------
pw_energy pw_energy::operator--(int)
{
    if(0 == a_value)
    {
        SC_REPORT_WARNING(PW_ENERGY_UNDERFLOW_TYPE_,
                          PW_ENERGY_UNDERFLOW_MSG_);
        return *this;
    }

    pw_energy l_tmp;
    l_tmp.a_value = a_value;
    --a_value;
    return l_tmp;
}


// ---------------------------------------------------------------------------
const std::string pw_energy::to_string(void) const
{
    if(a_value == 0)
    {
        std::ostringstream l_os;
        l_os << "0" << " " << PW_JOULE;
        return l_os.str();
    }

    pw_assert(pw_energy::a_resolution_value <=
              (static_cast<double>(PW_UINT64_MAX))
              and "Incoherent energy resolution value.");

    uint64 l_resolution_value =
        static_cast<uint64>(pw_energy::a_resolution_value);
    unsigned l_zero_cpt = 0;
    while((l_resolution_value % 10) == 0)
    {
        ++l_zero_cpt;
        l_resolution_value /= 10;
    }
    uint64 l_value = a_value;
    while((l_value % 10) == 0)
    {
        ++l_zero_cpt;
        l_value /= 10;
    }
    std::ostringstream l_os;
    l_os << l_value;

    pw_energy_unit l_unit = PW_fJ;
    while(l_zero_cpt >= 3 and l_unit < PW_JOULE)
    {
        ++l_unit;
        l_zero_cpt -= 3;
    }
    for(unsigned l_cpt = 0;
        l_cpt < l_zero_cpt;
        ++l_cpt)
        l_os << "0";
    l_os << " " << g_tab_energy_units[l_unit];

    return l_os.str();
}


// ---------------------------------------------------------------------------
uint64 pw_energy::value(void) const
{
    return this->a_value;
}


// ---------------------------------------------------------------------------
double pw_energy::to_double(void) const
{
    return static_cast< double >(this->a_value);
}


// ---------------------------------------------------------------------------
double pw_energy::to_joules(void) const
{
    return static_cast< double >(this->a_value) \
        * (pw_energy::a_resolution_value / g_tab_energy_scales[PW_JOULE]);
}


// ---------------------------------------------------------------------------
pw_energy pw_energy::max(void)
{
    pw_energy l_energy(pw_get_energy_resolution());
    l_energy.a_value = PW_UINT64_MAX;
    return l_energy;
}


// ---------------------------------------------------------------------------
bool operator==(const pw_energy& p_energy1,
                const pw_energy& p_energy2)
{
    return p_energy1.a_value == p_energy2.a_value;
}


// ---------------------------------------------------------------------------
bool operator!=(const pw_energy& p_energy1,
                const pw_energy& p_energy2)
{
    return p_energy1.a_value != p_energy2.a_value;
}


// ---------------------------------------------------------------------------
bool operator<(const pw_energy& p_energy1,
               const pw_energy& p_energy2)
{
    return p_energy1.a_value < p_energy2.a_value;
}


// ---------------------------------------------------------------------------
bool operator>(const pw_energy& p_energy1,
               const pw_energy& p_energy2)
{
    return p_energy1.a_value > p_energy2.a_value;
}


// ---------------------------------------------------------------------------
bool operator<=(const pw_energy& p_energy1,
                const pw_energy& p_energy2)
{
    return p_energy1.a_value <= p_energy2.a_value;
}


// ---------------------------------------------------------------------------
bool operator>=(const pw_energy& p_energy1,
                const pw_energy& p_energy2)
{
    return p_energy1.a_value >= p_energy2.a_value;
}


// ---------------------------------------------------------------------------
pw_energy operator+(const pw_energy& p_energy1,
                    const pw_energy& p_energy2)
{
    pw_energy l_energy(p_energy1);
    //std::cout << "pw_energy operator+\n";
    return l_energy += p_energy2;
}


// ---------------------------------------------------------------------------
pw_energy operator-(const pw_energy& p_energy1,
                    const pw_energy& p_energy2)
{
    pw_energy l_energy(p_energy1);
    return l_energy -= p_energy2;
}


// ---------------------------------------------------------------------------
pw_energy operator*(const pw_energy& p_energy,
                    double p_value)
{
    pw_energy l_energy(p_energy);
    return l_energy *= p_value;
}


// ---------------------------------------------------------------------------
pw_energy operator*(double p_value,
                    const pw_energy& p_energy)
{
    pw_energy l_energy(p_energy);
    return l_energy *= p_value;
}


// ---------------------------------------------------------------------------
double operator/(const pw_energy& p_energy1,
                 const pw_energy& p_energy2)
{
    return p_energy1.to_double() / p_energy2.to_double();
}


// ---------------------------------------------------------------------------
pw_energy operator/(const pw_energy& p_energy,
                    double p_value)
{
    pw_energy l_energy(p_energy);
    return l_energy /= p_value;
}


// ---------------------------------------------------------------------------
std::ostream& operator<<(std::ostream& p_os,
                         const pw_energy& p_energy)
{
    std::ostream::sentry l_init(p_os);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_ENERGY_OSTREAM_ERROR_TYPE_,
                        PW_ENERGY_OSTREAM_ERROR_MSG_);
        return p_os;
    }

    if(p_os.flags()
       & (std::ios_base::scientific | std::ios_base::fixed))
        return p_os << p_energy.to_joules() << " "
                    << g_tab_energy_units[PW_JOULE];
    else
        return p_os << p_energy.to_string();
}


// ---------------------------------------------------------------------------
std::istream& operator>>(std::istream& p_is,
                         pw_energy& p_energy)
{
    std::istream::sentry l_init(p_is,
                                false);
    if(!l_init)
    {
        SC_REPORT_ERROR(PW_ENERGY_ISTREAM_ERROR_TYPE_,
                        PW_ENERGY_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    double l_value = 0;
    p_is >> l_value;
    if(not p_is)
    {
        SC_REPORT_ERROR(PW_ENERGY_ISTREAM_ERROR_TYPE_,
                        PW_ENERGY_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    pw_energy_unit l_unit;
    p_is >> l_unit;
    if(not p_is)
    {
        SC_REPORT_ERROR(PW_ENERGY_ISTREAM_ERROR_TYPE_,
                        PW_ENERGY_ISTREAM_ERROR_MSG_);
        return p_is;
    }

    p_energy = pw_energy(l_value,
                       l_unit);
    return p_is;

}


// ---------------------------------------------------------------------------
void pw_set_energy_resolution(double p_value,
                              pw_energy_unit p_unit)
{
    // Sanity checks
    if(sc_core::sc_is_running())
    {
        SC_REPORT_ERROR(PW_ENERGY_RESOLUTION_ELABORATION_TYPE_,
                        PW_ENERGY_RESOLUTION_ELABORATION_MSG_);
        return;
    }
    if(pw_energy::a_is_resolution_fixed)
    {
        SC_REPORT_ERROR(PW_ENERGY_RESOLUTION_FIXED_TYPE_,
                        PW_ENERGY_RESOLUTION_FIXED_MSG_);
        return;
    }

    if(p_value == 0)
    {
        SC_REPORT_ERROR(PW_ENERGY_RESOLUTION_ZERO_TYPE_,
                        PW_ENERGY_RESOLUTION_ZERO_MSG_);
        return;
    }

    if(p_value < 0)
    {
        SC_REPORT_ERROR(PW_ENERGY_RESOLUTION_NEGATIVE_TYPE_,
                        PW_ENERGY_RESOLUTION_NEGATIVE_MSG_);
        return;
    }

    if(p_unit < PW_fJ or p_unit > PW_JOULE)
    {
        SC_REPORT_ERROR(PW_ENERGY_UNIT_OVERFLOW_TYPE_,
                        PW_ENERGY_UNIT_OVERFLOW_MSG_);
        return;
    }

    if(not is_pow10(p_value))
    {
        SC_REPORT_ERROR(PW_ENERGY_RESOLUTION_POW10_TYPE_,
                        PW_ENERGY_RESOLUTION_POW10_MSG_);
        return;
    }

    double l_resolution_value = pw_rint(p_value * g_tab_energy_scales[p_unit]);
    if(l_resolution_value < 1.0
       or l_resolution_value > (static_cast<double>(PW_UINT64_MAX)))
    {
        SC_REPORT_ERROR(PW_ENERGY_RESOLUTION_OVERFLOW_TYPE_,
                        PW_ENERGY_RESOLUTION_OVERFLOW_MSG_);
        return;
    }

    pw_energy::a_resolution_value = l_resolution_value;
    pw_energy::a_is_resolution_fixed = true;

    pw_info << "Resolution Fixed to: " << pw_energy::a_resolution_value
            << " Femto Joule" << pw_endl;
}

  // ---------------------------------------------------------------------------

  // This simply rounds downwards.
  pw_energy pw_energy::round3sf(void) const
  {
    if (a_value == 0) return *this;
    sc_dt::uint64 v0 = a_value;
    sc_dt::uint64 v1 = a_value;
    sc_dt::uint64 v2 = a_value;
    int count = 0;
    while (true)
      {
	sc_dt::uint64 n = v0/10U;
	if (n == 0) break;
	count ++;
	v2 = v1; v1 = v0; v0 = n;
      }
    pw_energy result(*this);
    result.a_value  = v2;
    for (int i = 0; i<count-2; i++) result.a_value *= 10;
    //     std::cout << "Rounded " << *this << " to " << result << "\n";
    return result;
  }


// ---------------------------------------------------------------------------
pw_energy pw_get_energy_resolution(void)
{
    if(not pw_energy::a_is_resolution_fixed)
        pw_info << "Resolution Fixed to: " << pw_energy::a_resolution_value
                << " Femto Joule" << pw_endl;

    pw_energy l_energy;
    l_energy.a_value = 1;
    pw_energy::a_is_resolution_fixed = true;
    return l_energy;
}


} // namespace sc_pwr


