// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_vcd_observer.h,v 1.2 2011/07/25 15:32:39 my294 Exp $

/** @file pw_vcd_observer.h
 * @brief Record the power variation for generating a VCD trace.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 *
 * This class is used by the VCD trace file for generating a VCD trace of the 
 * power variation of a SystemC component.
 */

#ifndef _PW_VCD_OBSERVER_
#define _PW_VCD_OBSERVER_

#include <string>
#include <systemc>
#include "pw_subject.h"
#include "pw_module_base.h"
#include "pw_kernel_ids.h"
#include "pw_accounting_base.h"
#include "pw_vcd_trace.h"

// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
/**
 * @brief Record the power variation of a SystemC component.
 */
  class pw_vcd_observer: public pw_accounting_base
  {
    
  public: // CONSTRUCTOR AND DESTRUCTOR


    /**
     * @brief Constructor of a VCD observer.
     * @param[in] p_obj SystemC object monitored by this observer.
     * @param[in] p_name SystemC attribute name.
     */
    pw_vcd_observer(sc_core::sc_object& p_obj, const std::string& p_name);


    public: // SUBJECT MANAGEMENT FUNCTIONS
        /**
         * @brief Method called by a subject monitored by this observer.
         * @param[in] p_subject Subject which called this observer.
         *
         * The subject monitored by this observer use this method for 
         * indicating a change of its state.
         */
        void update(pw_subject& p_subject);

        /**
         * @brief Add a trace file where to record the power variation.
         * @param[in,out] p_file_pt Output trace file.
         * @param[in] p_str Name identifier of the signal in the trace file.
         * @param[in] p_str Name identifier of the signal in the trace file.
	 * @param[in] p_plot_control how to include this object in the (gnu-)plot style output file (see VCD forms).
         */
        void trace(sc_pwr::pw_vcd_trace_file* p_file_pt,
                   const std::string& p_str,
		   plot_control_t p_plot_control = DoNotPlot
		   );

    public: // MISCELLANOUS FUNCTIONS
        //! Return an informative string about this observer.
        const std::string to_string(void);

    private: // PRIVATE ATTRIBUTES

	class vcd_monitor
	{
	  pw_energy last_e;
	  pw_vcd_trace_file::traced_variable *trc;
	  vcd_monitor() // constructor
	    {
	      trc = 0;
	      last_e = PW_ZERO_ENERGY;
	    }

	protected:
	  friend class pw_vcd_observer;
	  void new_energy(pw_energy new_e);
	};
	vcd_monitor a_monitors[PW_ACCT_MAX_NO_ACCOUNTS];

        sc_time a_utilization;


        //! % utilization during current log interval
        double a_utilization_value;



    private: // POWER UPDATE FUNCTIONS


	void update_energy_infos(const pw_module_base& p_module);


        //! Update the log values.
        void update_log_values(const pw_module_base& p_module);



    private: // DISABLED FUNCTIONS
        //! DISABLE default constuctor.
        pw_vcd_observer(void);
        //! DISABLE copy constructor.
        pw_vcd_observer(const pw_vcd_observer&);
        //! DISABLE assignment operator.
        pw_vcd_observer& operator=(const pw_vcd_observer&);
}; // pw_vcd_observer


// ---------------------------------------------------------------------------
// Output stream operator of energy object.
std::ostream& operator<<(std::ostream&, pw_vcd_observer&);

} // namespace sc_pwr


#endif // _PW_VCD_OBSERVER_
