#ifndef PW_TLM_PAYLOAD_H
#define PW_TLM_PAYLOAD_H

// TLM_POWER3 Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $

// pw_tlm_payload.h - bus distance energy accounting.
// This part of TLM_POWER3 is included if the user has configured TLM_PATH to the TLM2.0 library.

/** @file pw_tlm_payload.h
 * @brief Defines an alternate generic payload needed for transition hamming distances.
 * @author David Greaves
 *
 */


//
// We may wish to use a modified generic payload provided by this file.
// This enables distance and bit-level energy consumption to be modelled.
// If not desired, please set PW_TLM_PAYLOAD to 0.
//
#include "systemc.h"


#ifndef PW_TLM_PAYLOAD

// No mode: Do nothing if not using generic payloads
// This file can still be included, but does not define anything.
// This is so this file can be compiled even when TLM2.0 is not being used for transactions.
//
#define PW_TLM3(X)

#else
// The rest of this file is now enabled:


#if PW_TLM_PAYLOAD < 1
#define PW_TLM3(X)
#else
#define PW_TLM3(X) X
#endif

#if PW_TLM_PAYLOAD==0

// Mode 0: Use default TLM types  (unmodified) in this mode.
// This whole header file becomes a nop.


typedef tlm::tlm_base_protocol_types PW_TLM_TYPES;
typedef tlm::tlm_generic_payload PW_TLM_PAYTYPE;
#else


#include "pw_length.h"
#include "pw_energy.h"
#include "pw_module_base.h"
#include "pw_confidence_switcher.h"

#define PW_TGP_ADDRESS   1
#define PW_TGP_DATA      2
#define PW_TGP_LANES     4
#define PW_TGP_NOFIELDS  8
#define PW_TGP_MUXD_AD   16
#define PW_TGP_FIELD_STICKY_MASK  0xFF

#define PW_TGP_ONCHIP    256
#define PW_TGP_OFFCHIP   512
#define PW_TGP_AUXSTYLE  1024

#define PW_TGP_STYLE_STICKY_MASK  0xF00

#define PW_TGP_ACCT_DEST   (1U << 12)
#define PW_TGP_ACCT_SRC    (2U << 12)
#define PW_TGP_ACCT_CKP    (4U << 12) // Checkpoint. Non-sticky.

#define PW_TGP_ACCT_STICKY_MASK  0x3000

#define PW_TGP_NONSTICKY_MASK (PW_TGP_ACCT_CKP)
#define PW_TGP_DEFAULT_FLAGS (PW_TGP_ONCHIP | PW_TGP_ACCT_SRC)

#include "tlm_generic_payload/tlm_generic_payload.h"


#define MAX_TRACKED_DATA_WIDTH_BYTES 128 // This is not the maximum burst size, which can be larger.


namespace sc_pwr
{

  typedef unsigned int xitions_t;

  class tlm_bit_transition_tracker // A model of a physical bus.
  {
    sc_dt::uint64 m_old_address;
    unsigned char m_old_databus[MAX_TRACKED_DATA_WIDTH_BYTES];
#define PW_XITION_SWITCHER_ADDR 0
#define PW_XITION_SWITCHER_DATA 1
#define PW_XITION_SWITCHERS 2
    confidence_switcher<xitions_t> *xition_switchers[PW_XITION_SWITCHERS];

  public:
    sc_module *src; // Bus src

    // Constructor
    tlm_bit_transition_tracker(sc_module *src);
    int data_transitions(unsigned char *data, int words, int buswidth);
    int address_transitions(sc_dt::uint64 new_address);
      
  };

  class pw_agent_record_mm; // Foward declaration.



  // Our payload is an extension of the standard TLM 2.0 payload.
  class pw_tlm_generic_payload: public tlm::tlm_generic_payload
    {
      sc_module *m_whence;
      unsigned int m_flags;
      xitions_t m_xitions;
      sc_pwr::pw_energy m_wiring_energy_accumulator;
      sc_pwr::pw_energy m_dmi_energy_accumulator;
      sc_pwr::pw_length m_distance_accumulator;
      
      sc_pwr::tlm_bit_transition_tracker *m_src_transtracker;
      class pw_agent_record *previous;
  
  
      void local_reset() // Reset payload before each new transaction
      {
	previous = 0;
	m_xitions = 0;
	m_flags = 0;
	//    llsc_ptr = 0;
	m_whence = 0;
	// We have wiring energy a non DMI transaction  which is generally logged at points along the way, depending on the users flags.
	// The DMI energy needs keeping separate: it is logged at the source in lieu of a real transaction and consists of both the wiring energy AND the bus and target component transaction energy.
	m_wiring_energy_accumulator = sc_pwr::PW_ZERO_ENERGY;
	m_dmi_energy_accumulator = sc_pwr::PW_ZERO_ENERGY;
	m_distance_accumulator = sc_pwr::PW_ZERO_LENGTH;
	m_src_transtracker = 0;
      }

  void pw_energy_checkpoint(sc_module *where); // Write out energy in use to logging entity.
  void compute_xitions(tlm_bit_transition_tracker *new_bus);
 protected:
  friend class pw_tlm_paytype_mm_t;
  friend class pw_agent_record;
  pw_tlm_generic_payload *mm_next;  
 public:
  pw_energy get_dmi_energy_record();

  void pw_set_origin(sc_module *where, unsigned int flags=0, sc_pwr::tlm_bit_transition_tracker *transition_reference=0);
  void pw_terminus(sc_module *where);
  pw_agent_record pw_log_hop(sc_module *where, unsigned int flags=0, sc_pwr::tlm_bit_transition_tracker *transition_reference=0);

public:
  // Constructors 1/2
  pw_tlm_generic_payload() : 
  tlm_generic_payload(),
    mm_next(0)
  {
    local_reset();
  };

  // Constructors 2/2
  explicit pw_tlm_generic_payload(tlm::tlm_mm_interface* mm) : 
  tlm_generic_payload(mm),
    mm_next(0)
  {
    local_reset();
  };


};

  
  class pw_agent_record
  {
    pw_tlm_generic_payload *payload;
    sc_module *module; // One per module ?
    tlm_bit_transition_tracker *trk;
    pw_agent_record *next_, *previous_; // doubly-linked list along transaction path
    //agent_record_mm *mm;
    int account_;

    sc_time utilisation_; // TODO make confidence tricksters and start to use - DMI-2 mode.
    pw_energy energy_;


  public:
    //! Same semantics as the equivalent call in pw_module_base
    void record_energy_use(const pw_energy &e, int account= PW_ACCT_DYNAMIC_DEFAULT, bool noscale=true);


    //! Same semantics as the equivalent call in pw_module_base
    void record_utilisation(const sc_time util, const sc_time *startref=0);

  pw_agent_record(sc_module *where,  pw_tlm_generic_payload * payload, tlm_bit_transition_tracker *trk): // , pw_agent_record *previous):
    payload(payload),
      module(where),
      trk(trk)
      //previous(previous)
      {
	energy_ = PW_ZERO_ENERGY;
	account_ = 0;
	utilisation_ = SC_ZERO_TIME;
	next_ = 0;
      }
      

    
  };



}; // end namspace


// This is outside the namespace so that application code can be compiled with alternative implementations.
typedef sc_pwr::pw_tlm_generic_payload PW_TLM_PAYTYPE;

struct tlm_pw_base_protocol_types
{
  typedef PW_TLM_PAYTYPE tlm_payload_type;
  typedef tlm::tlm_phase tlm_phase_type;
};

typedef tlm_pw_base_protocol_types PW_TLM_TYPES;

namespace sc_pwr
{
 

  class pw_tlm_paytype_mm_t: public tlm::tlm_mm_interface
    {
    private:
      PW_TLM_PAYTYPE *freelist;
      
    public:
      PW_TLM_PAYTYPE* allocate();
  
  
      void free(tlm::tlm_generic_payload* disposeme)
      {
	free(dynamic_cast<PW_TLM_PAYTYPE *>(disposeme));
      }
      
      void free(PW_TLM_PAYTYPE *disposeme);
    };
  
  
}; // end namespace

#endif
#endif
#endif
// eof
