// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_module_base.h,v 1.2 2011/07/25 15:32:39 my294 Exp $

/** @file pw_module_base.h
 * @brief Base class used by the SystemC module for recording their power 
 * consumption.
 * @author Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 *
 * This class has to be used as a base class of the SystemC components for 
 * recording their power consumption.
 */

#ifndef _PW_MODULE_BASE_
#define _PW_MODULE_BASE_

#include <string>
#include <iostream>
#include <systemc>
#include <unordered_map>

#include "pw_subject.h"
#include "pw_power.h"
#include "pw_energy.h"
#include "pw_voltage.h"
#include "pw_length.h"
#include "pw_area.h"

#define PW_ACCT_MAX_NO_ACCOUNTS 8
#define PW_ACCT_STATIC_DEFAULT 0
#define PW_ACCT_DYNAMIC_DEFAULT 1
#define PW_ACCT_WIRING_DEFAULT 2


// ===========================================================================
using namespace sc_core;

namespace sc_pwr
{

// ---------------------------------------------------------------------------
/**
 * @brief Base class for the SystemC object begin monitored - called a 'component' in the manual.
 */
class pw_module_base:
    public pw_subject
{
    public: // ENUM
        enum attribute_id
        {
            INIT          = 0,
            PHASEMODE_POWER  = 1,
            TRANSACTION   = 2,
            QUANTUM_UP    = 3,
            LOG_EVENT     = 4,
            NONE          = 5 
        };

    public: // CONSTRUCTOR AND DESTRUCTOR
        //! Default Construtor
        pw_module_base(void);
	~pw_module_base();
	void destructor_prologue();
	static void flush_all_modules(sc_object *pos, std::fstream *osp);

    public: // Callbacks
       //! Called when Vcc is changed and so on.
       virtual void recompute_pvt_parameters() {};

    public: // OBSERVERS MANAGEMENT FUNCTIONS
        /**
         * @brief Attach an observer to this subject.
         * @param[in] p_observer Observer to attach.
         */
       virtual void attach(pw_observer_if& p_observer, const char *p_aid);

    public: // POWER RECORDING FUNCTIONS

        //! Do housekeeping for one quantum / log interval
        void quantum_up(void);
        void log_event(void);

        //! Add transactional energy (and utilisation) cost to dynamic energy consumption.
	//! We need a utilisation starting time for LT mode (it does not start at sc_time_stamp() but at some point in the future.
        void record_energy_use(const pw_energy&, int account= PW_ACCT_DYNAMIC_DEFAULT, bool noscale=true); 
	void record_utilisation(const sc_time util, const sc_time *startref=0);
	//! Set the new dynamic power (for POWER2-style phase/mode approaches).
	void set_dynamic_power(const pw_power& p_power)
	{
	  set_static_power(p_power, PW_ACCT_DYNAMIC_DEFAULT);
	}


        //! Set the static power consumption.
        void set_static_power(const pw_power&, int l_acct = PW_ACCT_STATIC_DEFAULT);
        //! Set the static power consumption.
        void set_static_power(double, pw_power_unit, int l_acct = PW_ACCT_STATIC_DEFAULT);

        //! Set scaled static power consumption.
        void set_scaled_static_power(const pw_power&);
        //! Set the static power consumption.
        void set_scaled_static_power(double, pw_power_unit);


	//! Set the mode of power consumption.
        void set_power_mode(const std::string&);
        //! Set the phase of power consumption.
        void set_power_phase(const std::string&);



        //! Set working voltage of the module. The update_children should be false by default for ease of use in constructors that have not built their children yet. The child will later ask its parent as a default behaviour anyway.
        void set_vcc(const pw_voltage&, bool update_children=false);   
        void set_vcc(double, pw_voltage_unit, bool update_children=false); 

        //! Set area of the module beyond that of its children.
	void set_excess_area(pw_length x, pw_length y, float max_aspect_ratio=2.5, const char *chipname=0);
	void set_excess_area(pw_area a, float max_aspect_ratio=2.5, const char *chipname=0);

        //! Set fixed dimensions of the module
	void set_fixed_dimensions(pw_length x, pw_length y, const char *chipname=0);

        //! Set fixed area of the module
	void set_fixed_area(pw_area a, const char *chipname=0);

        //! Set chip or region name and also, by default, that of all children not yet set.
	void set_chip_region(const char *name, bool update_children=true);
	void set_chip_region(std::string &name, bool update_children=true);

        void set_modified_attribute(attribute_id);
        //! Return the modified attribute.
        attribute_id get_modified_attribute(void) const;
        //! Return account no of current transaction
        const int get_transaction_acct(void) const;
        //! Return energy consumed by current transaction
        const pw_energy& get_transaction_energy(void) const;
        //! Return time consumed by current transaction
        const sc_time& get_transaction_delay(void) const;
        //! Return area
	const pw_area& get_area(void); //const;
        /**
         * @brief Return the voltage nominal VCC associated to a module.
         * @return voltage value.
         * @note A report of severity SC_ERROR is thrown when their is no 
         * nominal Vcc voltage value associated to the parameter power mode and phase. 
         */
        const pw_voltage& get_vcc( void );

        //! Return the current standing power consumption.
        const pw_power& get_phasemode_power(int acct) const
	{
	  return a_paccts[acct].m_phasemode_power;
	}


        //! Return the current power mode of consumption.
        const std::string& get_power_mode(void) const;
        //! Return the current power phase of consumption.
        const std::string& get_power_phase(void) const;

        //! Return the chip/region name (can set it as a side effect if not been set yet.)
        const std::string& get_chip_region(void);

        //! Return the SystemC object linked to this module.
        sc_core::sc_object& get_sc_object(void);

        //! Return the SystemC module linked to this module.
	sc_core::sc_module& get_sc_module(void);


    protected:
	friend class pw_tlm_generic_payload;
	class pw_parent_cache_entry
	{
	public:
	  pw_parent_cache_entry(sc_module *peer,
				sc_module *lowest_common_parent,
				pw_length distance_to_peer):
	  peer(peer), lowest_common_parent(lowest_common_parent), distance_to_peer(distance_to_peer)
	  { };
					  
	  sc_module *peer;
	  sc_module *lowest_common_parent;
	  pw_length distance_to_peer;
	};

    public: // MISCELLANOUS FUNCTIONS
        //! Return an informative string about this power module.
        virtual const std::string to_string(void) const;

        //! Lowest Common Parent : slow scan
        pw_parent_cache_entry *lowest_common_parent(sc_module *other);

	//! Return name - if one can be found, else "noname".
	sc_core::sc_module_name get_name(void);

    private: // PRIVATE ATTRIBUTES
	// These are horribly non-rentrant: needs locks if using multithreaded SystemC!

        //! Which account the notified informatin should be credited to.
        int a_transaction_acct;
        //! Define the last modified attribute.
        attribute_id a_last_modified_attribute;





        //! Power accounts.
	// Normal operation :Power is kept here in the module bases and energy is kept in the observers.
	// The Observers can also track power but this is not used via the module_base API (or at all ?).
	class pacct
	{
	public: 

	  void end_of_simulation(sc_time when);

	  pw_module_base *m_parent;

	  //! Steady-state (phase-based) power consumption being charged to this account.
	  pw_power m_phasemode_power;
	  //! Time of the last energy bill 
	  sc_time m_last_baseline_update;

	  sc_time get_last_update() const { assert(inuse()); return m_last_baseline_update; }

	  // Set the new standing power and return energy consumed since last change.
	  pw_energy set_local_phasemode_power(const pw_power& p_power, sc_time when);

	  void start(pw_module_base *parent) { m_parent = parent; };
	  pacct() // constructor
	    {
	      m_last_baseline_update = SC_ZERO_TIME;
	      m_phasemode_power = PW_ZERO_POWER;
	      m_parent = 0; // Not yet in use
	    }

	  bool inuse() const { return m_parent != 0; }
	  void destructor_prologue(sc_time);
	};

	pacct a_paccts[PW_ACCT_MAX_NO_ACCOUNTS];


        //! The area of this module beyond that of its children.
	pw_area m_excess_area;
        //! The area of this module including its children.
	pw_area m_area;
        //! Whether the area has been set
	bool m_area_set;
        //! Whether the placement co-ordinates have been set
	bool m_placement_set;

        //! The chip name that this component is on, or the named physical region of a chip or PCB etc..
	std::string m_chip_region;



    private:
	std::unordered_map<sc_module *, pw_parent_cache_entry *> m_lowest_common_parent;
        //! Lowest Common Parent : slow scan
        sc_object *lowest_common_parent_scan(sc_module *other);

    protected:
        //! Dynamic energy consumed by current transaction.
        pw_energy a_transaction_energy;
        //! Delay incurred owing to current transaction
        sc_time a_transaction_delay;      
        //! Current Vcc of module
        pw_voltage a_vcc;
	//! Whether vcc has yet been set.
	bool a_vcc_set;
        //pw_power a_scaled_static_power.
        sc_time a_utilization;

        //! Power mode value.
        std::string a_power_mode;
        //! Power phase value.
        std::string a_power_phase;


 protected:
	void physical_report(std::fstream *osp, sc_module &module);

 private:
	sc_object *m_sc_object;
	void scan_all_module_maps(sc_object *pos);


    private: // DISABLED FUNCTIONS
        //! DISABLE copy constructor.
        pw_module_base(const pw_module_base&);
        //! DISABLE assignment operator.
        pw_module_base& operator=(const pw_module_base&);
}; // class pw_module_base

// ---------------------------------------------------------------------------
/**
 * @brief Return associated power module
 */
pw_module_base *get_pw_module_base(sc_object &it);
	

// ---------------------------------------------------------------------------
// Output stream operator of power module object.
std::ostream& operator<<(std::ostream&,
                         const pw_module_base::attribute_id);

  // ---------------------------------------------------------------------------
  // Output stream operator of power module object.
  std::ostream& operator<<(std::ostream&,
			   const pw_module_base&);

  // ---------------------------------------------------------------------------
  // Mean distance given sensible place-and-route.
  pw_length rentian_distance(sc_module *parent); // first model is rentian...

} // namespace sc_pwr

#endif // _PW_MODULE_BASE_
