// TLM_POWER3: Energy-based for loosely-timed TLM.
// (C) 2011 DJ Greaves & MM Yasin, University of Cambridge Computer Laboratory.
// $Id: $
/*****************************************************************************
 *                       Copyright (c) 2010, CEA-LETI
 * 
 * TLM POWER2 is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU Lesser General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version.
 *
 * TLM POWER2 has been developped in the framework of the MINALOGIC OpenTLM 
 * project.  For more information see http://www.opentlm.org
 *
 * For further information, questions or feedback on the delivery, please 
 * contact <pascal.vivet@cea.fr>
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public 
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License 
 * along with this library; if not, write to the Free Software Foundation, 
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 ****************************************************************************/

// $Id: pw_length.h,v 1.1 2011/08/22 13:39:59 my294 Exp $

/** @file pw_length.h
 * @brief Define length data type.
 * @author adapted from pw_length.h Cedric Koch-Hofer <cedric.koch-hofer@cea.fr>
 */

#ifndef _PW_LENGTH_
#define _PW_LENGTH_

#include <iostream>
#include <string>

#include <systemc>
#include "pw_length.h"


// ===========================================================================
namespace sc_pwr
{


// ---------------------------------------------------------------------------
//! Standard physical length unit.
enum pw_length_unit
{
    //! Femto Meter = 10e-15
    PW_fm   = 0,
    //! Pico Meter  = 10e-12
    PW_pm   = 1,
    //! Nano Meter  = 10e-9
    PW_nm   = 2,
    //! Micro Meter = 10e-6
    PW_um   = 3,
    //! Milli Meter = 10e-3
    PW_mm   = 4,
    //! Meter = 10e0
    PW_METER = 5,
};


// ---------------------------------------------------------------------------
// INCREMENT and DECREMENT OPERATORS

//! Prefix INCREMENT of length unit.
pw_length_unit& operator++(pw_length_unit&);
//! Postfix INCREMENT of length unit.
pw_length_unit operator++(pw_length_unit&,
                          int);
//! Prefix DECREMENT of length unit.
pw_length_unit& operator--(pw_length_unit&);
//! Postfix DECREMENT of length unit.
pw_length_unit operator--(pw_length_unit&,
                          int);


// ---------------------------------------------------------------------------
//! Output stream operator of length unit object.
std::ostream& operator<<(std::ostream&,
                         const pw_length_unit&);


// ---------------------------------------------------------------------------
//! Intput stream operator of length unit object.
std::istream& operator>>(std::istream&,
                         pw_length_unit&);


// ---------------------------------------------------------------------------
/**
 * @brief Represent a length value using the standard unit (METER).
 * @note Arithmetic operations generate a report of severity PW_WARNING or 
 * PW_ERROR (division by zero) when an overflow is detected.  In this case it 
 * is initialized to 0 or to the max length value (according to the performed 
 * arithmetic operation).
 */
class pw_length
{
    public: // CONSTRUCTORS
        /** @brief Default constructor.
         * @note
         * - Initialized to a value of PW_ZERO_METER.
         */
        pw_length(void);

        /**
         * Constructor of a length value.
         * @param[in] p_value Raw length value.
         * @param[in] p_unit Length unit multiple of the raw value.
         * @pre The parameter length value should be equal to 0, or >= to the 
         * length resoultion and <= to the max length value, otherwise a report 
         * of severity PW_WARNING or PW_ERROR is thrown.  In this last case, 
         * the constructed length value will be equal to 0 or to the maximal 
         * length value (according to the parameters value).
         */
        pw_length(double p_value,
                  pw_length_unit p_unit);

        //! Copy constructor.
        pw_length(const pw_length&);

        //! Assignment operator.
        pw_length& operator=(const pw_length&);

    public:  // ARITHMETIC ASSIGNMENT OPERATORS
        //! Assignment by SUM.
        pw_length& operator+=(const pw_length&);
        //! Assignment by DIFFERENCE.
        pw_length& operator-=(const pw_length&);
        //! Assignment by MULTIPLICATION.
        pw_length& operator*=(double);
        //! Assignment by DIVISION.
        pw_length& operator/=(double);

        //! Prefix INCREMENT.
        pw_length& operator++(void);
        //! Postfix INCREMENT.
        pw_length operator++(int);
        //! Prefix DECREMENT.
        pw_length& operator--(void);
        //! Postfix DECREMENT
        pw_length operator--(int);

    private: // RELATIONAL OPERATORS
        //! Relational EQUAL operator.
        friend bool operator==(const pw_length&,
                               const pw_length&);
        //! Relational NOT EQUAL operator.
        friend bool operator!=(const pw_length&,
                               const pw_length&);
        //! Relational LESS THAN operator.
        friend bool operator<(const pw_length&,
                              const pw_length&);
        //! Relational GREATER THAN operator.
        friend bool operator>(const pw_length&,
                              const pw_length&);
        //! Relational LESS THAN OR EQUAL operator.
        friend bool operator<=(const pw_length&,
                               const pw_length&);
        //! Relational GREATER THAN OR EQUAL operator.
        friend bool operator>=(const pw_length&,
                               const pw_length&);

    public: // MISCELLANOUS METHODS
        //! Return a string representing its pysical value.
        const std::string to_string(void) const;

        //! Return it raw value (multiple of the length resolution).
        sc_dt::uint64 value(void) const;

        //! Return its raw value (multiple of the length resolution).
        double to_double(void) const;

        //! Return its value in metres
        double to_meters(void) const;

        //! Return the maximal length value.
        static pw_length max(void);


	//! Round to two three sig figures
	pw_length round3sf() const;

    private: // MSICELLANOUS OPERATORS AND FUNCTIONS
        //! Set the length resolution
        friend void pw_set_length_resolution(double,
                                             const pw_length_unit);

        //! Return the current length resolution.
        friend pw_length pw_get_length_resolution(void);

    private: // PRIVATE MEMBERS

        //! Current length value.
        sc_dt::uint64 a_value;

 public: // needed outside class in pw_length.cpp (djg tmp hack)
        //! Length resolution in femto meter.
        static double a_resolution_value;
        //! True if a pw_length different of 0 have been constructed.
        static bool a_is_resolution_fixed;
}; // class pw_length



// ---------------------------------------------------------------------------
//! Relational EQUAL operator.
bool operator==(const pw_length&,
                const pw_length&);

// ---------------------------------------------------------------------------
//! Relational NOT EQUAL operator.
bool operator!=(const pw_length&,
                const pw_length&);

// ---------------------------------------------------------------------------
//! Relational LESS THAN operator.
bool operator<(const pw_length&,
               const pw_length&);

// ---------------------------------------------------------------------------
//! Relational GREATER THAN operator.
bool operator>(const pw_length&,
               const pw_length&);

// ---------------------------------------------------------------------------
//! Relational LESS THAN OR EQUAL operator.
bool operator<=(const pw_length&,
                const pw_length&);

// ---------------------------------------------------------------------------
//! Relational GREATER THAN OR EQUAL operator.
bool operator>=(const pw_length&,
                const pw_length&);


// ---------------------------------------------------------------------------
//! ADDITION operator.
pw_length operator+(const pw_length&,
                    const pw_length&);

// ---------------------------------------------------------------------------
//! SUBSTRACTION operator.
pw_length operator-(const pw_length&,
                    const pw_length&);

// ---------------------------------------------------------------------------
//! MULTIPLICATION with constant operator.
pw_length operator*(double,
                    const pw_length&);

// ---------------------------------------------------------------------------
//! MULTIPLICATION with constant operator.
pw_length operator*(const pw_length&,
                    double);

// ---------------------------------------------------------------------------
//! DIVISION operator.
double operator/(const pw_length&,
                 const pw_length&);

// ---------------------------------------------------------------------------
//! DIVISION with constant operator.
pw_length operator/(const pw_length&,
                    double);


// ---------------------------------------------------------------------------
// Output stream operator of length object.
std::ostream& operator<<(std::ostream&,
                         const pw_length&);

// ---------------------------------------------------------------------------
// Intput stream operator of length object.
std::istream& operator>>(std::istream&,
                         pw_length&);


// ---------------------------------------------------------------------------
/**
 * @brief Set the length resoultion.
 * @param[in] p_value Raw value of the length resolution.
 * @param[in] p_unit Length unit of the length resolution.
 * @pre This function has to be called during the elaboration phase, otherwise 
 * a report of severity PW_ERROR is thrown.
 * @pre This function can only be called one time, otherwise a report of 
 * severity PW_ERROR is thrown.
 * @pre This function can only be called before any pw_length different from 0 
 * are constructed, otherwise a report of severity PW_ERROR is thrown.
 * @pre The p_value value should be a length of 10, otherwise a report of 
 * severity PW_WARNING is thrown.
 * @note The default length resolution is equal to 1 pico meter. 
 */
void pw_set_length_resolution(double p_value,
                              pw_length_unit p_unit);


// ---------------------------------------------------------------------------
//! Return the current length resolution.
pw_length pw_get_length_resolution(void);


// ---------------------------------------------------------------------------
//! Constant for representing a length of 0 
extern const pw_length PW_ZERO_LENGTH;


} // namespace sc_pwr


#endif // _PW_LENGTH_
