(*
This is essentially the (typed) L1 lagnuage from Semantics of 
Programming Languages (SPL). The small differences are 

  --- unary operations are added to the language (Neg and Not)
  --- "*" (Mult) and "-" (Subtr) are added as binary operations 
  --- a print expression is added 
  --- binary operations are represented as  
      "Op of oper * expr * expr" rather than Op of expr * oper * expr
  --- optional type annotions have been added to the expr AST  (in this 
      way type_checking becomes expr annotation)

*) 
datatype type_expr =
         TEint
       | TEunit
       | TEbool

type loc = string

datatype oper = Plus | Mult | Subt | GTEQ

datatype unary_oper = Neg | Not  

datatype expr = 
         Skip
       | Integer of int
       | Boolean of bool
       | UnaryOp of unary_oper * expr
       | Op of oper * expr * expr
       | Assign of loc * (type_expr option) * expr
       | Deref of loc
       | Seq of expr * expr
       | If of expr * expr * expr
       | While of expr * expr
       | Print of (type_expr option) * expr


fun type_expr_to_string TEint = "int" 
  | type_expr_to_string TEunit = "unit" 
  | type_expr_to_string TEbool = "bool" 

fun unary_to_string Neg = "-"
  | unary_to_string Not = "~"

fun op_to_string Plus = "+"
  | op_to_string Mult = "*"
  | op_to_string Subt = "-"
  | op_to_string GTEQ = ">="


(* pretty printing using Mosml's PP module *) 
                         
fun ppe pps (Integer n) = PP.add_string pps (Int.toString n)
  | ppe pps (Boolean b) = PP.add_string pps (Bool.toString b)
  | ppe pps (UnaryOp (uopr,e)) = 
      (
       PP.add_string pps "("; 
       PP.add_string pps (unary_to_string uopr); 
       ppe pps e;  
       PP.add_string pps ")"
      )
  | ppe pps (Op (opr,e1,e2)) = 
      (
       PP.add_string pps "("; 
       ppe pps e1; 
       PP.add_string pps (" " ^ (op_to_string opr) ^ " "); 
       ppe pps e2;  
       PP.add_string pps ")"
      )
  | ppe pps (If (e1, e2, e3)) = 
      (PP.begin_block pps PP.CONSISTENT 0; 
      PP.add_string pps "if "; 
      ppe pps e1; 
      PP.add_break pps (0, 0); 
      PP.add_string pps " then "; 
      ppe pps e2; 
      PP.add_break pps (0, 0); 
      PP.add_string pps " else "; 
      ppe pps e3;
      PP.end_block pps)
  | ppe pps (Deref l) = PP.add_string pps ("!" ^ l)
  | ppe pps (Assign (l,NONE, e)) =  
       (
        PP.add_string pps l; 
        PP.add_string pps " := "; 
	ppe pps e
       )
  | ppe pps (Assign (l,SOME t, e)) =  
       (
        PP.add_string pps l; 
        PP.add_string pps " : " ; 
        PP.add_string pps (type_expr_to_string t); 
        PP.add_string pps " := "; 
	ppe pps e
       )
  | ppe pps (Skip) = PP.add_string pps "skip"	
  | ppe pps (Seq (e1,e2)) =  
       (PP.begin_block pps PP.CONSISTENT 0; 
        PP.add_string pps "("; 		 
        ppe pps e1; 
        PP.add_string pps "; "; 
        PP.add_break pps (0, 0); 	
	ppe pps e2;
        PP.add_string pps ")"; 		 
        PP.end_block pps)
  | ppe pps (While (e1,e2)) = 
      (PP.begin_block pps PP.CONSISTENT 0; 
       PP.add_string pps "while "; 
       ppe pps e1; 
       PP.add_string pps " do "; 
       PP.add_break pps (0, 0); 
       PP.begin_block pps PP.CONSISTENT 3;		
       PP.add_string pps "("; 
       ppe pps e2; 
       PP.add_string pps ")"; 
       PP.end_block pps;
       PP.end_block pps) 
  | ppe pps (Print (NONE, e)) = 
      (
       PP.add_string pps "print("; 
       ppe pps e; 
       PP.add_string pps ")" 
      ) 
  | ppe pps (Print (SOME t, e)) = 
      (
       PP.add_string pps "print("; 
       PP.add_string pps (type_expr_to_string t); 
       PP.add_string pps ", "; 
       ppe pps e; 
       PP.add_string pps ")" 
      ) 

fun pp_expr expr = 
    let val ppstrm = PP.mk_ppstream {
                consumer  = fn s => TextIO.output(TextIO.stdOut, s), 
                linewidth = 80,               
                flush     = fn () => TextIO.flushOut TextIO.stdOut}
    in 
        PP.begin_block ppstrm PP.CONSISTENT 0; 
        ppe ppstrm expr ;
        PP.end_block ppstrm; 
        PP.flush_ppstream ppstrm 
    end 
                   

(* this is used for debugging and for attaching "origin comments " 
   to bits of intermediate representations (such as assembly code).
   The string contains no line breaks. 
*) 
fun expr_to_string (Integer n) = Int.toString n
  | expr_to_string (Boolean b) = Bool.toString b
  | expr_to_string (UnaryOp (uopr,e))     = "(" ^ (unary_to_string uopr) ^ " " ^  (expr_to_string e) ^ ")"
  | expr_to_string (Op (opr,e1,e2))       = "(" ^ (expr_to_string e1) ^ " " ^ (op_to_string opr) ^ " " ^  (expr_to_string e2) ^ ")"
  | expr_to_string (If (e1, e2, e3))      = "(if " ^ (expr_to_string e1) ^ " then " ^ (expr_to_string e2) ^ " else " ^  (expr_to_string e3) ^ ")"
  | expr_to_string (Deref l)              = "!" ^ l
  | expr_to_string (Assign (l,NONE, e))   = "(" ^ l ^ ": " ^ " : = " ^  (expr_to_string e) ^")"
  | expr_to_string (Assign (l,SOME t, e)) = "(" ^ l ^ ": " ^ (type_expr_to_string t) ^ " : = " ^  (expr_to_string e) ^")"
  | expr_to_string (Skip)                 = "skip"	
  | expr_to_string (Seq (e1,e2))          = "(" ^ (expr_to_string e1) ^ "; " ^  (expr_to_string e2) ^ ")"  
  | expr_to_string (While (e1,e2))        = "(while " ^ (expr_to_string e1) ^ " do " ^ (expr_to_string e2) ^ ")"
  | expr_to_string (Print (NONE, e))      = "print(" ^ (expr_to_string e) ^") "
  | expr_to_string (Print (SOME t, e))    = "print(" ^ (type_expr_to_string t) ^ ", " ^  (expr_to_string e) ^ ")"
                   

