// SoCDAM Course Nominal Processor:   (C) DJ Greaves, 2009.  

// $Id: $

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef uint32
#define uint32 unsigned int
#endif

// Basic Processor Size Parameters 
#define ABUS 32
#define DBUS 32
#define REGS 16


// Dedicated Program Counter Values

#define RESET_PC 0
#define IRQ_PC   4  

// Friendly names for registers

#define PC 15  // program counter
#define LR 14  // link register
#define SP 13  // stack pointer 
#define FB 12  // fram base (P) pointer

#define RZERO 0 // Hardwired to zero
#define R1 1
#define R2 2
#define R3 3


// Machine Instructions
#define LOAD  1 //
#define STORE 2 //
#define ALU   3 //
#define JMP   4 //
#define BZ    5 //  
#define HALT  6 //
#define JL    7 //
#define BNZ   8 //  
#define IRET  9 // Interrupt return.


// ALU operations
#define MOV   ALU + (0 << 4)   
#define ADD   ALU + (1 << 4)   
#define SUB   ALU + (2 << 4)   
#define AND   ALU + (3 << 4)   
#define OR    ALU + (4 << 4)   
#define NOT   ALU + (5 << 4)   
#define XOR   ALU + (6 << 4)   
#define NEG   ALU + (7 << 4)   
#define RSB   ALU + (8 << 4)   



// Assembler directives
#define ORG   10
#define EQU   11
#define DEFW  12
#define DEFB  13
#define END   14

#define L(X) ((long long int)(#X))  

#

typedef struct assembly_t {  const char *label;  long long int mnemonic, a0, a1; const char *comment; } ASM;



class assembler
{

public:
  uint32 hwm;
  uint32 *machinecode;
  uint32 maxlen;
  uint32 point;
  ASM *program;

  typedef struct stable_entry_s
  {
    const char *label; long long int value;
    struct stable_entry_s *next;
  } STAB;
	
  STAB *stable;

  void emit(int pass, int mc)
  {
    if (pass==2 && point < maxlen) 
      {
	machinecode[point/4] = mc;  
	printf("%02X %08X", point, mc);
      }
    if (pass==2 && point >= maxlen) printf("Too much code\n");
    point += 4;
    if (point > hwm) hwm = point;
  }
	   
  // Constructor
  assembler(uint32 *machinecode_, uint32 maxlen_) : machinecode(machinecode_), maxlen(maxlen_)
  {
    hwm = 0;
    stable = 0;
    point = 0; 
  }


  int lookup(char *l, int pass)
  {
    if (!l) { printf("label missing from instruction\n"); return 0; }
    STAB *q;
    for (q=stable; q; q=q->next) { if (!(strcmp(q->label, l))) return q->value; }
    if (pass == 2) printf("Label not found %s\n", l);
    return 0;
  }

 void makepass(int pass)
  {
#define K(X)  if (pass==2) printf("  %18s // %s\n", X, program[p].comment ? program[p].comment: "")

    for (int p=0; program[p].mnemonic != END; p++)
      {
	STAB *q = 0;
	if (pass == 2) printf("%12s ", program[p].label ? program[p].label:"");

	if (program[p].label && strlen(program[p].label))
	  {
	    for (q=stable; q; q=q->next) { if (!(strcmp(q->label, program[p].label))) break; }
	    if (pass == 1)
	      {
		if (q) printf("label dmto %s\n", q->label);
		else 
		  {
		    q = new (STAB);
		    q->label = program[p].label;
		    q->value = point;
		    q->next = stable;
		    stable = q;
		  }
	      }

	  }

	switch(program[p].mnemonic & 15)
	  {
	  case END: 
	    K("END");
	    break;
	    

	  case BZ: case BNZ:
	    {
	      int reg = program[p].a0;
	      char *a = (char*)(program[p].a1);
	      int offset = ((lookup(a, pass) - point)>>2) & 0x3FFFF;
	      emit(pass, (program[p].mnemonic << 24) | (reg << 20) | offset );
	      K("BZ/BNZ");
	    break;  
	    }	  

	  case JMP: case JL:
	    {
	      char *a = (char*)(program[p].a0);
	      int dest = lookup(a, pass);
	      emit(pass, (program[p].mnemonic << 24) | dest );
	      K("JMP/JL");
	      break;  
	    }	  

	  case ALU:
	    {
	      int reg = program[p].a0;
	      int reg1 = program[p].a1;
	      emit(pass, (program[p].mnemonic << 24) | (reg << 20) | (reg1 << 16) );
	      K("ALU");
	      break;  
	    }	  

	  case HALT:
	    emit(pass, (program[p].mnemonic << 24));
	    K("HALT");
	    break;

	  case IRET:
	    emit(pass, (program[p].mnemonic << 24));
	    K("IRET");
	    break;


	  case LOAD:
	  case STORE:
	    {
	      int reg = program[p].a0;
	      char *a = (char*)(program[p].a1);
	      emit(pass, (program[p].mnemonic << 24) | (reg << 20) | lookup(a, pass));
	      K("STORE/LOAD");
	      break;  
	    }	  


	  case ORG:
	    point = program[p].a0;
	    K("                          ORG");
	    break;

	  case DEFW:
	    emit(pass, program[p].a0);
	    K("DEFW");
	    break;

	  case EQU:
	    if (q) q->value = program[p].a0;
	    else printf("No label in DEFW\n"); 
	    K("                         EQU");
	    break;

	  default: printf("Unimplemented MNEMONIC %llu\n", program[p].mnemonic);
	}
      }
  }

  void assemble(const char *id, ASM *program_)
  {
    program = program_;
    makepass(1);
    makepass(2);
    printf("Assembly complete %s\n", id);

  }

};


class nominalproc_iss
{
public:
  int regfile[REGS], ilink;
  bool runhaltb, interrupted;


  nominalproc_iss()
  {

  }


  virtual int codefetch(int a) = 0;
  virtual int datafetch(int a) = 0;
  virtual void datawrite(int a, int d)  = 0;



#define J(X, Y) printf(" %s %08X\n", X, Y)
  void step()
  {
    printf("Fetch %08x\n", regfile[PC]);
    int ins = codefetch(regfile[PC]);
    regfile[PC] += 4;
    switch ((ins >> 24) & 0xF) 
      {
      case JMP:
	{
	  int d = ins & 0x3FFFF;
	  regfile[PC] = d; 
	  J("JMP", d);
	  break;
	}

      case LOAD:
	{
	  int rd = (ins>>20) & (REGS-1);
	  int a = ins & 0x3FFFF;
	  int data = datafetch(a);
	  regfile[rd] = data; 
	  J("LOAD", data);
	  break;
	}

      case STORE:
	{
	  int rd = (ins>>20) & (REGS-1);
	  int a = ins & 0x3FFFF;
	  int data = regfile[rd];
	  datawrite(a, data);
	  J("STORE", data);
	  break;
	}

      case HALT:
	runhaltb = 0;
	J("HALT", 0);
	break;

      case IRET:
	interrupted = false;
	regfile[PC] = ilink;
	J("IRET", 0);
	break;

      case BNZ:
	{
	  int rd = regfile[(ins>>20) & (REGS-1)];
	  int offset = ((ins & 0x3FFFF) << 14) >>14;
	  int dest = regfile[PC] + (offset << 2) - 4;
	  if (rd)
	    {
	      regfile[PC] = dest;
	      J("BNZ taken", dest);
	    }
	  else J("BNZ not taken", 0);

	  break;
	}

      case ALU:
	{
	  int rd = (ins>>20) & (REGS-1);
	  int lhs = regfile[rd];
	  int rhs = regfile[(ins>>16) & (REGS-1)];
	  int data;
	  switch ((ins >> 24) & 0xFF)
	  {

	  case ADD:
	    data = lhs+rhs;
	    break;
	  case SUB:
	    data = lhs-rhs;
	    break;
	  case XOR:
	    data = lhs^rhs;
	    break;
	  case AND:
	    data = lhs&rhs;
	    break;
	  case OR:
	    data = lhs|rhs;
	    break;

	  case MOV:
	    data = lhs;
	    break;

	  default: printf("illegal alu function\n");
	  }
	  regfile[rd] = data; 
	  J("ALU", data);
	  break;
	}

      default: printf("iss: pc=0x%08x 0x%08x Illegal instruction\n", regfile[PC]-4, ins);
	runhaltb = 0;
	return;
      }
  }

  void interrupt()
  {
    if (interrupted) return;
    ilink = regfile[PC];
    regfile[PC] = IRQ_PC;
    runhaltb = 1;
    interrupted = true;
  }

  void reset()
  {
    regfile[PC] = RESET_PC;
    runhaltb = true;
    interrupted = false;
  }

};

