// $Id: $
// djip: DMA controller.
//

// This is a single channel DMA controller
// 
// It only does block copies: a more advanced device would service
// a linked list of buffers in main RAM.


//
// There are three classes in here: a behavioural model, an RTL-like component and a TLM component.
//

#ifndef DMA_CONTROLLER_H
#define DMA_CONTROLLER_H

#define TRC(X) X

class dma_controller_bev_model
{
 public:
  sc_out <bool> *irq_net_ptr;
  
  // Programmers' model
  uint32 src, dest, length; // Main operand registers
  bool busy, int_enable;

  // Status read
  uint32 status() { return (busy << 31) | (int_enable << 30); }    

  // Target side programming model:
  //   0 Src Register
  //   4 Dest Register
  //   8 Length Register
  //  12 Status/control register. 

  uint32 target_read(int a) 
  { 
    return (a==0)? src: (a==1) ? dest: (a==2) ? (length) : status();
  }


#define M(X) ((X) & ~3) 
  void target_write(int a, uint32 d) 
  { 
    if (a==0) src=M(d); else if(a==1) dest=M(d); else if(a==2) length = M(d); 
     else if (a==3) { busy = d >> 31; int_enable = d >> 30; }
    irq_update();
  }

  // Call this function explicitly after changing any of the
  // combinational support for the interrupt condition.
  void irq_update()
  {
    // Combinational output update function: called manually.
    if (irq_net_ptr) *irq_net_ptr = int_enable && !busy;
  }


  virtual int initiator_busread(int a) = 0;
  virtual void initiator_buswrite(int a, int d) = 0;
  void onestep()
  {
    if (length == 0)
      { 
	busy = 0; 
	irq_update(); 
      }
    else
      {
	uint32 d = initiator_busread(src);
	TRC(printf("DMA : Copy word 0x%x from 0x%x to 0x%x\n", d, src, dest));
	initiator_buswrite(dest, d);

	src += 4;
	dest += 4;
	length -= 4;
      }
  }


  void reset()
  {
    busy = false;
    int_enable = false; 
    irq_update();
  }

  // Constructor
  dma_controller_bev_model()
    {
      irq_net_ptr = 0;
    }

};


//------------------------------------------------------------------
// Netlevel architecture for the DMA controller.
// Do not use the SC_MODULE macro to define this class since we wish to inherit the behavioural model

class DMA_CONTROLLER_net_level: public sc_module, public dma_controller_bev_model
{
 public:
  sc_in <bool> rst, clk;

  // Slave/target net-level port:
  sc_in <uint32> addr, wdata;
  sc_out <uint32> rdata;
  sc_in <bool> hren, hwen;
  sc_out <bool> opack;

  // Interrupt
  sc_out<bool> irq;

  // Master/initiator net-level port:
  sc_out <uint32> m_addr, m_wdata;
  sc_in <uint32> m_rdata;
  sc_out <bool> m_hren, m_hwen;
  sc_in <bool> m_opack;



  void target_behaviour()
  {
    if (hren.read() && hwen.read()) 
      rdata = target_read(addr.read()/4);
    else
      if (hren.read() && !hwen.read()) 
	target_write(addr.read()/4, wdata.read()/4);
  }



  int initiator_busread(int a)  // Net-level bus read initiator
  {
    TRC(printf("%s fetch code from A=0x%08X", name(), a));
    m_addr = a; m_hwen = 0; m_hren = 1;
    do { wait(clk.posedge_event()); } while (!m_opack.read());
    int r = m_rdata.read();
    m_hren = 0;
    return r;
  }


  void initiator_buswrite(int a, int d)   // Net-level bus write initiator
  {
    TRC(printf("%s Datawrite A=0x%08X  D=0x%08x\n", name(), a, d));
    m_addr = a; m_hwen = 1; m_hren = 0; m_wdata = d;
    do { wait(clk.posedge_event()); } while (!m_opack.read());
    m_hwen = 0;
  }

  // This function does the actual copy.
  void net_level_initiator()
  {
    while (true)
      {
	wait(clk.posedge_event());
	onestep();

      }

  }


  void hw_reset() { reset(); } 

  SC_CTOR(DMA_CONTROLLER_net_level)
  {
    irq_net_ptr = &irq;
    hw_reset();
    SC_METHOD(target_behaviour); sensitive << clk.pos(); 
    SC_THREAD(net_level_initiator); // sensitive << clk.pos(); 
    SC_METHOD(hw_reset); sensitive << rst.pos(); 
  }

};


//------------------------------------------------------------------
// TLM coomponent containing the DMA controller behavioural model.
// Do not use the SC_MODULE macro to define this class since we wish to inherit the behavioural model

class tlm_dma_controller: public dma_controller_bev_model, public memport_if, public sc_module
{
 public:

  sc_in <bool> rst, clk; // A few residual net-level connections (might not be used).
  sc_out <bool> irq;

  // Target, programmed I/O operations
  void b_transact(MEMPAYLOAD *p, sc_time &delay)
  {
    delay += sc_time(2, SC_NS);
    if (p->rwbar)
      {
	p->data = target_read(p->addr/4);
	printf("%s read data [0x%x] := 0x%x\n", name(), p->addr, p->data);
      }
    else
      {
	printf("%s write data [0x%x] := 0x%x\n", name(), p->addr, p->data);
	target_write(p->addr/4, p->data);
      }
  }



  // Initiator behaviour
  sc_port <memport_if> initiator_port;
  sc_time initiator_delay; 
  int initiator_busread(int a)  // TLM bus read initiator
  {
    MEMPAYLOAD p;
    p.addr = a; p.rwbar = 0; 
    TRC(printf("%s fetch code from A=0x%08X", name(), a));
    initiator_port->b_transact(&p, initiator_delay);
    return p.data;
  }


  void initiator_buswrite(int a, int d)   // TLM bus write initiator
  {
    MEMPAYLOAD p;
    p.addr = a; p.rwbar = 1; p.data = d;
    TRC(printf("%s Datawrite A=0x%08X  D=0x%08x\n", name(), a, d));
    initiator_port->b_transact(&p, initiator_delay);
  }


  void tlm_initiator()
  {
    // In a grander system, granularity is global and user settable, held in the quantum keeper class...
    sc_time quantum_granularity = sc_time(1000, SC_NS);

    initiator_delay = SC_ZERO_TIME;
    while (true)
      {
        initiator_delay += sc_time(2500, SC_PS); // Account for my clock period.
	onestep();
	if (initiator_delay > quantum_granularity) // Does quantum keeper need_sync() ? 
	  {
	    wait(initiator_delay + sc_time_stamp()); // Then perform sync()
	    initiator_delay = SC_ZERO_TIME;
	  }
      }

  }

  void hw_reset() { reset(); } 

  SC_CTOR(tlm_dma_controller)
  {
    irq_net_ptr = &irq;
    hw_reset();
    SC_THREAD(tlm_initiator); 
    SC_METHOD(hw_reset); sensitive << rst.pos(); 
  }


};


#endif
// eof
