/**
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 *	Copyright 2009 Andrew Rice (acr31@cam.ac.uk) and Vytautas Vaitukaitis (vv236@cam.ac.uk)
 */

package uk.ac.cam.acr31.android.geomessaging.servertools;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;

import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;

import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.mime.MultipartEntity;
import org.apache.http.entity.mime.content.FileBody;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.impl.client.DefaultHttpClient;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

/**
 * A helper class of static methods for interacting with the GeoMessagingServer
 * 
 * @author acr31
 * 
 */
public class ServerHelper {

	/**
	 * URL to use for uploading new messages
	 */
	private static final String URL_UPLOAD = "http://svr-acr31-acsmobile.cl.cam.ac.uk:9004/GeoMessagingServer/post";

	/**
	 * URL to use to query for the list of nearby messages
	 */
	private static final String URL_MESSAGE_LIST = "http://svr-acr31-acsmobile.cl.cam.ac.uk:9004/GeoMessagingServer/query";

	/**
	 * The URL to prefix onto image names to download them from the web
	 */
	private static final String URL_PICTURE_FOLDER = "http://svr-acr31-acsmobile.cl.cam.ac.uk:9004/GeoMessagingServer/img";

	/**
	 * Make a POST request to the URL using the form given and return an
	 * InputStream for the response body
	 */
	private static InputStream obtainInputStream(String urlAddress,
			MultipartEntity form) throws ClientProtocolException, IOException {
		DefaultHttpClient client = new DefaultHttpClient();
		HttpUriRequest request = new HttpPost(urlAddress);
		client.getParams().setBooleanParameter("http.protocol.expect-continue",
				false);

		// set the form as an entity of the request
		((HttpEntityEnclosingRequestBase) request).setEntity(form);

		// execute the request
		HttpResponse response = client.execute(request);

		// get the response input stream
		return response.getEntity().getContent();
	}

	/**
	 * Make a POST request to the URL using the form given and return a String
	 * for the response body
	 */
	private static String getResponseString(String urlAddress,
			MultipartEntity form) throws ClientProtocolException, IOException {
		InputStream in = obtainInputStream(urlAddress, form);
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		byte[] buf = new byte[512];
		int read;
		while ((read = in.read(buf)) != -1) {
			os.write(buf, 0, read);

		}
		in.close();
		return new String(os.toByteArray());
	}

	/**
	 * Return a list of all the messages which have been posted at the given
	 * latitude and longitude within the chosen proximity
	 */
	public static ArrayList<Message> getMessages(double lat, double lng,
			double proxMiles) {
		try {
			MultipartEntity requestForm = new MultipartEntity();
			requestForm
					.addPart("latitude", new StringBody(String.valueOf(lat)));
			requestForm.addPart("longitude",
					new StringBody(String.valueOf(lng)));
			requestForm.addPart("distance", new StringBody(String
					.valueOf(proxMiles)));

			/* Get a SAXParser from the SAXPArserFactory. */
			SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
			SAXParser saxParser = saxParserFactory.newSAXParser();

			/* Get the XMLReader of the SAXParser we created. */
			XMLReader xmlReader = saxParser.getXMLReader();
			/* Create a new ContentHandler and apply it to the XML-Reader */

			MyXMLHandler xmlHandler = new MyXMLHandler();

			xmlReader.setContentHandler(xmlHandler);

			xmlReader.parse(new InputSource(obtainInputStream(URL_MESSAGE_LIST,
					requestForm)));

			return xmlHandler.getMessages();
		} catch (Exception e) {
			e.printStackTrace();
			return new ArrayList<Message>();
		}
	}

	/**
	 * Submit this message to the server. 
	 * @param message is the text of the message to submit
	 * @param pictureFileName is the filename of a file on the phone's storage containing the image to include
	 * @param lat is the latitude of the current location
	 * @param lng is the longitude of the current location
	 */
	public static boolean sendMessage(String message, String pictureFilename,
			double lat, double lng) {
		try {
			MultipartEntity requestForm = new MultipartEntity();
			requestForm
					.addPart("latitude", new StringBody(String.valueOf(lat)));
			requestForm.addPart("longitude",
					new StringBody(String.valueOf(lng)));
			requestForm.addPart("message", new StringBody(message));
			if (pictureFilename != null)
				requestForm.addPart("picture", new FileBody(new File(
						pictureFilename)));

			String response = getResponseString(URL_UPLOAD, requestForm);

			if (response.equals("OK")) {

				return true;
			} else {

				return false;
			}
		} catch (Exception e) {
			e.printStackTrace();
			return false;
		}
	}

	/**
	 * Helper class to parse the XML response from the server
	 */
	private static class MyXMLHandler extends
			org.xml.sax.helpers.DefaultHandler {

		private ArrayList<Message> mMessages = new ArrayList<Message>();

		@Override
		public void startElement(String uri, String localName, String qName,
				Attributes attributes) throws SAXException {
			if (localName.equals("message")) {
				Message message = new Message();
				message.setMessage(attributes.getValue("message"));
				message.setPictureUri(URL_PICTURE_FOLDER + "/"
						+ attributes.getValue("picture"));
				mMessages.add(message);
			}
		}

		protected ArrayList<Message> getMessages() {
			return mMessages;
		}

	}

}
