#!/usr/bin/perl -w

#############################################################
# packet_capture gac1
#
# $Id: packet_generator.pl 4239 2008-07-03 08:05:55Z grg $
#
# Load packets from Pcap files into the Packet Generator and
# start the packet Generator
#
# Revisions:
#
##############################################################

use strict;
use warnings;
use POSIX;

use NF2::PacketGen;
use NF2::PacketLib;
use threads;                # pull in threading routines
use threads::shared;        # and variable sharing routines
use NF2::TestLib;
use NF2::RegAccess;
use Net::Pcap;
use Getopt::Long;
use IO::Handle;

require "reg_defines.ph";
require 'reg_defines_packet_generator.ph';


# Global 
my $default_queue_size;	
my $xmit_queue_size;
my $rcv_queue_size;

my $max_iter = PKT_GEN_MAX_ITER();

my $queue_addr_offset = OQ_OFFSET();

my $queue_max = 0x80000 / 8;
my @queue_current_offset = (0, 0, 0, 0);
my @queue_words = (0, 0, 0, 0);
my @queue_bytes = (0, 0, 0, 0);
my @queue_pkts = (0, 0, 0, 0);
my @queue_base_offset = (0, 0, 0, 0);
my @sec_current = (0, 0, 0, 0);
my @usec_current = (0, 0, 0, 0);
my $capture_enable = 0;
my @pcap_c = (0, 0, 0, 0);
my @num_pkts = (0, 0, 0, 0);
my @threads;

my @caplen_warned = (0, 0, 0, 0);

my $help = '';
my $num_queues = 4;

my @pcap_filename = ('', '', '', '');
my @capture_filename = ('', '', '', '');
my @final_capture_filename;
my @capture_interfaces;
my @rate = (-1, -1, -1, -1);
my @clks_between_tokens = (-1, -1, -1, -1);
my @number_tokens = (-1, -1, -1, -1);
my @last_len = (0, 0, 0, 0);
my @last_nsec = (0, 0, 0, 0);
my @last_sec = (0, 0, 0, 0);
my @extra_delay = (0, 0, 0, 0);
my @iterations = (1, 1, 1, 1);
my @delay = (-1, -1, -1, -1);
my $err;
my $xmit_done = 0;

unless ( GetOptions ( 'q0=s' => \$pcap_filename[0],
                      'q1=s' => \$pcap_filename[1],
                      'q2=s' => \$pcap_filename[2],
                      'q3=s' => \$pcap_filename[3],
                      'r0=s' => \$rate[0],
                      'r1=s' => \$rate[1],
                      'r2=s' => \$rate[2],
                      'r3=s' => \$rate[3],
                      'i0=s' => \$iterations[0],
                      'i1=s' => \$iterations[1],
                      'i2=s' => \$iterations[2],
                      'i3=s' => \$iterations[3],
                      'c0=s' => \$capture_filename[0],
                      'c1=s' => \$capture_filename[1],
                      'c2=s' => \$capture_filename[2],
                      'c3=s' => \$capture_filename[3],
                      'd0=i' => \$delay[0],
                      'd1=i' => \$delay[1],
                      'd2=i' => \$delay[2],
                      'd3=i' => \$delay[3],
                      'help' => \$help,
        )
  and ($help eq '')
       ) { usage(); exit 1 }

# Catch interupts (SIGINT)
$SIG{INT} = \&INT_Handler;

# Need to specify at least one pcap file to load into SRAM
if (!($pcap_filename[0] ne '' || $pcap_filename[1] ne '' || 
      $pcap_filename[2] ne '' || $pcap_filename[3] ne '')) {
	usage(); exit 1;
}

# Verify that the number of iterations is correct
for (my $i = 0; $i < $num_queues; $i++) {
	if ($pcap_filename[$i] ne '' && 
	    ($iterations[$i] < 1 || $iterations[$i] > $max_iter)) {
		    print "Error: Iteration count ($iterations[$i]) specified for queue $i is invalid. Must be between 1 and $max_iter.\n\n";
		    exit 1;
	    }
}

# determine if capture is enabled
if (($capture_filename[0] ne '' || $capture_filename[1] ne '' || 
      $capture_filename[2] ne '' || $capture_filename[3] ne '')) {
	$capture_enable = 1;
}

if ($capture_enable) {
	@capture_interfaces = determine_capture_interfaces(\@capture_filename);

	if ($#capture_interfaces >= 0) {
		print "Starting packet capture on: " . join(' ', @capture_interfaces) . "\n";

		my @interfaces = ("nf2c0", "nf2c1", "nf2c2", "nf2c3");
		nftest_init(\@ARGV,\@interfaces,);
		nftest_pkt_cap_start(\@capture_interfaces);
	}
}

# Disable the output queues by writing 0x0 to the enable register
packet_generator_enable (0x0);

# Reorganize the queues
queue_reorganize( 0x80000 / 8, 4096);

# Load the pcap files into the SRAM
for (my $i = 0; $i < $num_queues; $i++) {
	# Set the rate limiter
	($clks_between_tokens[$i], $number_tokens[$i]) = rate_limiter_set($i * 2, $rate[$i]);

	if ($pcap_filename[$i] ne '') {
		load_pcap($pcap_filename[$i], $i, $delay[$i]);
	}
}

# Set the rate limiter for CPU queues
for (my $i = 0; $i < 4; $i++) {
	rate_limiter_set($i * 2 + 1, 200000); 
}

# Set the number of iterations for the queues with pcap files
for (my $i = 0; $i < $num_queues; $i++) {
	if ($pcap_filename[$i] ne '') {
		set_length_queue ($i);	
		set_number_iterations ($iterations[$i], 1, $i);
	}
	# Enable the rate limiter
	if ($rate[$i] > 0) {
		rate_limiter_enable($i * 2);
	}
	else {
		rate_limiter_disable($i * 2);
	}
}

# Enable the rate limiter on the CPU queues
for (my $i = 0; $i < 4; $i++) {
	rate_limiter_enable($i * 2 + 1);
}

# Enable the packet generator hardware to send the packets
packet_generator_enable (0xF);

# Wait until the correct number of packets is sent
print "Sending packets...\n";
my $start = wait_for_last_packet();
$xmit_done = 1;

# Keep the capture running
if ($capture_enable) {
	wait_for_ctrl_c($start);
}

# Finish up
finish_gen();

exit(0);

###############################################################
# Name: usage
#
# Prints the usage information
#
###############################################################
sub usage {
  (my $cmd = $0) =~ s/.*\///;
  print <<"HERE1";
NAME
   $cmd - Load packets from Pcap into the PacketGenerator and 
          start sending packets

SYNOPSIS
   $cmd 
        -q<queue number> <pcap file> 
	[-r<queue number> <rate>] (Kbps)
	[-i<queue number> <number of iterations>]
	[-d<queue number> <delay between packets>] (ns)
	[-c<queue number> <capture file>

   $cmd --help  - show detailed help

HERE1

  return unless ($help);
  print <<"HERE";

DESCRIPTION

   This script loads pcap files into the associated queue in the
the packet generator.  The rate limiter is also set based on the input
parameters (optional).  The program then signals the packet generator
hardware to start sending the data.
   
OPTIONS
   -q<queue number> <pcap file>
     Specify the pcap file to load in and send from a queue 

   -r<queue number> <rate>
     Specify the rate for each queue in Kbps

   -i<queue number> <number of iterations>
     Specify the number of iterations per queue

   -d<queue number> <delay between packets>
     Specify the delay between packets in ns.
     If not specified then the delay recorded in the pcap file 
     is used. A value of 0 will disable the delay.

   -c<queue number> <capture file>
     Specify the capture file.

EXAMPLE

   % $cmd -q0 udp.pcap -r1 3

HERE
}

###############################################################
# Name: INT_Handler
#
# captures SigINT when capture is enabled.  Saves the capture 
# files prior to exit
#
# Arguments: 
#
###############################################################

sub INT_Handler {
	my $signame = shift;	

	print "\n\n";

	if (!$xmit_done) {
		print "Warning: Program interrupted during operation. Not all packets have\n";
		print "been sent. Some packets may continue to be sent after exiting.\n";
		print "\n";
	}

	finish_gen();
}


###############################################################
# Name: finish_gen
#
# Perform the necessary actions when finishing the packet generation
#
# Arguments: 
#
###############################################################

sub finish_gen {
	# Disable the packet generator
	#  1. disable the output queues
	#  2. reset the delay module
	#    -- do this multiple times to flush any remaining packets
	#       The syncfifo is 1024 entries deep -- we should need far 
	#       fewer than this to ensure the FIFO is flushed
	#  3. disable the packet generator
	for (my $i = 0; $i < $num_queues; $i++) {
		disable_queue($i + 8);
	}
	sleep(1);
	for (my $i = 0; $i < 1024; $i++) {
		reset_delay();
	}
	sleep(1);
	packet_generator_enable(0x0);
	reset_delay();

	if ($capture_enable) {
		save_pcap();
	}

	display_xmit_metrics();
	display_capture_metrics();

	if ($capture_enable) {
		print "Ignore warnings about scalars leaked...\n";
	}
	exit (0);
}


###############################################################
# Name: save_pcap
#
# Saves the pcap files prior to exiting
#
# Arguments: 
#
###############################################################

sub save_pcap {
	my %hdr;
	my $pcap_t;
	my $err;
	
	my $usec;
	my $sec;

	if ($capture_enable == 1) {

		my %packets = nftest_pkt_cap_finish();
		for (my $i = 0; $i < scalar(@capture_interfaces); $i++) {	
			print "Writing $final_capture_filename[$i] (" . scalar(@{$packets{$capture_interfaces[$i]}}) . " packets)\n";
			$pcap_t = Net::Pcap::open_live($capture_interfaces[$i], 2000, 0, 0, \$err)
		    or die "Can't open dev '$capture_interfaces[$i]': $err\n";

			# Open Pcap output file
	 		my $dumper = Net::Pcap::dump_open($pcap_t, $final_capture_filename[$i]);	
			my @array = @{ $packets{$capture_interfaces[$i]} };

			for ( my $i = 0; $i < scalar(@array); $i++) {
				my $pkt = $array[$i];
				my @unpacked_pkt = unpack("C*", $pkt);
				$pkt = substr($pkt, 8);
				my $high = ($unpacked_pkt[0] << 24) | ($unpacked_pkt[1] << 16) |
				           ($unpacked_pkt[2] << 8) | ($unpacked_pkt[3] << 0);
				my $low = ($unpacked_pkt[4] << 24) | ($unpacked_pkt[5] << 16) |
				          ($unpacked_pkt[6] <<  8) | ($unpacked_pkt[7] <<  0);	
  		 
				my $sec = `echo "($low+($high*2^32))/10^9" | bc`;
				my $usec = `echo "(($low+($high*2^32))%10^9)/(10^3)" | bc`;
				chomp($sec);
				chomp($usec);
				$hdr{"len"} = scalar(@unpacked_pkt);
				$hdr{"caplen"} = scalar(@unpacked_pkt);
				$hdr{"tv_usec"} = $usec;
				$hdr{"tv_sec"} = $sec;
				Net::Pcap::dump($dumper, \%hdr, $pkt);
			}
			Net::Pcap::dump_close($dumper);
		}
	}
}

###############################################################
# Name: queue_reorganize
#
# Reorganizes the queues
#
# Arguments: default_queue_size 
#            xmit_queue_size    
#
###############################################################

sub queue_reorganize {

	$default_queue_size = $_[0];	
	$xmit_queue_size = $_[1];
	$rcv_queue_size = $default_queue_size - $xmit_queue_size;

	$queue_addr_offset = OQ_ADDRESS_LO_REG_1() - OQ_ADDRESS_LO_REG_0();

	# Disable output queues
	for (my $i = 0; $i < 12; $i++) {
	
		nf_regwrite ('nf2c0', 
			(OQ_CONTROL_REG_0() + ($i) * $queue_addr_offset), 
			0x00);
	}

	# Resize the output queues
	for (my $i = 0; $i < 4; $i++) {
		# Set queue sizes for queues 0, 2, 4 and 6
		nf_regwrite ('nf2c0', 
			(OQ_ADDRESS_LO_REG_0() + ($i * 2) * $queue_addr_offset), 
			($default_queue_size * $i));
		nf_regwrite ('nf2c0', 
			(OQ_ADDRESS_HI_REG_0() + ($i * 2) * $queue_addr_offset), 
			($default_queue_size * $i + $xmit_queue_size - 1));
		nf_regwrite ('nf2c0', 
			(OQ_CONTROL_REG_0() + ($i * 2) * $queue_addr_offset), 
			0x02);

		# Set queue sizes for queues 1, 3, 5 and 7
		nf_regwrite ('nf2c0', 
			(OQ_ADDRESS_LO_REG_0() + ($i * 2 + 1) * $queue_addr_offset), 
			($default_queue_size * $i + $xmit_queue_size));
		nf_regwrite ('nf2c0', 
			(OQ_ADDRESS_HI_REG_0() + ($i * 2 + 1) * $queue_addr_offset), 
			($default_queue_size * ($i + 1) - 1));
		nf_regwrite ('nf2c0', 
			(OQ_CONTROL_REG_0() + ($i * 2 + 1) * $queue_addr_offset), 
			0x02);

		# Set queue sizes for queues 8, 9, 10, 11
		nf_regwrite ('nf2c0', 
			(OQ_ADDRESS_LO_REG_0() + ($i + 8) * $queue_addr_offset), 
			($default_queue_size * ($i + 4)));
		nf_regwrite ('nf2c0', 
			(OQ_ADDRESS_HI_REG_0() + ($i + 8) * $queue_addr_offset), 
			($default_queue_size * ($i + 4 + 1)) - 1);
		nf_regwrite ('nf2c0', 
			(OQ_CONTROL_REG_0() + ($i + 8) * $queue_addr_offset), 
			0x02);

		$queue_base_offset[$i] = (4 + $i) * $default_queue_size * 16;
		$queue_current_offset[$i] = SRAM_BASE_ADDR() + $queue_base_offset[$i];
	}

	# Enable Output Queues that are not associated with Packet Generation
	for (my $i = 0; $i < 8; $i++) {
	
		nf_regwrite ('nf2c0', 
			(OQ_CONTROL_REG_0() + ($i) * $queue_addr_offset), 
			0x01);
	}

	return 0;
}


###############################################################
# Name: reset_delay
#
# Reset the delay modules
#
###############################################################
sub reset_delay {
	nf_regwrite ('nf2c0', DELAY_RESET_REG(), 1);
}

###############################################################
# Name: disable_queue
#
# Disable one of the queues
#
# Arguments: queue             queue number (0-11)
#
###############################################################
sub disable_queue {
	my $queue = shift;

	nf_regwrite ('nf2c0', 
		OQ_CONTROL_REG_0() + $queue * $queue_addr_offset, 
		0x0);
}

###############################################################
# Name: set_number_iterations
#
# Sets the number of iterations for a Packet Generator Queue
#
# Arguments: number_iterations number of iterations for queue
#            iterations        enable the number of iterations
#            queue             queue number (0-3)
#
# Control register
#       bit 0 -- enable queue
#       bit 1 -- initialize queue (set to 1) 
#
###############################################################

sub set_number_iterations {

	my $number_iterations = $_[0];
	my $iterations_enable = $_[1];
	my $queue = $_[2];

	nf_regwrite ('nf2c0', 
                   OQ_CONTROL_REG_0() + ($queue + 8) * $queue_addr_offset,  
		   0x1); 
	nf_regwrite ('nf2c0', 
                   PKT_GEN_OQ_MAX_ITER_REG_0() + ($queue + 8) * $queue_addr_offset,  
		   $number_iterations); 

	return 0;
}

###############################################################
# Name: rate_limiter_enable
#
# Enables the rate limiter for a queue
#
# Arguments: queue    queue to enable the rate limiter on
#
###############################################################

sub rate_limiter_enable {

	my $queue = $_[0];

	nf_regwrite ('nf2c0', 
                RATE_LIMIT_CTRL_0_REG() + ($queue * RATE_LIMIT_OFFSET()),
 							  0x1); 

	return 0;
}

###############################################################
# Name: rate_limiter_disable
#
# Disables the rate limiter for a queue
#
# Arguments: queue    queue to disable the rate limiter on
#
###############################################################

sub rate_limiter_disable {

	my $queue = $_[0];

	nf_regwrite ('nf2c0', 
                RATE_LIMIT_CTRL_0_REG() + ($queue * RATE_LIMIT_OFFSET()),
 							  0x0); 

	return 0;
}

###############################################################
# Name: rate_limiter_set
#
# Set the rate limiter value of an output queue
#
# Arguments: queue  queue to enable the rate limiter on
#            rate   the rate to set for the output queue
#
###############################################################

sub rate_limiter_set {
	
	my $queue = $_[0];
	my $rate = $_[1];

	my $clks_between_tokens = 1000000;
	my $number_tokens = 1;
	my $clk_freq = 125*(10 ** 6);
	
	my $epsilon = 0.001;
	my $MAX_TOKENS = 84;

	# Check if we really need to limit this port
	return (1, 1000000) if ($rate < 1);

	$clks_between_tokens = 1;
	$rate = ($rate * 1000) / 8;
	$number_tokens = ($rate * $clks_between_tokens) / $clk_freq;

	# Attempt to get the number of tokens as close as possible to a 
	# whole number without being too large
	my $token_inc = $number_tokens;
	my $min_delta = 1;
	my $min_delta_clk = 1;
	while (($number_tokens < 1 || 
	        $number_tokens - floor($number_tokens) > $epsilon) && 
	       $number_tokens < $MAX_TOKENS) {

		$number_tokens += $token_inc;
		$clks_between_tokens += 1;

		# Verify that number_tokens exceeds 1
		if ($number_tokens > 1) {
			# See if the delta is lower than the best we've seen so far
			my $delta = $number_tokens - floor($number_tokens);
			if ($delta < $min_delta) {
				$min_delta = $delta;
				$min_delta_clk = $clks_between_tokens;
			}
		}
	}

	# Adjust the number of tokens/clks between tokens to get the closest to a whole number of
	# tokens per increment
	if ($number_tokens - floor($number_tokens) > $epsilon) {
		$clks_between_tokens = $min_delta_clk;
		$number_tokens = floor($token_inc * $clks_between_tokens);
	}

	# Calculate what the actual rate will be
	$rate = $number_tokens * $clk_freq / $clks_between_tokens;
	$rate = ($rate * 8) / 1000;

	print "Limiting " . queue_name($queue) . " to " . rate_str($rate) . " (";
	print "tokens = $number_tokens, ";
	print "clks = $clks_between_tokens)\n";
  
	nf_regwrite ('nf2c0',
                RATE_LIMIT_TOKEN_INTERVAL_0_REG() + ($queue * RATE_LIMIT_OFFSET()),
                $clks_between_tokens);
	nf_regwrite ('nf2c0', 
                RATE_LIMIT_TOKEN_INC_0_REG() + ($queue * RATE_LIMIT_OFFSET()),
                $number_tokens);

	return $clks_between_tokens, $number_tokens;
}	

###############################################################
# Name: set_length_queue
#
# Sets the length of the Packet Generator queue 
#
# Arguments: queue           Packet Generator queue (0-3)
#
###############################################################

sub set_length_queue {

	my $queue = $_[0];

	nf_regwrite ('nf2c0',
                   OQ_ADDRESS_HI_REG_0() + ($queue + 8) * $queue_addr_offset, 
                   ($queue_current_offset[$queue] - 0x10 - SRAM_BASE_ADDR()) / 16);

	return 0;
}	

###############################################################
# Name: load_pcap
#
# Loads the Pcap file a packet at a time and stores the packet 
# into SRAM in the appropriate Packet Generator Queue.  It will
# only load the full packets upto the queue size.  If there are
# more packets in the Pcap file than can fit in the queue, they
# are ignored. If there is no delay specified the delay between
# packets in the Pcap file is used.
#
# Arguments: pcap_filename      Filename of the Pcap to load
#            queue              Queue to load the Pcap into
#            delay              Specified delay between packets
#
###############################################################

sub load_pcap {

	my $pcap_filename = $_[0];
	my $queue = $_[1];
	my $delay = $_[2];

	my $err;
	my %hdr;
	my $packet;
	my $load_pkt = 0;
	
	my $pcap_t = Net::Pcap::open_offline($pcap_filename, \$err)
		or die "Can't read '$pcap_filename': $err\n";

	# While there are still packets in the Pcap file
	#   and there is space in the queue read them in

	$packet = Net::Pcap::next($pcap_t, \%hdr);

	$sec_current[$queue] = $hdr{"tv_sec"}; 
	$usec_current[$queue] = $hdr{"tv_usec"};
	
	while ( %hdr && !($load_pkt == 1) ) {

		#print "Packet Number:$num_pkts\n";
		if ($hdr{'len'} != $hdr{'caplen'} && !$caplen_warned[$queue]) {
			print "Warning: The capture length was less than the packet length for one";
			print " or more packets in '$pcap_filename'. Packets will be padded with zeros.\n";
			$caplen_warned[$queue] = 1;
		}

		$load_pkt = load_packet (\%hdr, $packet, $queue, $delay);
		
		undef %hdr;
		$packet = Net::Pcap::next($pcap_t, \%hdr);
		if ($load_pkt == 0) { 
			$num_pkts[$queue]++;
		}
	}

	print "Loaded $num_pkts[$queue] packet(s) into " . queue_name($queue + 8) . "\n";
	
	# Close the pcap file
	Net::Pcap::close($pcap_t);

	return 0;
}

###############################################################
# Name: determine_capture_interfaces
#
# Takes the capture filenames array and determines which 
# interfaces to capture on. Returns an array 
#
# Arguments: capture_filenames  Array of capture filenames from 
#                               from the command line
#
###############################################################

sub determine_capture_interfaces {
	my @capture_filenames = @{$_[0]};
	my @interfaces;

	for (my $i = 0; $i < scalar(@capture_filenames); $i++) {
		if ($capture_filenames[$i] ne '' && $capture_filenames[$i] ne '-') {
			push(@interfaces, "nf2c$i");
			push(@final_capture_filename, $capture_filenames[$i]);
		}
	}

	return @interfaces;
}

###############################################################
# Name: load_packet
#
# Load a packet into an SRAM queue with the specified delay.
#
# Arguments: hdr 								pcap header for the packet
#            packet             the packet form the Pcap file 
#            queue              SRAM queue to insert the packet
#                               (0-3)
#            delay              delay (if specified) 
#		                -1 uses Pcap delay
#
###############################################################
	
sub load_packet {

	my %hdr = %{$_[0]};
	my $packet = $_[1];
	my $queue = $_[2];
	my $delay = $_[3];
	my $src_port = 0;
	my $dst_port = 0x100;
	my $word_len = POSIX::ceil($hdr{"len"} / 8);
	my $sec = $hdr{"tv_sec"};
	my $usec = $hdr{"tv_usec"};
	my $total_words;

	$dst_port = ($dst_port << $queue);

	# If the delay is not specified assign based on the Pcap file
	if ($delay == -1) {
		$delay = $sec - $sec_current[$queue];
		$delay = $delay * 1000000; # convert to usec
		$delay = (($usec + $delay) - $usec_current[$queue]);
		$delay = $delay * 1000; # convert to nsec
	}

	# Check if there is room in the queue for the entire packet
	# 	If there is no room return 1
	$total_words = $word_len + 1 + ($delay > 0);
	if ( ($total_words + $queue_words[$queue]) > $queue_max) {
		return (1);
	}
	else {
		$queue_words[$queue] += $total_words;
		$queue_bytes[$queue] += $hdr{"len"};
		$queue_pkts[$queue]++;
	}
	
	# Load module hdr into SRAM
	nf_regwrite ('nf2c0', 
                   $queue_current_offset[$queue] + 0x4, 
                   0x000000FF);
	nf_regwrite ('nf2c0', 
                   $queue_current_offset[$queue] + 0x8, 
                   $word_len | ($dst_port << 16));
	nf_regwrite ('nf2c0', 
                   $queue_current_offset[$queue] + 0xc, 
                   ($hdr{"len"} | ($src_port << 16)));
	$queue_current_offset[$queue] += 0x10;
  
	$sec_current[$queue] = $sec;
	$usec_current[$queue] = $usec;
	
	# Add on time for the transmit
	my $usec_per_byte = 0.008;
	if ($rate[$queue] > 0) {
		$usec_per_byte *= (10**6) / $rate[$queue];
	}
	$usec_current[$queue] += ($hdr{"len"} + 4) * $usec_per_byte;

	while ($usec_current[$queue] > 10**6) {
		$usec_current[$queue] -= 10**6;
		$sec_current[$queue]++;
	}

	# Load delay into SRAM if it exists
	if ($delay > 0) {
		nf_regwrite ('nf2c0',
                     $queue_current_offset[$queue] + 0x4,
                     0x00000008);
		nf_regwrite ('nf2c0',
                     $queue_current_offset[$queue] + 0x8,
                     floor($delay / 2**32));
		nf_regwrite ('nf2c0',
                     $queue_current_offset[$queue] + 0xc,
                     $delay % 2**32);

		$queue_current_offset[$queue] += 0x10;
	} 

	# Store the packet into SRAM 
	my @pkt = unpack("C*", $packet);
	push @pkt, ((0) x ($hdr{'len'} - $hdr{'caplen'})) if ($hdr{'len'} != $hdr{'caplen'});
	push @pkt, ((0) x (8 - $hdr{"len"} % 8)) if ($hdr{"len"} % 8 != 0);

	for (my $i = 0; $i < scalar(@pkt); $i += 8){
		my $ctrl = 0x0;
		if ($i / 8 == $word_len - 1) {
			$ctrl = 0x100 >> ($hdr{"len"} % 8);
			$ctrl = $ctrl & 0xff | ($ctrl == 0x100);
		}
		my $word1 = ($pkt[$i + 0] << 24) | ($pkt[$i + 1] << 16) | 
                ($pkt[$i + 2] <<  8) | ($pkt[$i + 3] <<  0);
		my $word2 = ($pkt[$i + 4] << 24) | ($pkt[$i + 5] << 16) | 
                ($pkt[$i + 6] <<  8) | ($pkt[$i + 7] <<  0);

		nf_regwrite ('nf2c0', $queue_current_offset[$queue] + 0x4, $ctrl);
		nf_regwrite ('nf2c0', $queue_current_offset[$queue] + 0x8, 
                     $word1);
		nf_regwrite ('nf2c0', $queue_current_offset[$queue] + 0xC, 
                     $word2);
		$queue_current_offset[$queue] += 0x10;
	}

	# Update packet transmit time
	my $delay_delay = $delay;
	my $delay_rate = 0;
	if ($rate[$queue] >= 1) {
		$delay_rate = ceil($last_len[$queue] / $number_tokens[$queue]);
		$delay_rate *= $clks_between_tokens[$queue] * 8;
	}
	my $delay_max = $delay_delay > $delay_rate ? $delay_delay : $delay_rate;
	$delay_max -= $last_len[$queue] + 4 * 8;
	$delay_max = 0 if ($delay_max < 0);
	$delay_max += (($hdr{'len'} > 60 ? $hdr{'len'} : 60) + 4 + 20) * 8;

	$last_nsec[$queue] += $delay_max;
	$last_len[$queue] = $hdr{'len'} + 4;
	$extra_delay[$queue] = 0;
	if ($rate[$queue] >= 1) {
		$extra_delay[$queue] = ceil(($hdr{'len'} + 4) / $number_tokens[$queue]);
		$extra_delay[$queue] *= $clks_between_tokens[$queue];
		$extra_delay[$queue] -= $hdr{'len'} + 4;
		$extra_delay[$queue] *= 8;
	}

	while ($last_nsec[$queue] > 10**9) {
		$last_nsec[$queue] -= 10**9;
		$last_sec[$queue]++;
	}
	
	return 0;
}

###############################################################
# Name: packet_generator_enable
#
# Enable the Packet Generator Hardware
#
# Arguments: enable_queues      0xf enables all queues
#
# Enable register definition:
#	    Bits    Description
#     31:4    Ignored
#     3       Enable packet gen on nf2c3 (1=enable, 0=disable)
#     2       Enable packet gen on nf2c2 (1=enable, 0=disable)
#     1       Enable packet gen on nf2c1 (1=enable, 0=disable)
#     0       Enable packet gen on nf2c0 (1=enable, 0=disable) 
#	
###############################################################

sub packet_generator_enable {

	my $enable_queues = $_[0];

	# Start the queues that are passed into the function
	
	nf_regwrite ('nf2c0', PKT_GEN_CTRL_ENABLE_REG(),  $enable_queues);

	return 0;
}


###############################################################
# Name: queue_name
#
# Convert a queue number to a name
#
# Arguments: queue      Queue number
#	
###############################################################

sub queue_name {
	my $queue = shift;

	if ($queue < 0 || $queue >= 12) {
		return "Invalid queue";
	}
	elsif ($queue < 8) {
		if ($queue % 2 == 0) {
			return "MAC Queue " . ($queue / 2);
		}
		else {
			return "CPU Queue " . (($queue - 1) / 2);
		}

	}
	else {
		return "MAC Queue " . ($queue - 8);
	}
}


###############################################################
# Name: rate_str
#
# Convert a rate to a string. Attempts to choose the most 
# sensible set of units (Gbps, Mbps, Kbps)
#
# Arguments: rate      Data rate
# 
###############################################################

sub rate_str {
	my $rate = shift;

	if ($rate < 1000) {
		return "$rate Kbps";
	}
	elsif ($rate < 1000000) {
		return (sprintf("%1.3f Mbps", $rate / 1000));
	}
	else {
		return (sprintf("%1.3f Gbps", $rate / 1000000));
	}
}


###############################################################
# Name: wait_for_last_packet
#
# Wait until the last packet is scheduled to be sent
#
###############################################################

sub wait_for_last_packet {
	my $last_pkt = 0;
	my $start = time();
	my $delta = 0;

	# Work out when the last packet is to be sent
	for (my $i = 0; $i < scalar(@pcap_filename); $i++) {
		if ($pcap_filename[$i] ne '') {
			my $queue_last = ($last_sec[$i] * 1.0) + ($last_nsec[$i] * 10**-9);
			$queue_last *= ($iterations[$i] * 1.0);
			$queue_last += ($extra_delay[$i] * 10**-9) * ($iterations[$i] - 1.0);
			if ($queue_last > $last_pkt) {
				$last_pkt = $queue_last;
			}
		}
	}


	# Disable output buffering on stdout to enable status updates
	autoflush STDOUT 1;

	# Wait the requesite number of seconds
	printf "Last packet scheduled for transmission at %1.3f seconds\n", $last_pkt;
	while ($delta <= $last_pkt) {
		print "\r$delta seconds elapsed...";
		sleep 1;
		$delta = time() - $start;
	}
	autoflush STDOUT 0;
	print "\n\n";

	return $start;
}


###############################################################
# Name: wait_for_ctrl_c
#
# Wait until the user presses Ctrl-C
#
###############################################################

sub wait_for_ctrl_c {
	my $start = shift;
	my $delta = 0;

	# Disable output buffering on stdout to enable status updates
	autoflush STDOUT 1;

	# Wait the requesite number of seconds
	print "All packets should have been sent.\n";
	print "Press Ctrl-C to stop capture...\n\n";
	while (1) {
		$delta = time() - $start;
		print "\r$delta seconds elapsed...";
		sleep 1;
	}
	autoflush STDOUT 0;
	print "\n\n";
}


###############################################################
# Name: display_capture_metrics
#
# Display the metrics capture by the card
#
###############################################################

sub display_capture_metrics {
	my $offset = PKT_GEN_CTRL_PKT_CNT_1_REG() - PKT_GEN_CTRL_PKT_CNT_0_REG();

	print "Receive statistics:\n";
	print "===================\n\n";

	for (my $i = 0; $i < scalar(@capture_filename); $i++) {
		my $pkt_cnt = nf_regread('nf2c0', PKT_GEN_CTRL_PKT_CNT_0_REG() + $i * $offset);
		my $byte_cnt_hi = nf_regread('nf2c0', PKT_GEN_CTRL_BYTE_CNT_HI_0_REG() + $i * $offset);
		my $byte_cnt_lo = nf_regread('nf2c0', PKT_GEN_CTRL_BYTE_CNT_LO_0_REG() + $i * $offset);
		my $time_first_hi = nf_regread('nf2c0', PKT_GEN_CTRL_TIME_FIRST_HI_0_REG() + $i * $offset);
		my $time_first_lo = nf_regread('nf2c0', PKT_GEN_CTRL_TIME_FIRST_LO_0_REG() + $i * $offset);
		my $time_last_hi = nf_regread('nf2c0', PKT_GEN_CTRL_TIME_LAST_HI_0_REG() + $i * $offset);
		my $time_last_lo = nf_regread('nf2c0', PKT_GEN_CTRL_TIME_LAST_LO_0_REG() + $i * $offset);

		my $byte_cnt = ($byte_cnt_hi) * 2 ** 32 + ($byte_cnt_lo);
		my $delta_hi = $time_last_hi - $time_first_hi;
		my $delta_lo = $time_last_lo - $time_first_lo;

		if ($time_first_lo > $time_last_lo) {
			$delta_hi--;
			$delta_lo += 2**32;
		}

		my $sec = `echo "($delta_lo+($delta_hi*2^32))/10^9" | bc`;
		my $nsec = `echo "(($delta_lo+($delta_hi*2^32))%10^9)" | bc`;

		my $time = $sec + ($nsec / 10**9);
		my $rate = 0;
		if ($time != 0) {
			$rate = $byte_cnt / $time / 1000 * 8;
		}

		printf "%s:\n", queue_name($i + 8);
		printf "\tPackets: %u\n", $pkt_cnt;
		if ($pkt_cnt > 0) {
			printf "\tBytes: %1.0f\n", $byte_cnt;
			printf "\tTime: %1d.%09d s\n", $sec, $nsec;
			printf "\tRate: %s\n", rate_str($rate);
		}
	}
	print "\n\n";
}


###############################################################
# Name: display_xmit_metrics
#
# Display the metrics of sent packets maintained by the card
#
###############################################################

sub display_xmit_metrics {
	print "Transmit statistics:\n";
	print "====================\n\n";

	for (my $i = 0; $i < scalar(@pcap_filename); $i++) {
		if ($pcap_filename[$i] ne '') {
			my $pkt_cnt = nf_regread('nf2c0', OQ_NUM_PKTS_REMOVED_REG_0() + ($i + 8) * $queue_addr_offset);
			my $iter_cnt = nf_regread('nf2c0', PKT_GEN_OQ_CURR_ITER_REG_0() + ($i + 8) * $queue_addr_offset);


			printf "%s:\n", queue_name($i + 8);
			printf "\tPackets: %u\n", $pkt_cnt;
			printf "\tCompleted iterations: %1.0f\n", $iter_cnt;
		}
	}
	print "\n\n";
}
