(* SML Core language library
The following library functions are provided in the SML core language:

(* Overloaded operators *)
    val makestring: 'a -> string (* defined for types int,real,bool *)

(* Integer functions *)
    val min: int * int -> int
    val max: int * int -> int

(* String functions *)
    val size: string -> int      (* returns the length of a string *)

    exception Substring
    val substring: (string * int * int) -> string
        (* returns the substring starting from character position
	   of first int argument (0 is first in string) with length of
	   second int argument - raises Substring if string is too small *)

    val explodeascii: string -> int list
        (* fun explodeascii s = map ord (explode s) *)

    exception ImplodeAscii
    val implodeascii: int list -> string
        (* returns a string corresponding to the argument list of ascii
	   codes.  raises ImplodeAscii if a list element is <0 or >255 *)

(* List functions *)
    exception Hd and Tl
    val hd: 'a list -> 'a
    val tl: 'a list -> 'a list
        (* hd and tl return the head and tail of a list respectively,
	   respectively raising Hd and Tl if applied to empty lists *)

    val null: 'a list -> bool
        (* returns true is arg is the empty list, false otherwise *)

    val length: 'a lsit -> int
        (* returns the length of a list *)

    val map: ('a -> 'b) -> (('a list) -> ('b list)) 

    val fold: (('a * 'b) -> 'b) -> (('a list) -> ('b -> 'b))

    val revfold: (('a * 'b) -> 'b) -> (('a list) -> ('b -> 'b))

    val app: ('a -> 'b) -> (('a list) -> unit) 

    val revapp: ('a -> unit) -> (('a list) -> unit)

    exception Nth
    val nth: (('a list) * int) -> 'a
        (* returns the nth (0th is first element) of a list -
	   raises Nth if int arg is out of range *)

    val exists: (('a -> bool) * ('a list)) -> bool
        (* returns true if the function f(:'a -> bool ) returns true
	   when applied to at least one member of the list *)

(* Reference functions *)
    val inc: (int ref) -> unit
    val dec: (int ref) -> unit
        (* increment / decrement an int ref by 1 *)

(* Function composition *)
    infix 3 o
    val o: (('a -> 'b) * ('c -> 'a)) -> ('c -> 'b)

(* System and Input/Output functions *)
    val stdio: unit -> (instream * outstream)
        (* generates new standard input/output streams *)
 
    val input_line: instream -> string
        (* reads a line from instream and returns as a string *)

    val execute: string -> (instream * outstream)
        (* UNIX only - forks a process (named by the string arg)
	   from the shell and returns input and output streams for
	   that process *)

    val file_exists: string -> bool
        (* may use relative or full pathnames for filename in UNIX *)

    val use: string -> unit
        (* loads and compiles the SML programs in the named file *)

    val system: string -> unit
        (* passes the string to the operating system command interpreter
           for execution *)

    val CpuTime: unit -> int
	(* returns milliseconds used by process *)
    val ExportML: (string * string * (string list)) -> unit
        (* create a new saved state for SML:
	   arg 1 - filename for saved state
	   arg 2 - startup message
	   arg 3 - list of files to be "use"d on startup *)
    
*)
(***************************************************************************)

(********************)
(* lib/Overloadings *)
(********************)
(* Overloaded operators *)

overload ~ : 'a -> 'a;
overload + : 'a * 'a -> 'a;
overload - : 'a * 'a -> 'a;
overload * : 'a * 'a -> 'a;
overload > : 'a * 'a -> bool;
overload < : 'a * 'a -> bool;
overload >= : 'a * 'a -> bool;
overload <= : 'a * 'a -> bool;
overload abs: 'a -> 'a;
overload makestring: 'a -> string;

val Overloadings = 
  ["~","+","-","*","<",">","<=",">=","abs","makestring",
   "=","<>"];

(* Following infixes temporarily here as fixities are not exported by opens *)

infix 3 := o sub;   
(* infix before; RJG what is this? *)
infix 4 > < >= <= ;
infix 5 @ ;
infixr 5 :: ;
infix 6 ^ + - ;
infix 7 & / * div mod ;

(*******************)
(* lib/Integer.str *)
(*******************)

    nonfix ~ 33 1; fun ~(x:int):int = ~ x;
    infix 7 * 36 2; fun (x:int) * (y:int):int = x * y;
    infix 7 div 38 2; fun (x:int) div (y:int):int = x div y;
    infix 7 mod 39 2; fun (x:int) mod (y:int):int = x mod y;
    infix 6 + 34 2; fun (x:int) + (y:int):int = x + y;
    infix 6 - 35 2; fun (x:int) - (y:int):int = x - y;
    infix 4 > 44 2; fun (x:int) > (y:int):bool = x > y;
    infix 4 < 45 2; fun (x:int) < (y:int):bool = x < y;
    infix 4 >= 46 2; fun (x:int) >= (y:int):bool = x >= y;
    infix 4 <= 47 2; fun (x:int) <= (y:int):bool = x <= y;
    fun min(a:int,b:int) = if a<b then a else b;
    fun max(a:int,b:int) = if a>b then a else b;
    fun abs(x:int): int = if x<0 then ~x else x;
    nonfix makestring 52 1; fun makestring(x:int):string = makestring x;

(****************)
(* lib/Real.str *)
(****************)

    nonfix ~ 188 1; fun ~(x:real):real = ~ x;
    infix 6 + 184 2; fun (x:real) + (y:real):real = x + y;
    infix 6 - 185 2; fun (x:real) - (y:real):real = x - y;
    infix 7 * 186 2; fun (x:real) * (y:real):real = x * y;
    infix 7 / 187 2; fun (x:real) / (y:real):real = x / y;
    infix 4 > 192 2; fun (x:real) > (y:real):bool = x > y;
    infix 4 < 194 2; fun (x:real) < (y:real):bool = x < y;
    infix 4 >= 191 2; fun (x:real) >= (y:real):bool = x >= y;
    infix 4 <= 193 2; fun (x:real) <= (y:real):bool = x <= y;
    fun abs(x:real): real = if x<0.0 then ~x else x;
    nonfix real 195 1; fun real(x:int):real = real(x);
    nonfix floor 196 1; fun floor(x:real):int = floor(x);
    nonfix sqrt 220 1; fun sqrt(x:real):real = sqrt(x);
    nonfix sin 221 1; fun sin(x:real):real = sin(x);
    nonfix cos 222 1; fun cos(x:real):real = cos(x);
    nonfix arctan 223 1; fun arctan(x:real):real = arctan(x);
    nonfix exp 224 1; fun exp(x:real):real = exp(x);
    nonfix ln 225 1; fun ln(x:real):real = ln(x);
    nonfix makestring 197 1; fun makestring(x:real):string = makestring x;

(******************)
(* lib/String.str *)
(******************)

    nonfix size 100 1; fun size(x:string):int = size x;
    nonfix substring 101 3; 
    fun substring(x:string, y:int, z:int):string = substring(x,y,z);
    val substring = fn (x:string,y:int,z:int) => substring(x,y+1,z);

    nonfix explode 48 1; fun explode(x:string):string list = explode x;
    nonfix implode 49 1; fun implode(x:string list):string = implode x;
    nonfix explodeascii 50 1; 
    fun explodeascii(x:string):int list = explodeascii x;
    nonfix implodeascii 51 1; 
    fun implodeascii(x:int list):string = implodeascii x;

    infix 4 <= 107 2; fun (a:string) <= (b:string) = a<=b: bool;
    infix 4 < 108 2; fun (a:string) < (b:string) = a<b: bool;
    infix 4 >= 109 2; fun (a:string) >= (b:string) = a>=b: bool;
    infix 4 > 110 2; fun (a:string) > (b:string) = a>b: bool;
    infix 6 ^;  fun t1 ^ t2 = implode([t1,t2]);

    exception Chr;
    fun chr(n: int): string =
       if n>=0 andalso n<=255 
       then implodeascii[n] else raise Chr;
    exception Ord;
    local
       nonfix fetch_byte 226 2
       fun fetch_byte(x:int,y:string):int = fetch_byte(x,y)
       fun ordof(s:string, n:int): int = 
         if n<0 orelse n>=(size s)
         then raise Ord else fetch_byte(n,s)
    in
    fun ord(s: string): int = 
      if (size(s)<1) then raise Ord else fetch_byte(0,s)
    end

(****************)
(* lib/Bool.str *)
(****************)
    nonfix not 30 1; fun not(x:bool):bool = not(x);
    fun makestring true = "true" | makestring false = "false";

(****************)
(* lib/List.str *)
(****************)
    nonfix hd 17 1; fun hd(x: 'a list): 'a = hd x;
    nonfix tl 18 1; fun tl(x: 'a list): 'a list = tl x;
    nonfix null 19 1; fun null(x: 'a list):bool = null x;
    nonfix length 115 1; fun length(l: 'a list):int = length(l);
    infix 5 @; fun a@b = if null a then b else (hd a)::((tl a)@b);
    nonfix rev 179 1; fun rev(l: 'a list): 'a list = rev(l);
    fun map'(f,a) = if null a then [] else f(hd(a))::map'(f,tl(a));
    fun map f a = map'(f,a);
    fun fold'(f,l,x) =
      if null l then x else f((hd l), (fold'(f, (tl l), x)));
    fun revfold'(f,l,x) = 
      if null l then x else revfold'(f, (tl l), (f((hd l), x)));
    fun fold f l x = fold'(f,l,x);
    fun revfold f l x = revfold'(f,l,x);
    fun app'(f,l) = if null l then () else (f(hd(l)); app'(f,tl(l)));
    fun revapp'(f,l) = if null(l) then () else (revapp'(f,tl l); f(hd l));
    fun app f l = app'(f,l);
    fun revapp f l = revapp'(f,l);
    nonfix nth 178 2;
    local fun nth(l: 'a list, n:int): 'a = nth(l,n)
       in val nth = fn (l,n) => nth(l,n+1) end;
    fun exists(f,[]) = false
      | exists(f,h::t) = f(h) orelse exists(f,t);

(***************)
(* lib/Ref.str *)
(***************)
    nonfix ! 54 1; fun ! (x: 'a ref): 'a = ! x;
    infix 3 := 55 2; fun (x: 'a ref) := (y: 'a) = (x := y):unit; 
    nonfix inc 65 1; fun inc(x: int ref): unit = inc x;
    nonfix dec 67 1; fun dec(x: int ref): unit = dec x;

(*****************)
(* lib/Array.str *)
(*****************)
(*    nonfix array 103 2;
    infix  3 sub 9 2;
    nonfix update 25 3; 
    nonfix length 104 1; 
    nonfix arrayoflist 121 1;

(*    exception subscript = Subscript*)

    fun array(n:int, init: '_weak): '_weak array = array(n,init)

    fun (x: 'a array) sub (y:int) = (x sub y): 'a
    fun update(x: 'a array, y:int, z: 'a):unit = update(x,y,z)
    fun arrayoflist(x: '_weak list): '_weak array = arrayoflist(x);
    fun length(x: 'a array):int = length(x);
*)

(*******************)
(* lib/BasicIO.str *)
(*******************)
(* "Execute" incorporated by NICK, Jan '88 *)

    exception Io of string;

local
    nonfix Lookahead 112 1
    nonfix InputLine 111 1
    nonfix Execute'    59 1

    fun Lookahead(s:instream):string = Lookahead(s)
    fun InputLine(s:instream):string = InputLine(s)
    fun Execute'(s: string): (instream * outstream) = Execute'(s)
in
    fun stdio() = 
      let val {StdIn, StdOut, ...} = CurrentState() in
      (StdIn, StdOut) end

    val (std_in, std_out) = stdio()

    fun open_in(fname) = 
          Openstream(fname)
          handle _ (*OpenStream*) => raise Io("Cannot open " ^ fname)

    fun open_out(fname) = 
          Createstream(fname)
          handle _ (*CreateStream*) => raise Io("Cannot open " ^ fname)

    fun input(s,cnt) = 
          Readstream(s,cnt)
          handle EndOfInput => raise Io ("end of input")
              | ReadStream => raise Io ("input")
              | _ => raise Io ("stream link lost")

    fun input_line(s) = 
          InputLine(s)
          handle EndOfInput => raise Io ("end of input")
              | ReadStream => raise Io ("input")
              | _ => raise Io ("stream link lost")

    fun lookahead(s) =
          Lookahead(s)
          handle EndOfInput => raise Io ("end of input")
              | ReadStream => raise Io ("lookahead")
              | _ => raise Io ("stream link lost")

    fun close_in(s) = 
          CloseInStream(s) handle _ => raise Io ("stream link lost")

    fun end_of_stream(s) =
          (Lookahead(s)="")
          handle EndOfInput => true
              | ReadStream => raise Io ("end_of_stream")
              | _ => raise Io  ("stream link lost")

    fun output(s,st) =
          Writestream(s,st)
          handle WriteStream => raise Io ("output")
              | _ => raise Io ("Output stream is closed")
        
    fun close_out(s) = 
          CloseOutStream(s) 
          handle _ => raise Io ("stream link lost")

    fun execute(s) =
          Execute'(s)
          handle _ (*Execute*) => raise Io ("execute")

    fun file_exists(fname) =
          (let val s = Openstream fname
           in  CloseInStream s;
               true
           end
          ) handle _ (*OpenStream*) => false
end;

(*********************)
(* lib/ByteArray.str *)
(*********************)
(* ByteArray : brought up-to-date by NICK, 21st Jan '88 *)
(*
nonfix bytearray_create 160 1;
nonfix store_byte 230 3;
nonfix fetch_byte 226 2;
nonfix extract_ 101 3; 

 (* The above should be an "abstraction", but since that isn't implemented,
    we'll have to use old-fashioned abstype instead.
  *)

 (* Diags *)
    local val {StdOut, ...} = CurrentState()
    in    fun PStr str = Writestream(StdOut, str)
          fun PInt int = PStr((makestring:int->string) int)
    end

    abstype bytearray = BA of string
    with

    type ba_repr = string		(* the representation *)
    fun bytearray_create(n: int): ba_repr = bytearray_create(n);
    fun store_byte(x:int,y:ba_repr,z:int):unit = store_byte(x,y,z)
    fun fetch_byte(x:int,y:ba_repr):int = fetch_byte(x,y)
    fun extract_(x: ba_repr, y: int, z: int) = extract_(x, y, z)

(*    exception Subscript: unit;
    exception Range: unit; *)









    fun array(size, initval): bytearray =
      if initval < 0 orelse initval > 255 then raise Range
      else if size < 0 then raise Subscript
      else let val newarr = bytearray_create(size) 
            fun initarr index = 
              if index >= size then newarr
              else ( store_byte(initval,newarr,index); initarr(index+1) ) 
         in BA(initarr 0) end

    fun update(BA ba: bytearray, pos: int, value: int): unit =
      if pos < 0 orelse pos >= (size ba) then raise Subscript
      else if value < 0 orelse value > 255 then raise Range
      else store_byte(value,ba,pos);

    fun length(BA ba) = size ba

    fun (BA ba) sub (pos: int): int =
      if pos < 0 orelse pos >= (size ba) then raise Subscript
      else fetch_byte(pos,ba)

    fun extract(BA x,y,z) =
       extract_(x,y+1,z) handle Substring => raise Subscript

    fun app f (BA ba) = 
      let val len = size ba
          fun app'(i) = 
            if i >= len then ()
            else (f(fetch_byte(i,ba)); app'(i+1))
       in app'(0) end;

    fun revapp f (BA ba) = 
      let fun revapp'(i) = 
            if i < 0 then ()
            else (f(fetch_byte(i,ba)); revapp'(i-1))
       in revapp'(size ba-1) end;

    fun fold f (BA ba) x =
      let fun fold'(i,x) = 
          if i < 0 then x else fold'(i-1,f(fetch_byte(i,ba),x))
       in fold'(size ba-1, x) end;

    fun revfold f (BA ba) x = 
      let val len = size ba
          fun revfold'(i,x) =
            if i >= len then x else revfold'(i+1,f(fetch_byte(i,ba),x))
       in revfold'(0,x) end
   end;   (* abstype *)
*)
(***************)
(* lib/Pfl.str *)
(***************)
(*    type 'a port = 'a port;
    type beh = beh;
    val NIL = NIL;
    val pfl_read = pfl_read;
    val pfl_write = pfl_write;
    val port = port;
    fun p + q = pfl_choice(p,q);
    fun p & q = pfl_par(p,q);
    val exec = exec;
    val pexec = pexec; *)

(******************)
(* lib/Output.str *)
(******************)
    val lineterm = lineterm;
    val output_string_fn = output_string_fn
    val newline_fn = newline_fn
    val check_minimum_fn = check_minimum_fn
    val flush_fn = flush_fn

(**************************)
(* new use: string-> unit *)
(**************************)
local
    val olduse = use
in
    fun use s = olduse [s]
end;

(**********)
(* system *)
(**********)
nonfix system 92 1;
fun system (x:string):unit = system x;
nonfix CpuTime 163 0;
fun  CpuTime():int = CpuTime();

(***************)
(* Composition *)
(***************)
infix 3 o 76 2; fun (x:('b -> 'c)) o (y:('a -> 'b)):('a -> 'c) = x o y;

(***********)
(* Graphic *)
(***********)

datatype colour = rgb of int * int * int;

val black   = rgb(0,0,0);
val red     = rgb(255,0,0);
val green   = rgb(0,255,0);
val blue    = rgb(0,0,255);
val yellow  = rgb(255,255,0);
val cyan    = rgb(0,255,255);
val magenta = rgb(255,0,255);
val white   = rgb(255,255,255);

abstype picture
   = a000 of real
   | a001 of real
   | a002 of colour * picture
   | a003 of real * picture
   | a004 of real * picture
   | a005 of real * picture
   | a006 of real * real * picture
   | a007 of picture * picture
   | a008 of int
   | a009 of int
   | a010 of real
with
  nonfix convertToPicture 154 1;
  fun convertToPicture(x:picture):picture = convertToPicture x;
  fun circle r      = convertToPicture(a000 r);       (* circle: radius r, origin at centre *)
  fun square x      = convertToPicture(a001 x);       (* square: side x, origin in middle *)
  fun paint(c, p)   = convertToPicture(a002(c, p));   (* force colour of entire sub-object *)
  fun scale(r, p)   = convertToPicture(a003(r, p));   (* multiply size of ofject by r *)
  fun scalex(r, p)  = convertToPicture(a004(r, p));   (* multiply size in x direction only *)
  fun rotate(a, p)  = convertToPicture(a005(a, p));   (* rotate a (degrees, clockwise) *)
  fun shift(x, y, p)= convertToPicture(a006(x, y, p));(* translate object by (x, y) *)
  fun add(p1, p2)   = convertToPicture(a007(p1, p2)); (* form composite picture *)
  val nullpic       = convertToPicture(a008(0));      (* a null image *)
  val point         = convertToPicture(a009(0));      (* an isolated point *)
  fun line(z)       = convertToPicture(a010(z));      (* horizontal line, length z, centred *)
end;

