#############################################################
# vim:set shiftwidth=2 softtabstop=2 expandtab:
#
# $Id: PerlOutput.pm 5517 2009-05-07 23:19:59Z grg $
#
# Perl file output
#
#############################################################

package NF2::RegSystem::PerlOutput;

use Exporter;

@ISA = ('Exporter');

@EXPORT = qw(
                genPerlOutput
            );

use Carp;
use NF2::RegSystem::File;
use NF2::Utils;
use NF2::RegSystem qw($PROJECTS_DIR $LIB_DIR);
use POSIX;
use Math::BigInt;
use strict;

# Path locations
my $LIB_PERL = $LIB_DIR . '/Perl5';
my $PERL_PREFIX = 'reg_defines';

my $buf;
my $consts_buf;
my @exports;

#
# genPerlOutput
#   Generate the Perl defines file corresponding to the project
#
# Params:
#   project     -- Name of project
#   layout      -- Layout object
#   usedModules -- Hash of used modules
#   constsHash  -- Hash of constants
#   constsArr   -- Array of constant names?
#   typesHash   -- Hash of types
#   typesArr    -- Array of type names?
sub genPerlOutput {
  my ($project, $layout, $usedModules, 
    $constsHash, $constsArr, $typesHash, $typesArr) = @_;

  my $memalloc = $layout->getMemAlloc();
  my $verilogOnlyMemalloc = $layout->getVerilogOnlyMemAlloc();

  # Output the constants
  outputConstants($constsArr);

  # Output the modules
  outputMemAlloc($memalloc, $verilogOnlyMemalloc);

  # Output the registers
  outputRegisters($memalloc);

  # Get a file handle
  # FIXME: Work out if ph or pm is most appropriate
  my $moduleName = "${PERL_PREFIX}_${project}";
  my $fh = openRegFile("$PROJECTS_DIR/$project/$LIB_PERL/${moduleName}.pm");

  outputHeader($fh, $project, $moduleName);
  outputBody($fh);
  outputFooter($fh);

  # Finally close the file
  closeRegFile($fh);
}

#
# outputConstants
#   Output the constants associated with each module
#
# Params:
#   fh         -- file handle
#   constsArr  -- Array of constant names
#
sub outputConstants {
  my ($constants) = @_;

  return if (length(@$constants) == 0);

  $consts_buf .= <<CONSTANTS_HEADER;
# -------------------------------------
#   Constants
# -------------------------------------

CONSTANTS_HEADER

  my $maxStrLen = 0;
  for my $constant (@$constants) {
    my $len = length($constant->name());
    $maxStrLen = $len if ($len > $maxStrLen);
  }

  # Walk through the constants
  my $currFile = '';
  for my $constant (@$constants) {
    my $name = uc($constant->name());
    my $desc = $constant->desc();
    my $value = $constant->value();
    my $width = $constant->width();
    my $wantHex = $constant->wantHex();
    my $file = $constant->file();

    if ($file ne $currFile) {
      $consts_buf .= "\n" if ($currFile ne '');
      $consts_buf .= "# ===== File: $file =====\n\n";
      $currFile = $file;
    }
    
    my $pad = (' ') x ($maxStrLen - length($name)); 

    if (defined($desc) && $desc ne '') {
      $consts_buf .= "# $desc\n";
    }

    if ($wantHex) {
      my $hexWidth = ceil($width / 4);
      my $bigVal = Math::BigInt->new($value);
      my $hexStr = substr($bigVal->as_hex(), 2);
      $hexStr = (('0') x ($hexWidth - length($hexStr))) . $hexStr;

      # Print the constant split up over multiple 32-bit values if it's
      # wider than 32 bits
      if ($width > 32) {
        outputWideConstant($name, $value, $width, $pad);
      }
			else {
				addExport("$name");
				$consts_buf .= sprintf("sub $name () $pad  { 0x%0${hexWidth}s;}\n", $hexStr);
			}

    }
    else {
			addExport("$name");
      $consts_buf .= sprintf("sub $name () $pad { $value;}\n");    
    }
    $consts_buf .= "\n";
  }
  $consts_buf .= "\n\n";
    
}

#
# outputWideConstant
#   Split a wide constant and output it as multiple small constants
#
# Params:
#
sub outputWideConstant {
  my ($name, $value, $width, $pad) = @_;

  my $bigVal = Math::BigInt->new($value);

  # Adjust the pad
  $pad = trimString($pad, 2);

  # Work out the number of sub constants
  my $numSubConsts = $width >> 5;
  $numSubConsts += 1 if (($width & (32 - 1)) != 0);

  if ($numSubConsts == 2) {
    $pad = trimString($pad, 1);
  }

  for (my $i = 0; $i < $numSubConsts; $i++) {
    my $subConst = $bigVal >> (($numSubConsts - $i - 1) * 32);
    $subConst &= 2 ** 32 - 1;

    my $hexWidth = 32;
    if ($i == 0 && (($width & (32 - 1)) != 0)) {
      $hexWidth = $width & (32 - 1);
    }
    $hexWidth = ceil($hexWidth / 4);

    my $hexStr = substr($subConst->as_hex(), 2);
    $hexStr = (('0') x ($hexWidth - length($hexStr))) . $hexStr;

    my $suffix = $i;
    if ($numSubConsts == 2) {
      $suffix = $i == 0 ? 'HI' : 'LO';
    }
		addExport("${name}_${suffix}");
    $consts_buf .= sprintf("sub ${name}_${suffix} () $pad { 0x%0${hexWidth}s;}\n", $hexStr);
  }
}

#
# outputMemAlloc
#   Output the module allocations
#
# Params:
#   memalloc    -- memory allocations
#   voMemalloc  -- Verilog only memory allocations
#
sub outputMemAlloc {
  my ($memalloc, $voMemalloc) = @_;

  return if (scalar(@$memalloc) == 0 && scalar(@$voMemalloc) == 0);

  $buf .= <<MEMALLOC_HEADER;
## -------------------------------------
##   Modules
## -------------------------------------

MEMALLOC_HEADER

  # Sort the memory allocation
  my %localMemalloc;
  my @localMemalloc;
  for my $memallocObj (@$voMemalloc, @$memalloc) {
    my $start = sprintf("%07x", $memallocObj->start());
    if (!defined($localMemalloc{$start})) {
      $localMemalloc{$start} = [$memallocObj];
    }
    else {
      push @{$localMemalloc{$start}}, $memallocObj;
    }
  }
  for my $start (sort(keys(%localMemalloc))) {
    push @localMemalloc, @{$localMemalloc{$start}};
  }

  # Walk through the list of memory allocations and print the tag/addr widths
  # for each module
  my %modules;
  my $maxModuleLen = 0;
  my $maxMemAllocLen = 0;
  for my $memallocObj (@$memalloc, @$voMemalloc) {
    # Record the module
    my $prefix = $memallocObj->{module}->prefix();
    $modules{$prefix} = $memallocObj->{module};

    # Update the max abbrev/full name lengths
    my $len = length($prefix);
    $maxModuleLen = $len if ($len > $maxModuleLen);

    $len = length($memallocObj->name());
    $maxMemAllocLen = $len if ($len > $maxMemAllocLen);
  }

  # Walk through the list of memory allocations and print them
  $buf .= "# Module tags\n";
  for my $memallocObj (@localMemalloc) {
    my $prefix = uc($memallocObj->name());
    my $start = $memallocObj->start();

    my $pad = (' ') x ($maxMemAllocLen - length($prefix));

    $buf .= sprintf("sub ${prefix}_BLOCK_ADDR () $pad  { 0x%07x; }\n", $start);
  }
  $buf .= "\n\n";
}

#
# outputRegisters
#   Output the registers associated with each module
#
# Params:
#   memalloc  -- memory allocation
#
sub outputRegisters {
  my ($memalloc) = @_;

  return if (length(@$memalloc) == 0);

  $buf .= <<REGISTER_HEADER;
# -------------------------------------
#   Registers
# -------------------------------------

REGISTER_HEADER

  # Walk through the list of memory allocations and print them
  for my $memallocObj (@$memalloc) {
    my $module = $memallocObj->module();
    my $name = $module->name();
    my $desc = $module->desc();
    my $file = $module->file();

    my $prefix = uc($memallocObj->name());
    my $start = $memallocObj->start();
    my $len = $memallocObj->len();
    my $tag = $memallocObj->tag();
    my $tagWidth = $memallocObj->{module}->tagWidth();

    $buf .= "# Name: $name ($prefix)\n";
    $buf .= "# Description: $desc\n" if (defined($desc));
    $buf .= "# File: $file\n" if (defined($file));

    my $regs = $module->getRegDump();
    outputModuleRegisters($prefix, $start, $module, $regs);
    $buf .= "\n";

    if ($module->hasRegisterGroup()) {
      my $regGroup = $module->getRegGroup();
      outputModuleRegisterGroupSummary($prefix, $start, $module, $regGroup);
    }
  }
  $buf .= "\n\n";
}

#
# outputModuleRegisters
#   Output the registers associated with a module (non register group)
#
# Params:
#   prefix    -- prefix
#   start     -- start address
#   module    -- module
#   regs      -- register array dump
#
sub outputModuleRegisters {
  my ($prefix, $start, $module, $regs) = @_;

  #my $prefix = uc($module->prefix());

  my $maxStrLen = 0;
  for my $reg (@$regs) {
    my $len = length($reg->{name});
    $maxStrLen = $len if ($len > $maxStrLen);
  }

  for my $reg (@$regs) {
    my $regName = uc($reg->{name});
    my $addr = $reg->{addr};

    my $pad = (' ') x ($maxStrLen - length($regName));

    addExport("${prefix}_${regName}_REG");
    $buf .= sprintf("sub ${prefix}_${regName}_REG () $pad  { 0x%07x;}\n", $addr + $start);
  }
}

#
# outputModuleRegisterGroupSummary
#   Output the register group summary associated with a module
#
# Params:
#   prefix    -- prefix
#   start     -- start address
#   module    -- module
#   regGroup  -- hash of used modules
#
sub outputModuleRegisterGroupSummary {
  my ($prefix, $start, $module, $regGroup) = @_;

  my $grpName = uc($regGroup->name());
  my $instSize = $regGroup->instSize();
  my $offset = $regGroup->offset();

  addExport("${prefix}_${grpName}_GROUP_BASE_ADDR");
  addExport("${prefix}_${grpName}_GROUP_INST_OFFSET");

  $buf .= sprintf("sub ${prefix}_${grpName}_GROUP_BASE_ADDR ()  { 0x%07x; }\n", $start + $offset);
  $buf .= sprintf("sub ${prefix}_${grpName}_GROUP_INST_OFFSET() { 0x%07x; }\n", $instSize);
  $buf .= "\n";
}

#
# addExport
#   Add a string to the list of exports to generate
#
# Params:
#   var       -- variable to export
#
sub addExport {
  my ($var) = @_;

  push @exports, $var;
}

#
# outputHeader
#   Output the header of the module
#
# Params:
#   fh        -- file handle
#   project   -- name of project
#   module    -- name of module
#
sub outputHeader {
  my ($fh, $project, $module) = @_;

  print $fh <<REGISTER_HEADER_1;
#############################################################
#
# Perl register defines file for $project
#
#############################################################

package $module;

use Exporter;

\@ISA = ('Exporter');

\@EXPORT = qw(
REGISTER_HEADER_1

  print $fh join("\n", map { "                " . $_} @exports) . "\n";

  print $fh <<REGISTER_HEADER_2;
            );


REGISTER_HEADER_2
}

#
# outputBody
#   Output the body of the module
#
# Params:
#   fh        -- file handle
#
sub outputBody {
  my $fh = shift;

	# Output Constants
	print $fh $consts_buf;

  # Output the content to the file
  print $fh $buf;
}

#
# outputFooter
#   Output the footer of the module
#
# Params:
#   fh        -- file handle
#
sub outputFooter {
  my ($fh) = @_;

  print $fh <<REGISTER_FOOTER;


1;

__END__
REGISTER_FOOTER

}

#
# trimString
#   Trim a string by knocking off the first n characters
#
# Params:
#   str -- string to trim
#   amt -- amount to trim by
#
# Return:
#   trimmed string (note: shortest string is an empty string)
#
sub trimString {
  my ($str, $amt) = @_;

  if (length($str) <= $amt) {
    return '';
  }
  else {
    return substr($str, $amt);
  }
}

1;

__END__
