package uk.ac.cam.cl.rkh23.webserver;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.Socket;
import java.util.StringTokenizer;


/**
 * This class embodies a really dumb HTTP connection
 * 
 * The basic idea is that we only do things when we receive
 * a request to do it from the other end of the connection 
 * (presumably a web browser).
 * 
 * The requests come in as text (Strings) and look like:
 * 
 * GET /index.html HTTP/1.0...
 * This is a request to get a file called index.html
 * 
 * HEAD /index.html HTTP/1.0...
 * This is a request to see whether there is an index.html to get
 * 
 * The response looks like:
 * 
 * HTTP/1.0 <result description>
 * Connection: close
 * Server: WebServer
 * Content-Type: text/html
 * <empty line>
 * <file contents if a GET request>
 * 
 * Don't worry about all this - it's just what we need to make
 * this work.  It won't feature in the exam!
 * 
 * @author rkh23
 *
 */
public class HTTPConnection {
	
	/**
	 * The underlying connection
	 */
	private final Socket mSocket;
	
	
	/**
	 * Basic constructor
	 * @param s The connected socket
	 */
	public HTTPConnection(Socket s) {
		mSocket = s;
	}
	
	
	/**
	 * This function runs until we have processed a single
	 * request from the connecting browser
	 */
	public void process(int delay) {
		
		try {
			// This just gets us something we can read from
			BufferedReader input = new BufferedReader(new InputStreamReader(mSocket.getInputStream()));
		
			// Wait for a message to come in
			String line = input.readLine();
			
			// Insert a manual delay
			try {
	 			Thread.sleep(delay);
	 		} catch (InterruptedException e1) {
	 			// TODO Auto-generated catch block
	 			e1.printStackTrace();
	 		}
			
			// handle the request
			handleRequest(line);
			
		}
		catch (IOException ioe) {
			// Something went wrong but it doesn't matter
			// since we will be closing the connection whatever happens
		}
		
		finally {
			// We're done here
			try {   mSocket.close();   }
			catch(IOException ioe) {} // Ignore failure - the object will be destroyed anyway
		}
		
	}
	
	
	/**
	 * Handle a single request from a browser
	 * @param line the GET or HEAD request
	 */
	private void handleRequest(String line) {
		boolean is_get_request=false;
		
		// Check what type of request it is
		if (line.startsWith("GET")) is_get_request=true;
		else if (line.startsWith("HEAD")) is_get_request=false;
		else {
			// It's not a GET or a HEAD so tell the browser
			// that we can't help
			sendHeader(501);
			return;
		}
		
		// Extract the filename (second bit of text in line)
		StringTokenizer st = new StringTokenizer(line);
		st.nextToken(); // That's the first token, either "GET" or "HEAD"
		String file = st.nextToken();
		
		// Whether it is a GET or a HEAD we need to find out about the file
		file = "."+file;
		System.out.println("Received request for file: "+file);
		
		DataInputStream input=null;
		try {
			FileInputStream f = new FileInputStream(file);
			input = new DataInputStream(f);
		}
		catch (Exception e) {
	    	// We failed to find the file so send an error
	    	System.out.println("Unable to find "+file+" so sending 404 error");
	    	sendHeader(404);
	    	return;
	    }
	    
	     
	    // OK if we are here, the file exists!
	    System.out.println("Found "+file+" so sending 200 header");
	    if (!sendHeader(200)) return;
	    
	    // If this is a GET request, we need to send the entire file too!
	    if (is_get_request) {
	    	
	    	try {
	    		DataOutputStream output =
	    			new DataOutputStream(mSocket.getOutputStream());
	    		while (true) {
	    			int b = input.read();
	    			if (b == -1) {
	    				break; //end of file
	    			}
	    			output.write(b);
	    		}
	    		System.out.println("Sent "+file+" in response to GET");
	    		
	    	}
	    	catch(Exception e) {
	    		System.out.println("Error sending file "+file);
	    	}
	    }

	}
	
	
	
	/**
	 * This method sends the header of a response
	 * to a GET or HEAD request
	 * @param result_code What type of message to send
	 */
	private boolean sendHeader(int result_code) {
		 String s = "HTTP/1.0 ";
		 
		 // You may be familiar with these codes, especially
		 // the 404 Not Found one.
		 switch (result_code) {
		 case 200:
			 s = s + "200 OK";
			 break;
		 case 404:
			 s = s + "404 Not Found";
			 break;
		 case 501:
			 s = s + "501 Not Implemented";
			 break;
		 }			

		 s = s + "\r\n"; // Add new lines
		 s = s + "Connection: close\r\n"; // We will close the connection after
		 s = s + "Server: WebServer\r\n"; //server name
		 s = s + "Content-Type: text/html\r\n"; // We only do html files
		 s = s + "\r\n"; // This marks the end of the header
		 
		
	
		 // Now send all of that over
		 try {
	    		DataOutputStream output = new DataOutputStream(mSocket.getOutputStream());
	    		output.writeBytes(s);
		 }
		 catch(Exception e) {
			 System.out.println("Failed to send header to browser");
			 return false;
		 }
		 return true;
	}
	
	
	
	
}
