package uk.ac.cam.cl.rkh23.photo.model;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.*;

import uk.ac.cam.cl.rkh23.photo.ModelInterface;
import uk.ac.cam.cl.rkh23.photo.ViewInterface;



/**
 * A PhotoCatalogue stores all the raw information
 * about photos, albums and their relations
 * @author rkh23
 *
 */
public class PhotoOrganiserModel implements ModelInterface {
	

	
	/**
	 * A set of View objects that we need to alert
	 * if anything changes
	 */
	private Set<ViewInterface> mViews = new HashSet<ViewInterface>();
	
	
	/**
	 * This maps each album name to a set of photos
	 */
	private final TreeMap<String, Set<Photo> > mPhotos = new TreeMap< String, Set<Photo> >();
	
	
	
	/**
	 * Simple constructor to initialise things
	 */
	public PhotoOrganiserModel() {
		Load();
	}
	
	
	/**
	 * Call this method when something changes and the Views
	 * need to be told. Private because we don't want anyone
	 * to be able to do this, just the class's own methods
	 */
	private void alertViews() {
		for (ViewInterface vi : mViews) {
			vi.update();
		}	
	}
	
	
	
	
	
	// ===================================================
	// Implementations of  methods from ModelViewInterface
	// ===================================================
	@Override
	public void register(ViewInterface vi) {
		mViews.add(vi);
	}
	
	@Override
	public void deregister(ViewInterface vi) {
		mViews.remove(vi);
	}
	
	@Override
	public Set<Photo> getAlbumPhotos(String album) {
		return mPhotos.get(album);
	}
	
	
	@Override
	public Set<String> getAlbums() {
		// The clone below is shallow but that's OK
		// because we just care about the Strings
		// which are immutable
		Set<String> ss = ((TreeMap)mPhotos.clone()).keySet();
		
		return ss;
	}
	
	
	// ===================================================
	// Implementations of  methods from ModelControllerInterface
	// ===================================================
	@Override
	public void createAlbum(String a) {
		mPhotos.put(a, new HashSet<Photo>());
		alertViews();
		Save();
	}
	
	@Override
	public void deleteAlbum(String a) {
		mPhotos.remove(a);
		alertViews();
		Save();
	}
	
	@Override
	public void addPhoto(String path, String a) {
		Photo p = new Photo(path);
		Set hs = mPhotos.get(a);
		if (hs!=null) hs.add(p);
		alertViews();
		Save();
	}

	@Override
	public void deletePhoto(Photo p, String a) {
		Set<Photo> hs = mPhotos.get(a);
		if (hs!=null) hs.remove(p);
		alertViews();
		Save();
	}
	
	
	
	/**
	 * This loads our data from a file
	 */
	private void Load() {
		try {
			BufferedReader input =  new BufferedReader(new FileReader("albums.txt"));
			String line = null; 
			while (( line = input.readLine()) != null){
		          StringTokenizer st = new StringTokenizer(line);
		          String album = st.nextToken();
		          if (!st.hasMoreTokens()) {
		        	  mPhotos.put(album,new HashSet<Photo>());
		          }
		          else {
		          String file = st.nextToken();
		          if (!mPhotos.containsKey(album)) {
		        	  mPhotos.put(album,new HashSet<Photo>());
		          }
		          mPhotos.get(album).add(new Photo(file));
		          }
			}

			input.close();
		}
		catch( IOException e) {
			// Ignore for now
		}
	}
	
	/**
	 * THis saves our data to a file
	 */
	private void Save() {
		try {
			PrintWriter pw = new PrintWriter(new FileOutputStream(new File("albums.txt"), false));
			Iterator it = mPhotos.entrySet().iterator();
			String album=null;
			while (it.hasNext()) {
				Map.Entry pairs = (Map.Entry)it.next();
				Set<Photo> p = (Set<Photo>)pairs.getValue();
				if (p.size()==0) {
					pw.println(pairs.getKey());
				}
				Iterator<Photo> photoit = p.iterator();
				while (photoit.hasNext()) {
					Photo pp = photoit.next();
					pw.println(pairs.getKey()+" "+pp.getPath());
				}
			}
			pw.close();
		}
		catch( IOException e) {
			// Ignore for now
		}
		
	}
	
	
	
	

}
