`define KBIDLE 0
`define KBWRITE 1
`define KBREAD 10
`define KBSETLEDS 17

`define KBNEXT kbstate+1

module ps2keyboard(	input CLK,
					input reset,
					inout PS2_CLK,
					inout PS2_DAT,
					input read,
					output dataavailable,
					output [15:0] readdata,
					output irq					
				);
	
	wire empty;			
	assign irq=!empty;
	assign dataavailable=!empty;
	
	wire [15:0]kbfifoout;
	assign readdata=kbfifoout;

	reg [15:0] kbfifoin;
	reg kbfifowrite;
	kbfifo kbfifo(
	.aclr(reset),
	.clock(CLK),
	.data(kbfifoin),
	.rdreq(read),
	.wrreq(kbfifowrite),
	.empty(empty),
	.q(kbfifoout)
	);
	
	reg [8:0]asciiaddr;
	wire [7:0]asciidata;
	keyboard_ascii kb_a(
		.address(asciiaddr),
		.clock(CLK),
		.q(asciidata)
	);
	
	reg [10:0] count60us; //1080 clocks
	reg wait60us;
	
	reg psout;
	reg psoute;
	
	reg psdat;

	reg psclk;
	reg pspclk;
	reg psclke;
	
	assign PS2_DAT=psoute?psout:1'bz;
	assign PS2_CLK=psclke?1'b0:1'bz;
	
	reg [7:0] kbsendbuf;
	
	reg [1:0] state;
	
	reg [8:0]kbbuf;
	reg [8:0]kblastscancode;

	reg [4:0]kbbufcount;

	reg kcstate;
	reg [4:0]kbretstate;
	reg [4:0]kbstate;
	reg [2:0]kbleds;
	
	reg kbextended,kbreleased,kbshift;
	
	always @(posedge CLK)
	begin
		//////////////
		//Data buffers
		psclk<=PS2_CLK;
		pspclk<=psclk;
		psdat<=PS2_DAT;

		//////////////////
		//Keyboard reset
		case(state)
		0:begin
			kbstate<=`KBWRITE;
			kbsendbuf<=8'hFF;
			state<=state+1;
		end
		1:if(kbstate==`KBIDLE)state<=state+1;
		endcase
		
		//////////////////
		//60us pauser
		if(wait60us==1)
			if(count60us==1080)
			begin
				count60us<=0;
				wait60us<=0;
			end
			else
				count60us<=count60us+1;

		/////////////////
		//Keyboard sender
		case(kbstate)
		`KBIDLE:begin
			kbretstate<=`KBIDLE;
			if(psdat==0)kbstate<=`KBREAD;
		end
		
		
		`KBWRITE:begin
			//Load data into our buffer and calculate parity
			kbbufcount<=0;
			kbbuf[7:0]<=kbsendbuf;
			kbbuf[8]<=!(^(kbsendbuf[7:0]));
			
			kbstate<=`KBNEXT;
		end
		`KBWRITE+1:begin
			//Pause 60us with the clock held low
			psclke<=1;
			if(wait60us==0)
			begin
				wait60us<=1;
				kbstate<=`KBNEXT;
			end
		end
		`KBWRITE+2:if(wait60us==0)kbstate<=`KBNEXT; //Wait for 60us
		`KBWRITE+3:begin
			//Start waiting for clock edges
			psclke<=0;
			psoute<=1;
			psout<=0;
			kbstate<=`KBNEXT;
		end
		`KBWRITE+4:if(pspclk==0 && psclk==1)kbstate<=`KBNEXT; //Wait for first positive edge, and send 0 for start bit
		`KBWRITE+5:begin
			case(kcstate)
			0:if(pspclk==0 && psclk==1)kcstate<=1; //Load a new data bit on the positive edge
			1:begin
				kcstate<=0;

				if(kbbufcount==9)
				begin
					psoute<=0;
					kbstate<=`KBNEXT;
				end
				else
				begin
					psout<=kbbuf[kbbufcount];
					kbbufcount<=kbbufcount+1;
				end
			end
			endcase
		end
		`KBWRITE+6:if(pspclk==1 && psclk==0)kbstate<=`KBNEXT; //Wait for next clock from KB
		//`KBWRITE+7:if(pspclk==1 && psclk==0)kbstate<=`KBNEXT; //Wait for next clock from KB
		`KBWRITE+7:begin
			//Check ACK bit sent by KB
			if(psdat==1)
			begin
				//No ACK bit, try resending
				kbstate<=`KBWRITE+1;
			end
			else
			begin
				//ACK bit good, finish
				kbstate<=kbretstate;
			end
		end
		
		//Reading from keyboard
		`KBREAD:begin
			kbbufcount<=0;
			kbstate<=`KBNEXT;
		end
		`KBREAD+1:if(pspclk==1 && psclk==0)kbstate<=`KBNEXT; //Wait for first negative edge and ignore the start bit
		`KBREAD+2:begin
			case(kcstate)
			0:if(pspclk==1 && psclk==0)kcstate<=1; //Wait for data bits on negative edge
			1:begin
				kcstate<=0;

				if(kbbufcount==9)
				begin
					kbstate<=`KBNEXT;
				end
				else
				begin
					kbbuf[kbbufcount]<=psdat;
					kbbufcount<=kbbufcount+1;
				end
			end
			endcase
		end
		`KBREAD+3:begin
			//Check parity bit is correct, otherwise retransmit
			if(kbbuf[8]==(^(kbbuf[7:0])))
			begin
				//Ask for a retransmit
				kbsendbuf<=8'hFE;
				kbstate<=`KBWRITE;
			end
			else
			begin
				//Parity is good, so put this scancode in buffer
				case(kbbuf[7:0])
				8'hFA:
					//Ignore ACK
					begin
						kbstate<=`KBIDLE;
					end
				8'hE0:
					//Set extended flag
					begin
						kbextended<=1;
						kbstate<=`KBIDLE;
					end
				8'hF0:
					//Set released flag
					begin
						kbreleased<=1;
						kbstate<=`KBIDLE;
					end
				8'h7E:
					//Scroll lock
					begin
						if(kbreleased==1)
							kbbuf<=9'b0;
						else if(kblastscancode!=kbbuf)
							kbleds[0]<=!kbleds[0];
						kbstate<=`KBSETLEDS;
					end
				8'h77:
					//Num lock
					begin
						if(kbreleased==1)
							kbbuf<=9'b0;
						else if(kblastscancode!=kbbuf)
							kbleds[1]<=!kbleds[1];

						kbstate<=`KBSETLEDS;
					end
				8'h58:
					//Caps lock
					begin
						if(kbreleased==1)
							kbbuf<=9'b0;
						else if(kblastscancode!=kbbuf)
							kbleds[2]<=!kbleds[2];

						kbstate<=`KBSETLEDS;
					end
				default:
					begin
						if(kbbuf[7:0]==8'h12 || kbbuf[7:0]==8'h59)
							kbshift<=!kbreleased;
						kblastscancode<=kbbuf;
						kbstate<=`KBNEXT;
						asciiaddr<={!kbextended && (kbshift ^ (kbleds[2] && (
							//Long list of capital letters
							kbbuf[7:0]==8'h15
							||
							kbbuf[7:0]==8'h4B
							||
							kbbuf[7:0]==8'h4D
							||
							kbbuf[7:0]>=8'h1A && kbbuf[7:0]<=8'h1D
							||
							kbbuf[7:0]>=8'h21 && kbbuf[7:0]<=8'h24
							||
							kbbuf[7:0]>=8'h2A && kbbuf[7:0]<=8'h2D
							||
							kbbuf[7:0]>=8'h31 && kbbuf[7:0]<=8'h35
							||
							kbbuf[7:0]>=8'h3A && kbbuf[7:0]<=8'h3C
							||
							kbbuf[7:0]>=8'h42 && kbbuf[7:0]<=8'h44
							))),kbbuf[7:0]};
					end
				endcase
			end
		end
		`KBREAD+4:kbstate<=`KBNEXT;
		`KBREAD+5:begin
			//Write kbfifo
			if(asciidata==8'h0 || (kbextended && kbbuf[7:0]!=8'h4A && kbbuf[7:0]!=8'h5A))
			begin
				kbfifoin<={3'b100,kbleds,kbextended,kbreleased,kbbuf[7:0]};
			end
			else
			begin
				kbfifoin<={3'b000,kbleds,kbextended,kbreleased,asciidata};
			end
			kbfifowrite<=1'b1;

			kbstate<=`KBNEXT;
		end
		`KBREAD+6:begin
			//Clear internal flags
			kbfifowrite<=1'b0;
			kbextended<=0;
			kbreleased<=0;

			kbstate<=`KBIDLE;
		end

		`KBSETLEDS:begin
			//Clear internal flags
			kblastscancode<=kbbuf;
			kbextended<=0;
			kbreleased<=0;

			//Send Status LED set byte
			kbsendbuf<=8'hED;
			kbretstate<=`KBSETLEDS+1;
			kbstate<=`KBWRITE;
		end
		`KBSETLEDS+1:begin
			//Send status LED byte
			kbsendbuf<={5'b0,kbleds};
			kbretstate<=`KBIDLE;
			kbstate<=`KBWRITE;
		end
		endcase
	end
endmodule