/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.remotehrd.client;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.Socket;
import java.util.LinkedList;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import t3.hrd.input.InputDeviceException;
import t3.hrd.input.PointInputDevice;
import t3.hrd.input.ShapeInputDevice;
import t3.hrd.state.JOGLHelper;
import t3.hrd.util.LoadConfigsFromProperties;


/**
 * This class represents an InputSource. InputSources are like RemoteHRDClients but they
 * provide only input, not output. They are useful in situations when input devices have to go
 * through seperate physical computers (eg you need multiple bluetooth adapters). Set the clientid
 * to be the same as the clientid of the display around which the participants using the
 * devices are sitting. If the connection cannot be established, or it fails, then the system
 * tries to reconnect every 30s.   
 * @author pjt40
 *
 */
public class RunInputSource {
	private static final Logger logger = Logger.getLogger("t3.remotehrd.client");
	
	public static void main(String[] args) {
		try {
			
			Properties properties = new Properties();
			properties.load(new FileInputStream(new File(args[0])));

			String serverName = JOGLHelper.props_getString(properties,"serverName");
			int serverPort = JOGLHelper.props_getInt(properties,"serverInputSourcePort");
			int clientId = JOGLHelper.props_getInt(properties,"clientId");
			
			LinkedList<PointInputDevice> pointInputDevices = LoadConfigsFromProperties.loadPointInputDevicesFromProperties(clientId,false,properties);
            ShapeInputDevice shapeInputDevice = LoadConfigsFromProperties.loadShapeInputDeviceFromProperties(clientId,properties);
            
			RunInputSource s = new RunInputSource(serverName,serverPort, pointInputDevices, shapeInputDevice, 40);
			s.doInputLoop();
			
		} catch(Throwable e) {
			//includes exceptions and assertionerrors
			if(logger!=null) {
				logger.log(Level.SEVERE,"Caught exception and about to exit",e);
			} else {
				e.printStackTrace();
			}
			System.exit(1);
		}
	}
	
	
	
	
	private ObjectOutputStream oos;
	private Socket socket;
	private final LinkedList<PointInputDevice> pointInputDevices;
	private final ShapeInputDevice shapeInputDevice;
	private final int sleepTime;
	private final int port;
	private final String host;
	
	public RunInputSource(
			String host,
			int port,
			LinkedList<PointInputDevice> pointInputDevices,
            ShapeInputDevice shapeInputDevice,
			int sleepTime			
	)  {
		this.host = host;
		this.port = port;
		this.sleepTime = sleepTime;
		this.pointInputDevices =pointInputDevices;
        this.shapeInputDevice = shapeInputDevice;
	}
	
	
	private int messageCount=0;
	
	/**
	 * Serializes an object and sends it to the server. If the connection fails it
	 * retries every thirty seconds.
	 * @param msg
	 */	
	private void sendMessageToServer(Serializable msg) {
		boolean reconnect = false;
		try {
			oos.writeObject(msg);
			logger.fine("Sent message to server "+msg);
			if(++messageCount>1000) {
				// reset so that we don't leak memeory
				oos.reset();
				messageCount=0;
			}
			oos.flush();
		} catch(IOException e) {
			e.printStackTrace();
			this.reconnect();
			this.sendMessageToServer(msg);
		}
	}
	
	private void reconnect() {
		boolean r = true;
		while(r) {
			try {
				if(this.socket!=null && !this.socket.isClosed()) {
					this.socket.close();
				}
				Socket socket = new Socket(this.host, this.port);
				this.socket = socket;
				this.socket.setTcpNoDelay(true);
				this.oos = new ObjectOutputStream(this.socket.getOutputStream());
				r = false;
			} catch(IOException e) {
				logger.log(Level.WARNING,"Couldn't open connection to server. Will try again in 30s...", e);
				r = true;
				try {
					
					Thread.sleep(30*1000);
				} catch(InterruptedException f) {
					throw new RuntimeException(f);
				}	
			}
			logger.log(Level.INFO,"Connected to server: "+this.socket);
		}
	}
	
	public void doInputLoop() {
		this.reconnect();		
		while(true) {
			try {
				Thread.sleep(this.sleepTime);
			} catch(InterruptedException e) {
				throw new RuntimeException(e);
			}
			for(PointInputDevice pid: this.pointInputDevices) {
				try {
					if(pid.updateState()) {
						// we have to clone the state if we are going to serialize it later.
						this.sendMessageToServer(pid.state.clone());
					}
				} catch (InputDeviceException e) {
					throw new RuntimeException(e);
				}
			}
            if(this.shapeInputDevice!=null) {
                try {
                    if(this.shapeInputDevice.updateState()) {
                        this.sendMessageToServer(shapeInputDevice.state);
                    } else {
                        // nothing to update
                    }
                } catch (InputDeviceException e) {
                    throw new RuntimeException(e);
                }
            } else {
                // don't have a shape input device
            }
		}
	}
}
