/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.remotehrd.client;



import java.awt.event.MouseEvent;
import java.awt.geom.Area;
import java.io.File;
import java.io.FileInputStream;
import java.net.Socket;
import java.util.LinkedList;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import t3.hrd.input.KeyboardInput;
import t3.hrd.input.PointInputDevice;
import t3.hrd.input.PointInputDeviceState;
import t3.hrd.input.ShapeInputDevice;
import t3.hrd.renderer.BlendOptions;
import t3.hrd.renderer.Projector;
import t3.hrd.renderer.ProjectorConfig;
import t3.hrd.state.JOGLHelper;
import t3.hrd.util.LoadConfigsFromProperties;



/**
 * A program that uses a specified properties file to create a RemoteHRD client and begin
 * rendering.
 * <p>
 * The first argument to the program is the name of the properties file. The file should
 * contain properties describing the projectors and the input devices, and also "serverName" 
 * and "serverPort" describing the name and port of the RemoteHRD server.
 * 
 * 
 * @author pjt40
 *
 */
public class RunBasicRemoteHRDClient {

	private static final Logger logger = Logger.getLogger("t3.remotehrd.client");
	static Socket socket;

	public static void main(String[] args) {
		try {
				
				
			Properties properties = new Properties();
			properties.load(new FileInputStream(new File(args[0])));

			String serverName = JOGLHelper.props_getString(properties,"serverName");
			int serverPort = JOGLHelper.props_getInt(properties,"serverPort");

			LinkedList<ProjectorConfig> projectorConfigs = LoadConfigsFromProperties.loadProjectorConfigsFromProperties(properties);
						
			int clientId = JOGLHelper.props_getInt(properties,"clientId");
			socket = new Socket(serverName, serverPort);
			LinkedList<PointInputDevice> pointInputDevices = LoadConfigsFromProperties.loadPointInputDevicesFromProperties(clientId,false,properties);
            ShapeInputDevice shapeInputDevice = LoadConfigsFromProperties.loadShapeInputDeviceFromProperties(clientId,properties);
            BlendOptions bo = new BlendOptions();
            bo.aDESKvisibleAreaMaskOrNull = LoadConfigsFromProperties.loadDESKvisibleAreaMaskFromProperties(properties);
            bo.blend=true;
            bo.blackOut = bo.BLACKOUT_RECTONLY;
            
			RemoteHRDClient remoteHRDclient= new RemoteHRDClient(
					clientId,
					projectorConfigs,
					socket, 
					new MyRemoteClientCallBacks(),
					pointInputDevices,
                    shapeInputDevice,
					JOGLHelper.props_getBoolean(properties,"shareTexturesBetweenContexts"),
					pointInputDevices.size()==0,
                    bo
				);
			
			remoteHRDclient.doRenderLoop();
            
            for(PointInputDevice pid: pointInputDevices) {
                pid.close();
            }
            if(shapeInputDevice!=null) {
                shapeInputDevice.close();
            }

		} catch(Throwable e) {
			//includes exceptions and assertionerrors
			if(logger!=null) {
				logger.log(Level.SEVERE,"Caught exception and about to exit",e);
			} else {
				e.printStackTrace();
			}
			System.exit(1);
		}
	}	
	
	
		
	


	private static class MyRemoteClientCallBacks implements RemoteHRDClientCallBacks {

		public void remoteClientCallBack_oglPaintOverlay( javax.media.opengl.GL gl, Projector d) {	}
		
		public boolean remoteClientCallBack_needToRepaintOverlay(Projector p) {
			return false;
		}
		public void remoteClientCallBack_closingBeforeLastOGL() { }
		public void remoteClientCallBack_closingAfterWindowsClosed() {
			try {
				socket.close();
			} catch(Throwable e) {
				// we're not allowed to declare that we throw exceptions so we'll throw them as runtime excpetions
				throw new RuntimeException(e);
			}
		}
		public void remoteClientCallBack_mouseClicked(MouseEvent e, Projector p) {}
        public void remoteClientCallBack_mouseEntered(MouseEvent e, Projector p) {}
        public void remoteClientCallBack_mouseExited(MouseEvent e, Projector p) {}
        public void remoteClientCallBack_mousePressed(MouseEvent e, Projector p) { }
        public void remoteClientCallBack_mouseReleased(MouseEvent e, Projector p) {  }
        public void remoteClientCallBack_mouseDragged(MouseEvent e, Projector p) { }
        public void remoteClientCallBack_mouseMoved(MouseEvent e, Projector p) { }
		
		public void remoteClientCallBack_receivedMessageFromServer(Object msg) {
			throw new IllegalStateException("Received unexpected object "+msg+" from server.");
		}
		public void remoteClientCallBack_oncePerFrame() { }
		public boolean remoteClientCallBack_keyboardInput(KeyboardInput k) {return true; }
		public boolean remoteClientCallBack_pointInputDeviceStateChanged(PointInputDeviceState pids) { return true; }
		
	}
	
	
	
}

