/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios;


import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.geom.GeneralPath;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Set;
import java.util.logging.Logger;

import javax.imageio.ImageIO;

import t3.hrd.state.JOGLHelper;
import t3.hrd.state.OrderedElement;
import t3.hrd.state.ScaRotTraTransformImmutableUniformScale;
import t3.hrd.state.Tile;
import t3.portfolios.IdentifyUnwarpedRects.UnwarpedRect;
import Jama.Matrix;



/**
 * A portfolio consists of a tile 
 * (an image of a specified size displayed on the display surface) (optional)
 * and zero or more child portfolios. 
 * It is sized, oriented and positioned on the desk surface with respect to its parent. You
 * can subclass it and override callback methods to specify event handlers and a paint procedure.
 * <p> 
 * None of these methods are thread safe. You should call them only either from a callback method, or
 * from code running in portfolioServer.peformActionAsynchronously(...).
 * <p>
 * Coordinate spaces:
 * <ul>
 * <li>DESK: 2D homogenous coordinate space, in mm, on the display surface, as defined by the primary point input device
 *  (or 2D unhomogenous DESKU).
 * <li>PORT: 2D homogenous coordinate space corresponding to this portfolio's coordinate space.
 * <li>PPORT: 2D homogenous coordinate space corresponding to this portfolio's parent's coordinate space.
 * <li>TILE: 2D homogenous coordinate space corresponding to the coordinate space of the tile's image.
 * </ul>
 * <p>
 * Making sense of the coordinate spaces:
 * <ul>
 * <li>DESK is defined.
 * <li>The root portfolio's PPORT is the same as DESK.
 * <li>For each portfolio, PORT to PPORT (i.e. its coordinate space relative to its parent's coordinate space) is
 * specified by a scale, rotation and translation. So each portfolio's coordinate space is a scaled rotate translated
 * version of its parent portfolio's coordinate space. This means that when a Portfolio moves on the display surface,
 * all its anscestors move with it automatically. 
 * <li>The tile is positioned relative to PORT (the portfolio's coordinate space) centered at (0,0) with zero rotation.
 * The width and height of the tile as it appears on the surface are specified relative to PORT. 
 * </ul>
 * <p>
 * There are methods to set the portfolio's position, orientation and size, either relative to its
 * parent portfolio or absolutely on the display surface.
 * <p>
 * Visibility is also defined relative to the parent portfolio: if the parent portfolio is hidden
 * then this portfolio is also hidden. If the parent portfolio is visible then this protfolio can
 * be either hidden or visible.
 * <p>
 * Tiles are drawn onto the display surface in the following order: this portfolio's own tile
 * is at the bottom; we recurse on the children portfolios, drawing their tiles and links on top; 
 * and then any PortfolioLinks from this portfolio are drawn on top of that.
 * <p>
 * The event model is described in the javadoc for PortfolioServer.
 * <p>
 * If you wish to call any of Portfolio's methods from code outside of portfolio callback methods 
 * then you must use performActionAsynchronouslyByCurrentThread, as described in the javadoc for 
 * PortfolioServer. 
 * 
 * @author pjt40
 *
 */
public abstract class Portfolio {
	
	private static final Logger logger = Logger.getLogger("t3.hrd.portfolios");
	

	public static final ArrayList<UnwarpedRect> unwarpedRectWordsList = new ArrayList<UnwarpedRect>(200);
	public static final ArrayList<UnwarpedRect> unwarpedRectLinesList = new ArrayList<UnwarpedRect>(200);
	
	
	public final PortfolioServer portfolioServer;
	public List<Portfolio> childrenTopToBottomReadOnly;
	public final List<PortfolioLink> portfolioLinksFromThisReadOnly;
	public final PortfolioCommonBehaviour commonBehaviour;
	
	// if parent=null then portfolio is unattached.
	private Portfolio parent; 
	private final boolean isRoot;
	private boolean destroyed;
    
    public int tileUpdateCompressionHints = 0;
	
	private ScaRotTraTransformImmutableUniformScale tPORTtoPPORT;
	private ScaRotTraTransformImmutableUniformScale tPORTtoDESK;
	private Matrix mDESKtoPORT;
	
	// DESKxRelToParent and DESKyRelToParent can be negative and it will
	// still work.
	private Rectangle2D rDESKboundingBoxOfOurTileAndAllChildrensTilesIfComputed;
	private double tileWidthInPORT, tileHeightInPORT;
	
	private final Tile tile;
	private boolean visible;

	// if true then we follow our parent's visibility. if false then we're always invisible indep of our parent
	private boolean visibleWhenParentVisible; 
	public final int portfolioFlags;
	private final int hereld_minw,  hereld_maxw, hereld_minh, hereld_maxh, hereld_contrastThresh;
	public static final int FLAG_USES_UNWARPED_RECTANGLES = 1;
    public static final int FLAG_PREALLOCATE_IMAGE_BUFFERS = 2;
    
	// if tile null then we have no tile
	// if tile not null then tile centre is at (0,0) within this portfolio space 
	// and has width and height DESKtileWidth, DESKtileheight
	// tile visible exactly when visible is true
	// when we set visible we also set visible on each of our children
	
	private List<Portfolio> childrenTopToBottom;
	

	private List<PortfolioLink> portfolioLinksFromThis;
	private List<PortfolioLink> portfolioLinksToThis;
		
	private BufferedImage[] imageBufferPreAllocations;
    private int imageBufferPreAllocation_last=0;
	
	/**
	 * Creates a new Portfolio, using sensible default values for unwarped rectangle identification.
	 * 
	 * @param isRoot			Should be false unless this is the root portfolio.
	 * @param parent			The parent portfolio. This must not be null unless you are creating a root portfolio.
	 * @param commonBehaviour	Specifies behaviour common across many portfolios, such as dragging.
	 * @param hasTile			True iff the portfolio has an associated tile.
	 * @param tileWidth			The tile width in pixels.
	 * @param tileHeight		The tile height in pixels.
	 * @param tileFlags			Tile flags. See t3.hrd.state.Tile for details. Specifying as 0 should do most of the time.
	 * @param portfolioFlags
	 */
	public Portfolio(
			boolean isRoot,
			PortfolioServer portfolioServer, 
			Portfolio parent,
			PortfolioCommonBehaviour commonBehaviour,
			boolean hasTile,
			int tileWidth,
			int tileHeight,
			int tileFlags,
			int portfolioFlags
	) {
		this(
			isRoot, portfolioServer, parent, commonBehaviour, hasTile, tileWidth, tileHeight, tileFlags, portfolioFlags,
			IdentifyUnwarpedRects.RECOMMENDED_WIDTH_MIN,
			IdentifyUnwarpedRects.RECOMMENDED_WIDTH_MAX,
			IdentifyUnwarpedRects.RECOMMENDED_HEIGHT_MIN,
			IdentifyUnwarpedRects.RECOMMENDED_HEIGHT_MAX,
			IdentifyUnwarpedRects.RECOMMENDED_CONTRAST_THRESH
		);
	}
	
	/**
	 * @param isRoot			Should be false unless this is the root portfolio.
	 * @param portfolioServer	The PortfolioServer which you wish to create the portfolio.
	 * @param parent			The parent portfolio. This must not be null unless you are creating a root portfolio.
	 * @param commonBehaviour	Specifies behaviour common across many portfolios, such as dragging.
	 * @param hasTile			True iff the portfolio has an associated tile.
	 * @param tileWidth			The tile width in pixels.
	 * @param tileHeight		The tile height in pixels.
	 * @param tileFlags			Tile flags. See t3.hrd.state.Tile for details. Specifying as 0 should do most of the time.
	 * @param usesUnwarpedRects		True iff the tile should use unwarped rectangles for improved legibility.
	 * @param hereld_areaMax	Maximum unwarped rectangles area in pixels for identifying unwarped rectangles.
	 * @param hereld_areaMin	Minimum unwarped rectangles area in pixels for identifying unwarped rectangles.
	 * @param hereld_contrastThresh		Contrast threshold for identifying unwarped rectangles.
	 */
	public Portfolio(
			boolean isRoot,
			PortfolioServer portfolioServer, 
			Portfolio parent,
			PortfolioCommonBehaviour commonBehaviour,
			boolean hasTile,
			int tileWidth,
			int tileHeight,
			int tileFlags,
			int portfolioFlags,
			int hereld_minw, 
			int hereld_maxw,
			int hereld_minh, 
			int hereld_maxh, 
			int hereld_contrastThresh
	) {
		if(!isRoot && parent==null) {
			throw new IllegalArgumentException("Either specify a parent portfolio or set isRoot to true when creating portfolios.");
		}
		if(portfolioServer == null) {
			throw new IllegalArgumentException("Must specify a portfolios manager when creating a portfolio.");
		}
		
		this.isRoot = isRoot;
		this.portfolioServer = portfolioServer;
		this.destroyed = false;
		this.commonBehaviour = commonBehaviour;
		
		this.visible = false;

		this.childrenTopToBottom = new LinkedList<Portfolio>();
		this.childrenTopToBottomReadOnly = Collections.unmodifiableList(this.childrenTopToBottom);
		
		this.portfolioLinksFromThis = new LinkedList<PortfolioLink>();
		this.portfolioLinksFromThisReadOnly = Collections.unmodifiableList(this.portfolioLinksFromThis);
		this.portfolioLinksToThis = new LinkedList<PortfolioLink>();
		
		// need to set these up before we can attach to our parent
		this.tPORTtoPPORT = new ScaRotTraTransformImmutableUniformScale( JOGLHelper.get2hmIdentity() );

        this.portfolioFlags = portfolioFlags;
        
		if(hasTile) {
			this.tileWidthInPORT=1.0;
			this.tileHeightInPORT=1.0;
			this.hereld_minw=hereld_minw;
			this.hereld_maxw=hereld_maxw;
			this.hereld_minh=hereld_minh;
			this.hereld_maxh=hereld_maxh;
			this.hereld_contrastThresh=hereld_contrastThresh;
			
			this.tile = portfolioServer.stateManager.opCreateTileAutogenerateId(tileWidth, tileHeight, tileFlags);
			this.portfolioServer.tileToPortfolio.put(this.tile, this);
		} else {
			this.tile = null;
			this.hereld_minw=0;
			this.hereld_maxw=0;
			this.hereld_minh=0;
			this.hereld_maxh=0;
			this.hereld_contrastThresh=0;
		}		
		
		if(!this.isRoot) {
			makeChildOf(parent);
		} else {
			// we have no parent!
		}
		
		// now let's get our state consistant
		this.setPORTtoPPORT( new ScaRotTraTransformImmutableUniformScale( JOGLHelper.get2hmIdentity() ) );
		if(hasTile) {
			this.setTileWidthAndHeightInPORT(1.0,1.0);
		}
        
        if(hasTile && (this.portfolioFlags & FLAG_PREALLOCATE_IMAGE_BUFFERS)!=0) {
            this.imageBufferPreAllocations = new BufferedImage[3];
            for(int i=0; i<imageBufferPreAllocations.length; i++) {
                this.imageBufferPreAllocations[i] = this.tile.createCompatibleBufferedImage(this.tile.tileWidth, this.tile.tileHeight);
            }
        }
	}	
		
	
	
	
	/* Child and parent stuff */
	
	/**
	 * Returns true iff this is the root portfolio.
	 * @return
	 */
	public final boolean isRoot() {
		return this.isRoot;
	}
	
	
	/**
	 * Returns the parent portfolio, or throws an exception if this is the root portfolio.
	 * @return
	 */
	public final Portfolio getParent() {
		if(isRoot) {
			throw new IllegalStateException("Portfolio has no parent.");
		}
		return this.parent;
	}
	
	/**
	 * Returns a list of this portfolio and all its anscestors in order
	 * with this portfolio at the head of the list and the root portfolio
	 * at the tail.
	 * @return
	 */
	public final List<Portfolio> getThisAndAnscestorsList() {
		List<Portfolio> l = this.getAnscestorsList();
		l.add(0,this);	
		return l;
	}
	
	/**
	 * Returns a set containing this portfolio and all its anscestors.
	 * @return
	 */
	public final Set<Portfolio> getThisAndAnscestorsSet() {
		Set<Portfolio> l = this.getAnscestorsSet();
		l.add(this);	
		return l;
	}
	
	/**
	 * Returns a list of all this portfolio's anscestors in order
	 * with this portfolio's parent at the head of the list and the root portfolio
	 * at the tail.
	 * @return
	 */
	public final List<Portfolio> getAnscestorsList() {
		if(this.isRoot) {
			return new LinkedList<Portfolio>();
		} else {
			List<Portfolio> l = this.parent.getAnscestorsList();
			l.add(0,this.parent);
			return l;
		}
	}
    
    
	
	/**
	 * Returns a set containing all this portfolio's anscestors.
	 * @return
	 */
	public final Set<Portfolio> getAnscestorsSet() {
		// return a set of all its parents
		if(this.isRoot) {
			return new HashSet<Portfolio>();
		} else {
			Set<Portfolio> l = this.parent.getAnscestorsSet();
			l.add(this.parent);
			return l;
		}
	}
	
	/**
	 * Changes the ancestry tree so that this portfolio becomes a parent
	 * of the specified portfolio.
	 * @param p
	 */
	public final void unhookFromParentAndMakeChildOf(Portfolio p) {
		this.checkNotDestroyed();
		if(isRoot) {
			throw new IllegalStateException("Portfolio has no parent.");
		}
		removeFromParent();
		makeChildOf(p);
	}

	
	/**
	 * Returns the portfolio corresponding to the topmost tile at the given coordinates
	 * or null if no such tile belongs to this portfolio or its anscestors.
	 * @param DESKx
	 * @param DESKy
	 * @return
	 */
	public final Portfolio getPortfolioAtCoordinates(double DESKx, double DESKy) {
		boolean inOurTileAndAllChildrensTilesBBox =
			this.getRDESKboundingBoxOfOurTileAndAllDescendantsTiles().contains(DESKx, DESKy)
			&& this.visible;
		if(inOurTileAndAllChildrensTilesBBox) {
			for(Portfolio child: this.childrenTopToBottom) {
				Portfolio result = child.getPortfolioAtCoordinates(DESKx, DESKy);
				if(result!=null) {
					return result;
				} else {
					// try another
				}
			}
			if(
				this.tile!=null
				&& this.visible
				&& this.tile.getGpDESKoutline().contains(DESKx, DESKy)
			) {
				return this;
			} else {
				return null;
			}
		} else {
			return null;
		}
	}

	
	
	
	/* order stuff */
	
	
	
	/**
	 * Sets the new child order, which governs the order in which tiles appear on the
	 * display surface. The specified portfolio will be brought to the front.
	 * @param c
	 */
	public final void bringChildToFront(Portfolio c) {
		this.checkNotDestroyed();
		if(!this.childrenTopToBottom.remove(c)) {
			throw new IllegalArgumentException("Child to be brought to the front must be a current child of the portfolio.");
		}
		this.childrenTopToBottom.add(0,c);
		this.portfolioServer.childOrderChanged();
	}
	
	/**
	 * Sets the new child order, which governs the order in which tiles appear on the
	 * display surface. The specified list must contain each child at most once.
	 * Omitted children will appear behind the specified children and their relative
	 * ordering will be preserved.
	 * @param newChildrenTopToBottom
	 */
	public final void bringChildrenToFront(List<Portfolio> newChildrenTopToBottom) {
		this.checkNotDestroyed();
		for(Portfolio c: newChildrenTopToBottom) {
			if(!this.childrenTopToBottom.remove(c)) {
				throw new IllegalArgumentException("List of children must be a subset of the portfolio's current children.");
			}
		}
		ListIterator<Portfolio> i = newChildrenTopToBottom.listIterator(newChildrenTopToBottom.size());
		while(i.hasPrevious()) {
			this.childrenTopToBottom.add(0,i.previous());
		}
		this.portfolioServer.childOrderChanged();
	}
	
	/**
	 * Sets the new child order, which governs the order in which tiles appear on the
	 * display surface. The specified list must contain each child exactly once.
	 * @param newChildrenTopToBottom
	 */
	public final void setNewChildrenOrder(List<Portfolio> newChildrenTopToBottom) {
		this.checkNotDestroyed();
		if(this.childrenTopToBottom.size()!=newChildrenTopToBottom.size()) {
			throw new IllegalArgumentException("New children list must be same size as current children list.");
		}
		for(Portfolio p: this.childrenTopToBottom) {
			if(!this.childrenTopToBottom.contains(p)) {
				throw new IllegalArgumentException("New children list must contain all of current children.");
			}
		}
		this.childrenTopToBottom = newChildrenTopToBottom;
		this.childrenTopToBottomReadOnly = Collections.unmodifiableList(this.childrenTopToBottom);
		this.portfolioServer.childOrderChanged();
	}
    
    
    public final void setChildToPosition(Portfolio child, int newIndexTopToBottom) {
        this.checkNotDestroyed();
        int oldIndOfChild = this.childrenTopToBottom.indexOf(child);
        if(oldIndOfChild==-1) {
            throw new IllegalArgumentException("Child must be a one of the portfolio's current children.");
        }
        this.childrenTopToBottom.remove(oldIndOfChild);
        this.childrenTopToBottom.add(newIndexTopToBottom, child);
        this.portfolioServer.childOrderChanged();
    }   
    
	
	final void addOrderedElsAndPortfoliosToListsInOrderForThisAndChildren(List<OrderedElement> l, List<Portfolio> ll) {
		this.checkNotDestroyed();
		// head of the returned list is the back-most tile
		// while the head of this.children is the front-most child
		if(this.tile!=null) {
			l.add(this.tile);
		} else {
			// no need to add our tile cos we don't have one!
		}
        ll.add(this);
		ListIterator<Portfolio> i = this.childrenTopToBottom.listIterator(this.childrenTopToBottom.size());
		while(i.hasPrevious()) {
			Portfolio child = i.previous();
			child.addOrderedElsAndPortfoliosToListsInOrderForThisAndChildren(l,ll);
		}
		for(PortfolioLink pl: this.portfolioLinksFromThis) {
			l.add(pl.getLink());
		}
	}
	
	
	/* destroy stuff */
	
	/**
	 * Destroys the portfolio and all its descendents.
	 */
	public final void destroyThisAndAllDescendants() {
		this.checkNotDestroyed();
        while(this.childrenTopToBottom.size()>0) {
            this.childrenTopToBottom.get(0).destroyThisAndAllDescendants();
        }
        this.customProcessAboutToBeDestroyed();
		if(!isRoot) {
			removeFromParent();
		} else {
			// no need to remove from parent
		}
		if(this.tile != null) {
			this.portfolioServer.stateManager.opDestroyTile(this.tile);
			this.portfolioServer.tileToPortfolio.remove(this.tile);
		}
		this.destroyed = true;
	}	
	
	private void checkNotDestroyed() {
		if(this.destroyed) {
			throw new IllegalStateException("Portfolio is destroyed.");
		}
	}
	
	/**
	 * Returns true iff this portfolio has been destroyed.
	 * @return
	 */
	public final boolean isDestroyed() {
		return this.destroyed;
	}
	
	/* visibility */
	
	/**
	 * Sets the visibility of this portfolio relative to its parent.
	 * @param v
	 */
	public final void setVisibleWhenParentVisible(boolean v) {
		this.checkNotDestroyed();
		this.visibleWhenParentVisible = v;
		
		//	tell all our children, and our tile about it
		this.updatedParentVisOrOurVis();
		
		// tell our parents about it
		if(!this.isRoot) {
			this.parent.childOrThisTileHasChangedBoundingRectOrVisibility();
		}
	}
	
	
	
	/**
	 * Returns true iff this portfolio is currently visible.
	 * @return
	 */
	public final boolean getVisibility() {
		return this.visible;
	}
	
	/**
	 * Returns true iff this portfolio would be visible supposing its
	 * parent portfolio is visible.
	 * @return
	 */
	public final boolean getVisibileWhenParentVisible() {
		return this.visibleWhenParentVisible;
	}
	
	
	/* aff */
	
	/**
	 * Set this portfolio's coordinate space relative to
	 * the display surface's coordinate space.
	 * @param mDESKtoPORT
	 */
	public final void setDESKtoPORT(Matrix mDESKtoPORT) {
		this.checkNotDestroyed();
		this.setPORTtoDESK( new ScaRotTraTransformImmutableUniformScale(mDESKtoPORT.inverse()) );					
	}
	
	/**
	 * Set this portfolio's coordinate space as a scaled, rotated and translated version
	 * of the display surface's coordinate space.
	 * @param tPORTtoDESK
	 */
	public final void setPORTtoDESK(ScaRotTraTransformImmutableUniformScale tPORTtoDESK) {
		this.checkNotDestroyed();
		if(this.isRoot()) {
            this.setPORTtoPPORT(tPORTtoDESK);
		} else {
			Matrix desiredmPORTtoPPORT = this.parent.mDESKtoPORT.times(tPORTtoDESK.getMatrix2dHomogReadOnly());
			this.setPORTtoPPORT( new ScaRotTraTransformImmutableUniformScale(desiredmPORTtoPPORT) );					
		}
	}
	
	/**
	 * Set this portfolio's coordinate space as a scaled, rotated and translated version
	 * of the its parent portfolio's coordinate space.
	 * @param tPORTtoPPORT
	 */
	public final void setPORTtoPPORT(ScaRotTraTransformImmutableUniformScale tPORTtoPPORT) {
		this.checkNotDestroyed();
		this.tPORTtoPPORT = tPORTtoPPORT;
	
		// tell all our children, and our tile about it
		this.updatedParentAffOrOurRelAff();
		
		// tell our parents about it
		if(!this.isRoot) {
			this.parent.childOrThisTileHasChangedBoundingRectOrVisibility();
		}
	}
	
	/**
	 * Returns the transform from this portfolio's coordinate space to the
	 * display surface's coordinate space.
	 * @return
	 */
	public final ScaRotTraTransformImmutableUniformScale gettPORTtoDESK() {
		return tPORTtoDESK;
	}
	
	/**
	 * Returns the transform from the display surface's coordinate space to
	 * this portfolio's coordinate space.
	 * @return
	 */
	public final Matrix getmDESKtoPORTReadOnly() {
		return mDESKtoPORT;
	}
	
	/**
	 * Returns the transform from this portfolio's coordinate space to its
	 * parent portfolio's coordinate space.
	 * @return
	 */
	public final ScaRotTraTransformImmutableUniformScale gettPORTtoPPORT() {
		return tPORTtoPPORT;
	}
	
	/**
	 * Returns the bounding box, in the coordinate space of the display surface,
	 * of this portfolio's tile and of all its descendents tiles. If no area then
	 * it returns a rectangle with no area, so beware when unioning - you may need
	 * to use a special union function that ignores zero-area rectangles.
	 * @return
	 */
	public final Rectangle2D getRDESKboundingBoxOfOurTileAndAllDescendantsTiles() {
        if(this.rDESKboundingBoxOfOurTileAndAllChildrensTilesIfComputed==null) {
            Rectangle2D newrDESKboundingBoxOfThisTileAndChildren = new Rectangle2D.Double(0,0,0,0);
            if(this.tile!=null && tile.isVisible()) {
                newrDESKboundingBoxOfThisTileAndChildren = 
                    JOGLHelper.createUnionButIgnoreZeroAreaRects(
                            newrDESKboundingBoxOfThisTileAndChildren,
                            this.tile.getrDESKboundingBox()
                        );
            }
            for(Portfolio child: this.childrenTopToBottom) {
                newrDESKboundingBoxOfThisTileAndChildren =
                    JOGLHelper.createUnionButIgnoreZeroAreaRects(
                        newrDESKboundingBoxOfThisTileAndChildren,
                        child.getRDESKboundingBoxOfOurTileAndAllDescendantsTiles()
                    );
            }
            this.rDESKboundingBoxOfOurTileAndAllChildrensTilesIfComputed = newrDESKboundingBoxOfThisTileAndChildren;
        }
		return rDESKboundingBoxOfOurTileAndAllChildrensTilesIfComputed;
	}
	
    /**
     * Returns the bounding box, in the coordinate space of the display surface,
     * of this portfolio's tile. 
     * @return
     */
    public final Rectangle2D getRDESKboundingBoxOfOurTile() {
        this.checkHasTile();
        return this.tile.getrDESKboundingBox();
    }
    
    /**
     * Returns the bounding box, in the coordinate space of the display surface,
     * of this portfolio's tile. 
     * @return
     */
    public final GeneralPath getGpDESKoutlineOfOurTile() {
        this.checkHasTile();
        return this.tile.getGpDESKoutline();
    }
    
    
    
	/**
	 * Converts coordinates from the coordinate space of the desk into the coordinate
	 * space of this portfolio. Coordinates are unhomogenous in a double array with two elements.
	 * @param ud2DESK
	 * @return
	 */
	public final double[] getUd2PORTfromUd2DESK(double[] ud2DESK) {
		return 
			JOGLHelper.unhomogeniseD(
				JOGLHelper.getDFromM(
					this.mDESKtoPORT.times(
						JOGLHelper.getMFromD(ud2DESK[0], ud2DESK[1], 1.0)
					)
				)
			);		
	}

	
	/**
	 * Specify the width and height of this portfolio's tile, relative to this portfolio's
	 * coordinate space.
	 * @param tileWidthInPORT
	 * @param tileHeightInPORT
	 */
	public final void setTileWidthAndHeightInPORT(double tileWidthInPORT, double tileHeightInPORT) {
		this.checkNotDestroyed();
		this.tileWidthInPORT = tileWidthInPORT;
		this.tileHeightInPORT = tileHeightInPORT;
		
		// tell all our children, and our tile about it
		this.updatedParentAffOrOurRelAff();
		
		// tell our parents about it
		if(!this.isRoot) {
			this.parent.childOrThisTileHasChangedBoundingRectOrVisibility();
		}
	}
	
	/**
	 * Get the width of this portfolio's tile, relative to this portfolio's
	 * coordinate space.
	 */
	public final double getTileWidthInPORT() { return this.tileWidthInPORT; }
	
	/**
	 * Get the height of this portfolio's tile, relative to this portfolio's
	 * coordinate space.
	 */
	public final double getTileHeightInPORT() { return this.tileHeightInPORT; }
	
	
	/* tile stuff */
	
	
	/**
	 * Returns true iff this portfolio has a tile.
	 * @return
	 */
	public final boolean hasTile() {
			return this.tile != null;
	}
    
    private void checkHasTile() {
        if(this.tile==null) {
            throw new IllegalStateException("Portfolio has no tile.");
        }
    }
	
	/**
	 * Returns the width of this portfolio's tile, in tile space, i.e. the width
	 * of the tile image in pixels.
	 * @return
	 */
	public final int getTileWidthInTILE() {
        checkHasTile();
		return this.tile.tileWidth;
	}
	
	/**
	 * Returns the height of this portfolio's tile, in tile space, i.e. the height
	 * of the tile image in pixels.
	 * @return
	 */
	public final int getTileHeightInTILE() {
        checkHasTile();
		return this.tile.tileHeight;
	}
	
	
	/**
	 * Converts coordinates from the coordinate space of the desk into the coordinate space
	 * of this portfolio's tile, rounding to the nearest integer.
	 * Coordinates are unhomogenous in an array with two elements.
	 * @param ud2DESK
	 * @return
	 */
	public final boolean getIntegerTileSpaceCoordsFromDESK(int[] intTileSpaceCoords, double[] dDESK) {
        checkHasTile();
		return this.tile.getIntegerTileSpaceCoordsFromDESK(intTileSpaceCoords, dDESK);
	}
	
	
	private void recomputeUnwarpedRects(BufferedImage bi, Rectangle areaOfBI, int ox, int oy) {
		this.checkNotDestroyed();
		assert (this.portfolioFlags & FLAG_USES_UNWARPED_RECTANGLES) !=0 ;
		/*
		 * recompute unwarped rects
		 * BufferedImage b is the image, which is part of the tile image at (ox,oy)
		 * We recompute unwarped rects in area sub of b.
		 */
						
		// generate new unwrs list
		unwarpedRectLinesList.clear();
		unwarpedRectWordsList.clear();
		this.portfolioServer.javaHereld.doItFromBufferedImageSourceUsingOurBuffers(
			unwarpedRectWordsList,
			unwarpedRectLinesList,
			bi,
			areaOfBI.x,areaOfBI.y,areaOfBI.width,areaOfBI.height,
			this.hereld_minw,
			this.hereld_maxw,
			this.hereld_minh,
			this.hereld_maxh,
			this.hereld_contrastThresh,
			true,
			false
		);
		
		try {
           ImageIO.write(
                bi,
                "png",
                new File("out-col.png"));   
		ImageIO.write(
				this.portfolioServer.javaHereld.havingDoneItNowGetGrayImage(areaOfBI.width,areaOfBI.height),
				"png",
				new File("out-g.png"));
		ImageIO.write(
                this.portfolioServer.javaHereld.havingDoneItNowGetContrastImage(areaOfBI.width,areaOfBI.height),
                "png",
				new File("out-c.png"));
        BufferedImage bb = new BufferedImage(areaOfBI.width,areaOfBI.height,BufferedImage.TYPE_INT_ARGB);
        Graphics2D gg = bb.createGraphics();
        gg.drawImage(bi,null,0,0);
        this.portfolioServer.javaHereld.drawRectsInto(unwarpedRectWordsList,gg);
		ImageIO.write(
				bb,
				"png",
				new File("out-r.png"));
		} catch(Exception e) {}
		
		logger.info("Found unwarped rects - lines: "+unwarpedRectLinesList.size()+" words: "+unwarpedRectWordsList.size());
		// create new unwrs from list
		for(UnwarpedRect ur: unwarpedRectLinesList) {
			if(!ur.filteredOut) {
				ur.r.translate(ox,oy);
				this.tile.opCreateUnwarpedRect(
					new t3.hrd.state.UnwarpedRect(ur.r,bi.getRGB(ur.bgx, ur.bgy),true)
				);
			}
		}
		for(UnwarpedRect ur: unwarpedRectWordsList) {
			if(!ur.filteredOut) {
				ur.r.translate(ox,oy);
				this.tile.opCreateUnwarpedRect(
					new t3.hrd.state.UnwarpedRect(ur.r,bi.getRGB(ur.bgx, ur.bgy),false)
				);
			}
		}
	}
	
	/**
	 * Creates a BufferedImage in a format compatible with this portfolio's tile.
	 * @param w
	 * @param h
	 * @return
	 */
	public final BufferedImage createCompatibleBufferedImage(int w, int h) {
		this.checkNotDestroyed();
        checkHasTile();
		return this.tile.createCompatibleBufferedImage(w,h);
	}
	
	
	
	/**
	 * Triggers a repaint of this portfolio's whole tile. The repaint callback will be called.
	 */
	public final void triggerRepaintEntireTile() {
		this.triggerRepaintTile(new Rectangle(0,0,this.getTileWidthInTILE(),this.getTileHeightInTILE()));
	}

	/**
	 * Triggers a repaint of a specified region of this portfolio's whole tile. The repaint callback will be called. If the tile uses unwarped rectangles then these are recomputed.
	 * @param r
	 */
	public final void triggerRepaintTile(Rectangle r) {
		 triggerRepaintTile(r,2);
	}
	
	/**
	 * Triggers a repaint of a specified region of this portfolio's whole tile. The repaint callback will be called.
	 * @param r
	 * @param recomputeUnwarpedRects (0=use same unwarped rects as before, 1=remove old unwarped rects, 2=remove old unwarped rects and recompute new ones.)
	 */
	public final void triggerRepaintTile(Rectangle r, int recomputeUnwarpedRects) {
		this.checkNotDestroyed();
        checkHasTile();
        
        long t1= System.currentTimeMillis();
        BufferedImage b;
        if((this.portfolioFlags & FLAG_PREALLOCATE_IMAGE_BUFFERS)!=0 && r.width==this.tile.tileWidth && r.height==this.tile.tileHeight && r.x==0 && r.y==0) {
            this.imageBufferPreAllocation_last=(this.imageBufferPreAllocation_last+1) % this.imageBufferPreAllocations.length;
            b = this.imageBufferPreAllocations[this.imageBufferPreAllocation_last];
        } else {
            b = this.tile.createCompatibleBufferedImage(r.width,r.height);
        }
        
        //System.out.println("Creating buffer took "+(System.currentTimeMillis()-t1));
        t1= System.currentTimeMillis();
        Graphics2D g = this.tile.createGraphics2DForTileUpdate(r,b);
        //System.out.println("Creating graphics2d took "+(System.currentTimeMillis()-t1));
        
        // text antialiasing turned off because the tile may be resized/rotated anyway.
        g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING,RenderingHints.VALUE_TEXT_ANTIALIAS_OFF);
		this.customRepaintTileForThisPortfolioNotChildren(r, b, g);
		if((this.portfolioFlags & FLAG_USES_UNWARPED_RECTANGLES) !=0) {
			if(recomputeUnwarpedRects==0) {
				// do nothing
			} else if(recomputeUnwarpedRects==1) {
				this.tile.opDestroyAllUnwarpedRectsIntersecting(r);
			} else if(recomputeUnwarpedRects==2) {
				this.tile.opDestroyAllUnwarpedRectsIntersecting(r);
				this.recomputeUnwarpedRects(b, new Rectangle(0,0,b.getWidth(),b.getHeight()), r.x, r.y);
			} else {
				assert false;
			}
		} else {
			// doesn't use unwarped rects
		}
		this.tile.opUpdateContents(r.x,r.y,b, this.tileUpdateCompressionHints);
	}
	
	/**
	 * Triggers a repaint of a specified region of this portfolio's whole tile by copying a region from
	 * another portfolio's tile. The image data is not sent across the network - the copy takes place at the
	 * actual clients, so this is more efficient than calling triggerRepaintTile. 
	 * The region is copied pixel for pixel. The repaint callback will be not be called.
	 * 
	 * @param r
	 */
	public final void triggerRepaintTileByCopyingFromOtherPortfoliosTile(int sx, int sy, int dx, int dy, int w, int h, Portfolio source) {
		this.checkNotDestroyed();
		source.checkNotDestroyed();
		if(this.tile==null || source.tile==null) {
			throw new IllegalStateException("Either this portfolio or the source portfolio has no tile.");
		}		
		this.tile.opUpdateContentsByCopyingFromOtherTile(sx, sy, dx, dy, w, h, source.tile);
	}

	/**
	 * Returns as a 3x3 matrix the transform from the coordinate space of the display surface
	 * to the coordinate space of this portfolio's tile.
	 * @return
	 */
	public final Matrix getmDESKtoTILE() {
        checkHasTile();
		return this.tile.getmDESKtoTILE();
	}
	
		
	
	
	
	
	/* custom event stuff */
		
	/**
	 * Implement this method to draw into the portfolio's tile.  The supplied Graphics2D
	 * is in the same coordinate space as TILE, with the specified rectangle set to
	 * draw into the supplied BufferedImage.
	 * 
	 * @param r			rectangle in TILE space that must be redrawn. 
	 * @param update	BufferedImage of same dimensions as r, into which you draw. 
	 * @param g			Graphics2D which is configured to draw into update.  
	 */
	public abstract void customRepaintTileForThisPortfolioNotChildren(Rectangle r, BufferedImage update, Graphics2D g);
	
	/**
	 * Implement this method to receive events. 
	 * See PortfolioServer for more information on the event model.
	 * Note that any events handled by this portfolio's PortfolioCommonBehaviour object will
	 * not be passed to this method.
	 * @param e			Event
	 * @param bubbled	True iff the event did not occur on this portfolio's tile but has been bubbled from one of its children.
	 * @return			True iff the event should not be bubbled to this portfolio's parent.
	 */
	protected abstract boolean customProcessEventForThisPortfolioNotChildren(PortfolioEvent e, boolean bubbled);
	
	
	/**
	 * Implement this method to receive FDOP mode events. 
	 * See PortfolioServer for more information on the event model.
	 * Note that any events handled by this portfolio's PortfolioCommonBehaviour object will
	 * not be passed to this method.
	 * @param e		Event
	 * @param PORTxWhenEnteredFDOPmode	PORT space coordinate of the PID when it entered FDOP mode.
	 * @param PORTyWhenEnteredFDOPmode	PORT space coordinate of the PID when it entered FDOP mode.
	 */
	protected abstract void customProcessFDOPevent(PortfolioEvent e, double PORTxWhenEnteredFDOPmode, double PORTyWhenEnteredFDOPmode);
	
	/**
	 * Implement this method to be notified when a PID stops being in FDOP mode. 
	 * See PortfolioServer for more information on the event model.
	 * Note that any notifications handled by this portfolio's PortfolioCommonBehaviour object will
	 * not be passed to this method.
	 * @param pid
	 * @param button
	 */
	protected abstract void customProcessEndOfFDOPmode(PointInputDevice pid, int button);
	
	/**
	 * Called when the portfolio is about to be destroyed.
	 */
	protected void customProcessAboutToBeDestroyed() {        
    }

		

	/* links stuff */
	
	final void notifyCreatedLinkFromThis(PortfolioLink l) {
		this.checkNotDestroyed();
		this.portfolioLinksFromThis.add(l);
		
	}	
	
	final void notifyDestroyedLinkFromThis(PortfolioLink l) {
		this.checkNotDestroyed();
		assert this.portfolioLinksFromThis.contains(l);
		this.portfolioLinksFromThis.remove(l);	
	}
	
	final void notifyCreatedLinkToThis(PortfolioLink l) {
		this.checkNotDestroyed();
		this.portfolioLinksToThis.add(l);	
	}	
	
	final void notifyDestroyedLinkToThis(PortfolioLink l) {
		this.checkNotDestroyed();
		assert this.portfolioLinksToThis.contains(l);
		this.portfolioLinksToThis.remove(l);	
	}
	
	
	/* private stuff */
	
	private void makeChildOf(Portfolio p) {		
		assert p!=this;
        
		this.parent = p;
		this.parent.childrenTopToBottom.add(0,this);
        
        if(tPORTtoDESK!=null) {
            // make sure it doesn't move position 
            Matrix desiredmPORTtoPPORT = this.parent.mDESKtoPORT.times(tPORTtoDESK.getMatrix2dHomogReadOnly());
            this.tPORTtoPPORT = new ScaRotTraTransformImmutableUniformScale(desiredmPORTtoPPORT) ;
        } else {
            // wasn't positioned relative to desk to begin with
            // this can happen when a portfolio is first initialised.
        }

		// tell all our children, and our tile about it
		this.updatedParentAffOrOurRelAff();
		this.updatedParentVisOrOurVis();
		
		// tell our parents about it
		if(!this.isRoot) {
			this.parent.childOrThisTileHasChangedBoundingRectOrVisibility();
		}
		
		// correct the child order
		this.portfolioServer.childOrderChanged();
	}
	
	private void removeFromParent() {		
		this.parent.childrenTopToBottom.remove(this);
		
		// tell our parents about it
		if(!this.isRoot) {
			this.parent.childOrThisTileHasChangedBoundingRectOrVisibility();
		}
		
		this.parent = null;
	}
	
	private void updatedParentAffOrOurRelAff() {		
		if(this.isRoot()) {
			this.tPORTtoDESK = this.tPORTtoPPORT;
		} else {
			this.tPORTtoDESK = new ScaRotTraTransformImmutableUniformScale(
				this.parent.tPORTtoDESK.getMatrix2dHomogReadOnly()
				.times(this.tPORTtoPPORT.getMatrix2dHomogReadOnly())
			);
		}
		this.mDESKtoPORT = this.tPORTtoDESK.getMatrix2dHomogReadOnly().inverse();
				
		if(this.tile!=null) {
			this.tile.opSetAff(
					this.tPORTtoDESK.getTx(), 
					this.tPORTtoDESK.getTy(), 
					this.tileWidthInPORT*this.tPORTtoDESK.getScale(), 
					this.tileHeightInPORT*this.tPORTtoDESK.getScale(), 
					this.tPORTtoDESK.getThetaClockwise()
				);
		}
        this.rDESKboundingBoxOfOurTileAndAllChildrensTilesIfComputed = null;
        
		for(Portfolio child: this.childrenTopToBottom) {
			child.updatedParentAffOrOurRelAff();
		}
		
        // we don't have to notify the parents using childOrThisHasChangedBoundingRect
		// because we're walking the tree and doing it anyway.
		
		for(PortfolioLink pl: this.portfolioLinksFromThis) {
			pl.notifyOfSourceOrDestPortfolioMove();
		}
		for(PortfolioLink pl: this.portfolioLinksToThis) {
			pl.notifyOfSourceOrDestPortfolioMove();
		}
	}
	
	private void updatedParentVisOrOurVis() {
		boolean oldVisibility = this.visible;
		if(isRoot) {
			this.visible = this.visibleWhenParentVisible;
		} else {
			this.visible = this.parent.getVisibility() && this.visibleWhenParentVisible;
		}
		if(this.visible != oldVisibility) {
			if(this.tile!=null) {
				this.tile.opSetVisibility(this.visible);
			}

            this.rDESKboundingBoxOfOurTileAndAllChildrensTilesIfComputed = null;
            
            for(Portfolio child: this.childrenTopToBottom) {
				child.updatedParentVisOrOurVis();
			}

            // we don't have to notify the parents using childOrThisHasChangedBoundingRect
			// because we're walking the tree and doing it anyway.
			
			for(PortfolioLink pl: this.portfolioLinksFromThis) {
				pl.notifyOfSourceOrDestPortfolioVisChange();
			}
			for(PortfolioLink pl: this.portfolioLinksFromThis) {
				pl.notifyOfSourceOrDestPortfolioVisChange();
			}
			
		} else {
			// nothing changed so don't waste time propogating
		}
	}
	
	private void childOrThisTileHasChangedBoundingRectOrVisibility() {
        this.rDESKboundingBoxOfOurTileAndAllChildrensTilesIfComputed = null;
        if(!this.isRoot) {
			this.parent.childOrThisTileHasChangedBoundingRectOrVisibility();
		}
	}


}
