/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.state;


import java.util.logging.Logger;

import Jama.Matrix;

/**
 * Represents a transform in 2D homogenous space, where the transform consists of
 * a positive scale, a rotation (about the origin) and a translation.
 * <p>
 * You can specify the transform either by geometric parameters (sx, sy, thetaClockwise, tx, ty)
 * or by a 3*3 matrix. 
 * <p>
 * If you specify a matrix and then subsequently call a get method to get one of the 
 * geometric parameters then the class will calculate the geometric parameters and
 * cache them for later reuse.
 * <p>
 * If you specify geometric parameters then the matrix is calculated when the object
 * is created.
 * 
 * @author pjt40
 *
 */
public class ScaRotTraTransformImmutable {
	
	private static final Logger logger = Logger.getLogger("t3.hrd.state");
	
	
	private Matrix mh2;
	private double tx, ty, sx, sy, thetaClockwise;
	private boolean paramsCalculatedFromMatrix;
		
	public ScaRotTraTransformImmutable(double sx, double sy, double thetaClockwise, double tx, double ty) {
		if(sx==0.0 || sy==0.0) {
			throw new IllegalArgumentException("Cannot have zero scale factor");
		}
		this.sx = sx;
		this.sy = sy;
		this.thetaClockwise = thetaClockwise;
		this.tx = tx;
		this.ty = ty;
		this.mh2 = JOGLHelper.get2hmScaRotTra(sx,sy,thetaClockwise,tx,ty);
			/*JOGLHelper.get2hmTranslation(tx,ty).times(
				JOGLHelper.get2hmRotation(thetaClockwise).times(
					JOGLHelper.get2hmScale(sx,sy)
				)
			);	*/		
		this.paramsCalculatedFromMatrix = true;		
	}

	public ScaRotTraTransformImmutable(Matrix mh2) {
		this.mh2 = mh2;
		this.paramsCalculatedFromMatrix = false;
	}
	
	public Matrix getMatrix2dHomogReadOnly() { return this.mh2; }
	public double getTx() { calculateParamsIfNotAlready(); return this.tx; }
	public double getTy() { calculateParamsIfNotAlready(); return this.ty; }
	public double getSx() { calculateParamsIfNotAlready(); return this.sx; }
	public double getSy() { calculateParamsIfNotAlready(); return this.sy; }
	public double getThetaClockwise() { calculateParamsIfNotAlready(); return this.thetaClockwise; }
	
	
	private void calculateParamsIfNotAlready() {
		if(!this.paramsCalculatedFromMatrix) {
			// decomposes M into scale by s then rotate by thetaclockwise, then translate by tx,ty
			// M is   
			// s cos theta,   s sin theta,     tx
			// - s sin theta, s cos theta,  ty
			// 0, 0, 1
			//
			// or for non-uniform s:
			//
			// M is   
			// sx cos theta,   sy sin theta,     tx
			// - sx sin theta, sy cos theta,  ty
			// 0, 0, 1
			//
			assert inRange( mh2.get(2,0), 0.0);
			assert inRange( mh2.get(2,1), 0.0);
			assert inRange( mh2.get(2,2), 1.0);	
			double sxCosTheta = mh2.get(0,0);
			double sySinTheta = mh2.get(0,1);	
			double sxSinTheta = -mh2.get(1,0);
			double syCosTheta = mh2.get(1,1);
			double sxSquared = sxCosTheta*sxCosTheta+sxSinTheta*sxSinTheta;
			double sySquared = syCosTheta*syCosTheta+sySinTheta*sySinTheta;
			
			this.sx = Math.sqrt(sxSquared);
			this.sy = Math.sqrt(sySquared);
			assert this.sx!=0.0;
			assert this.sy!=0.0;
			
			this.thetaClockwise = Math.atan2(sxSinTheta, sxCosTheta) ;
			
			this.tx = mh2.get(0,2);
			this.ty = mh2.get(1,2);		
			this.paramsCalculatedFromMatrix = true;
		}
	}
	
	private static boolean inRange(double a, double b) {
		return inRange(a,b,0.000001);
	}
	
	private static boolean inRange(double a, double b, double delta) {
		return a==b || (a>b-delta && a<b+delta); 
	}
	
	public String toString() {
		calculateParamsIfNotAlready();
		return super.toString() + " sx="+this.sx+" sy="+this.sy+" thetaClockwise="+this.thetaClockwise+" tx="+this.tx+" ty="+this.ty;
	}

}
