/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.state;

import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.awt.geom.Rectangle2D;
import java.util.logging.Logger;

import Jama.Matrix;

/**
 * Represents a scaled rotated translated rectangle.
 * @author pjt40
 *
 */
public class ScaRotTraRectangle {
	
	private static final Logger logger = Logger.getLogger("t3.hrd.state");
	
	private static Matrix[] mh2StandardRectCornerPointsClockwise = 
		new Matrix[] {
			JOGLHelper.getMFromD(0, 0, 1),
			JOGLHelper.getMFromD(0, 1, 1),
			JOGLHelper.getMFromD(1, 1, 1),
			JOGLHelper.getMFromD(1, 0, 1)
		};
	
	private static Matrix mh2StandardCentrePoint = JOGLHelper.getMFromD(0.5, 0.5, 1.0);
	
	private Matrix[] mh2ScaRotTraRectCornerPointsClockwiseIfComputed; 
	private Matrix mh2ScaRotTra;
	private Matrix mh2ScaRotTraCentrePoint;
	
	private Rectangle2D  rScaRotTraRectBoundingBoxIfComputed;
	private GeneralPath gpScaRotTraRectOutlineIfComputed;
	
	
	
	public ScaRotTraRectangle(Matrix mScaRotTra) {
		//	currently you may not specify negative scale!
		
		this.mh2ScaRotTra = mScaRotTra;
		this.gpScaRotTraRectOutlineIfComputed = null;
		this.rScaRotTraRectBoundingBoxIfComputed = null;
		this.mh2ScaRotTraCentrePoint=null;
	}
	
	
	


	public void translate(double dx, double dy) {
		this.mh2ScaRotTra = JOGLHelper.get2hmTranslation(dx,dy).times(this.mh2ScaRotTra);
		this.mh2ScaRotTraCentrePoint=null;
		if(this.gpScaRotTraRectOutlineIfComputed!=null) {
			AffineTransform at = new AffineTransform();
			at.translate(dx,dy);
			this.gpScaRotTraRectOutlineIfComputed.transform(at);
		}
		if(this.rScaRotTraRectBoundingBoxIfComputed!=null) {
			this.rScaRotTraRectBoundingBoxIfComputed.setRect(
				this.rScaRotTraRectBoundingBoxIfComputed.getX()+dx,
				this.rScaRotTraRectBoundingBoxIfComputed.getY()+dy,
				this.rScaRotTraRectBoundingBoxIfComputed.getWidth(),
				this.rScaRotTraRectBoundingBoxIfComputed.getHeight()
			);
		}
		// don't need to worry about extreme points. the indices will stay the same.
	}

	public Matrix get2hmScaRotTraCentrePoint() {
		if(this.mh2ScaRotTraCentrePoint==null) {
			this.mh2ScaRotTraCentrePoint = this.mh2ScaRotTra.times(mh2StandardCentrePoint);
		}
		return mh2ScaRotTraCentrePoint;
	}
	
	public GeneralPath getGpScaRotTraOutlineReadOnly() {
		if(this.gpScaRotTraRectOutlineIfComputed==null) {
			this.gpScaRotTraRectOutlineIfComputed=JOGLHelper.getGeneralPathFrom2hmPoly(
					this.get2hmScaRotTraRectCornerPointsReadOnly()
			);
		}
		return gpScaRotTraRectOutlineIfComputed;
	}

	public Rectangle2D getRScaRotTraBoundingBoxReadOnly() {
		if(this.rScaRotTraRectBoundingBoxIfComputed==null) {
			this.rScaRotTraRectBoundingBoxIfComputed = JOGLHelper.getBoundingRectFrom2hmPoly(this.get2hmScaRotTraRectCornerPointsReadOnly(),null);
		}
		return rScaRotTraRectBoundingBoxIfComputed;
	}


	public static Matrix[] get2hmStandardRectCornerPointsClockwiseReadOnly() {
		return mh2StandardRectCornerPointsClockwise;
	}
	
	
	public Matrix[] get2hmScaRotTraRectCornerPointsReadOnly() {
		if(mh2ScaRotTraRectCornerPointsClockwiseIfComputed==null) {
			this.mh2ScaRotTraRectCornerPointsClockwiseIfComputed = JOGLHelper.applyMatrixToSetOfMatrices(this.mh2ScaRotTra,mh2StandardRectCornerPointsClockwise);
		}
		return this.mh2ScaRotTraRectCornerPointsClockwiseIfComputed;
	}
	


	/*private int indLeftmost, indRightmost, indBottommost, indTopmost, indTopLeft, indTopRight, indBottomLeft, indBottomRight;
	private boolean aligned;
	private boolean calculatedExtremePoints;
	private boolean unhomogedThreeOnlyScaRotTraCornerPoints;*/
	
	/*
	public Matrix[] getMscaRotTraCornerPointsUnhomogStillThreeReadOnly() {
		if(!this.unhomogedThreeOnlyScaRotTraCornerPoints) {
			Matrix[] mPoints = getMscaRotTraCornerPointsReadOnly();
			for(Matrix m: mPoints) {
				JOGLHelper.unHomogeniseMInPlaceLeavingThreeEls(m);
			}
			this.unhomogedThreeOnlyScaRotTraCornerPoints=true;			
		}
		return this.mh2ScaRotTraCornerPoints;
	}
	
	
	
	
	private void calculateExtremePoints() {
		
		if(!this.calculatedExtremePoints) {
		
			Matrix[] mPoints = this.getMscaRotTraCornerPointsUnhomogStillThreeReadOnly();
			
					
			aligned = mPoints[0].get(0,0)==mPoints[2].get(0,0);
			
			if(!aligned) {
				indLeftmost=-1;
				for(int i=0; i<mPoints.length; i++) {
					if(indLeftmost==-1 || mPoints[i].get(0,0)<mPoints[indLeftmost].get(0,0)) {
						indLeftmost=i;
					}
				}
				indTopmost = (indLeftmost+1) % 4;
				indRightmost = (indLeftmost+2) % 4;
				indBottommost = (indLeftmost+3) % 4;
			} else {
				indTopLeft=-1;
				for(int i=0; i<mPoints.length; i++) {
					if(
						indTopLeft==-1 
						|| mPoints[i].get(1,0)>mPoints[indTopLeft].get(1,0)
						|| (mPoints[i].get(0,0)<mPoints[indTopLeft].get(0,0) && mPoints[i].get(1,0)==mPoints[indTopLeft].get(1,0))
					) {
						indTopLeft=i;
					}
				}
				indTopRight = (indTopLeft+1) % 4;
				indBottomRight = (indTopLeft+2) % 4;
				indBottomLeft = (indTopLeft+3) % 4;
			}		
			
			this.calculatedExtremePoints=true;
		}
		
	}

	public boolean isAligned() {
		this.calculateExtremePoints();
		return aligned;
	}

	public int getIndBottomLeft() {
		this.calculateExtremePoints();
		assert this.aligned;
		return indBottomLeft;
	}

	public int getIndBottommost() {
		this.calculateExtremePoints();
		assert !this.aligned;
		return indBottommost;
	}

	public int getIndBottomRight() {
		this.calculateExtremePoints();
		assert this.aligned;
		return indBottomRight;
	}

	public int getIndLeftmost() {
		this.calculateExtremePoints();
		assert !this.aligned;
		return indLeftmost;
	}

	public int getIndRightmost() {
		this.calculateExtremePoints();
		assert !this.aligned;
		return indRightmost;
	}

	public int getIndTopLeft() {
		this.calculateExtremePoints();
		assert this.aligned;
		return indTopLeft;
	}

	public int getIndTopmost() {
		this.calculateExtremePoints();
		assert !this.aligned;
		return indTopmost;
	}

	public int getIndTopRight() {
		this.calculateExtremePoints();
		assert this.aligned;
		return indTopRight;
	}
	*/
	
}
