/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.renderer;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.GraphicsEnvironment;
import java.awt.geom.AffineTransform;
import java.awt.geom.Area;
import java.awt.geom.GeneralPath;
import java.awt.geom.Rectangle2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Logger;

import javax.imageio.ImageIO;

import t3.hrd.state.JOGLHelper;
import Jama.Matrix;

/**
 * Class stores all transforms necessary for a single projector and also computes these transforms
 * based on the ProjectorConfig.ProjectorTransformConfig object supplied. This includes calculating
 * a least squares solution for the planar homography based on the calibration points supplied in
 * the ProjectorConfig object.
 * <p>
 * Coordinate spaces we use:
 * <ul>
 * <li>UOGL: 2D homogenous unwarped OpenGL. 
 * <li>&nbsp;&nbsp;&nbsp;&nbsp;(planar homography)
 * <li>WOGL: 2D homogenous warped OpenGL.
 * <li>&nbsp;&nbsp;&nbsp;&nbsp;(set z=zconst)
 * <li>TOGL: 3D homogenous warped OpenGL. What you actually draw in with OpenGL commands.
 * <li>&nbsp;&nbsp;&nbsp;&nbsp;(OpenGL's projective transform)
 * <li>FB: 3D homogenous framebuffer space
 * <li>&nbsp;&nbsp;&nbsp;&nbsp;(throw away z component)
 * <li>FBT: 2D homogenous framebuffer space
 * <li>&nbsp;&nbsp;&nbsp;&nbsp;(planar homography caused by projector positioning and optics wrt surface)
 * <li>DESK: 2D homogenous, in mm, on the display surface, as defined by the primary point input device
 *  (or 2D unhomogenous DESKU)
 * <li>&nbsp;&nbsp;&nbsp;&nbsp;()
 * </ul>
 * <p>
 * DESK to UOGL should therefore be just a translation and a positive scale
 * 
 * @author pjt40
 *
 */
public class ProjectorTransforms {
	// JAMA multiplication: if C=A*B then Matrix C = A.times(B);

	private static final Logger logger = Logger.getLogger("t3.hrd.renderer");
	
    public static final boolean BLEND = true;
    
	public static final double openGlView_left = -1.0;
	public static final double openGlView_width = 2.0;
	public static final double openGlView_bottom = -1.0;
	public static final double openGlView_height = 2.0;
	public static final double openGlView_z = -1.0;
	public static final double openGlView_near = 1.0;
	public static final double openGlView_far = 60.0;
	public static final double openGlView_depthRangeF = 1.0;
	public static final double openGlView_depthRangeN = 0.0;
	
	public final static Matrix mFBtoFBT = JOGLHelper.getm3hTo2hDiscardZDoNotProject();;
	public final static Matrix mWOGLtoTOGL = JOGLHelper.getm2hTo3hSettingZToConst(openGlView_z);
	public final static Matrix[] mWOGLcornersOfProjectorLimitClockwise = getmWOGLcornersClockwise();
	public final static Matrix[] mTOGLcornersOfProjectorLimitClockwise  = JOGLHelper.applyMatrixToSetOfMatrices(mWOGLtoTOGL, mWOGLcornersOfProjectorLimitClockwise);
	public final ProjectorConfig projectorConfig;

	public final Rectangle2D rDESKdeskSpaceAlignedRect;	
	public final Matrix[] mDESKcornersOfProjectorLimitClockwiseInWOGL;
	public final GeneralPath gpDESKcornersOfProjectorLimitClockwiseInWOGL;
	public final boolean UOGLtoWOGLflipsVert;
	public final boolean UOGLtoWOGLflipsHoriz;
	public final double UOGLtoWOGLrotation; // clockwise
	public final int fbWidth, fbHeight;
	public final Matrix mUOGLtoWOGL;
	public final Matrix mUOGLtoTOGL;
	public final Matrix mDESKtoUOGLpositiveScaleAndTranslateOnly;
	public final Matrix mTOGLtoFB;
	public final Matrix mDESKtoTOGL;
	public final Matrix mDESKtoFB;
	public final Matrix mFBtoTOGL;
	public final Matrix mFBTtoDESK;
	public final Matrix mDESKtoFBT;
	
	Rectangle2D rDESKdeskSpaceAlignedNonBlackedRect;	
	Area aDESKnonBlackedArea;
    Matrix[] mDESKcornersOfNonBlackedArea;
    
    
	
	ProjectorTransforms(ProjectorConfig c) {

		this.projectorConfig = c;
		
		this.fbWidth = GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices()[c.graphicsDeviceIndex].getDisplayMode().getWidth();
		this.fbHeight = GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices()[c.graphicsDeviceIndex].getDisplayMode().getHeight();
		
		this.mDESKcornersOfProjectorLimitClockwiseInWOGL = c.mDESKcornersOfProjectorLimitClockwiseInWOGL;
		this.gpDESKcornersOfProjectorLimitClockwiseInWOGL = JOGLHelper.getGeneralPathFrom2hmPoly(this.mDESKcornersOfProjectorLimitClockwiseInWOGL); 		
		this.rDESKdeskSpaceAlignedRect = this.gpDESKcornersOfProjectorLimitClockwiseInWOGL.getBounds2D();
		
		Matrix mWOGLtoDESK = JOGLHelper.getPlaneToPlaneHomogMatrixFromN2hmPointCorrespondances(getmWOGLcornersClockwise(), this.mDESKcornersOfProjectorLimitClockwiseInWOGL,4);
				
		
		this.mDESKtoUOGLpositiveScaleAndTranslateOnly =
			getmDESKtoUOGLpositiveScaleAndTranslateOnly(this.rDESKdeskSpaceAlignedRect);
		
		Matrix mWOGLtoUOGL = this.mDESKtoUOGLpositiveScaleAndTranslateOnly.times(mWOGLtoDESK);
		this.mUOGLtoWOGL =mWOGLtoUOGL.inverse();
		
		this.mTOGLtoFB = 
			JOGLHelper.getViewportMatrix(
				(double)this.fbWidth,
				(double)this.fbHeight,
				(double)this.fbWidth/2.0,
				(double)this.fbHeight/2.0,
				openGlView_depthRangeF,
				openGlView_depthRangeN)
			.times(
				JOGLHelper.getProjectionMatrixFromFrustum(
					openGlView_left, 
					openGlView_left+openGlView_width, 
					openGlView_bottom, 
					openGlView_bottom+openGlView_height, 
					openGlView_near, 
					openGlView_far
				)
			);
		this.mFBtoTOGL = this.mTOGLtoFB.inverse();
				
		// now work out the derived ones
		this.mUOGLtoTOGL = this.mWOGLtoTOGL.times(this.mUOGLtoWOGL);
		this.mDESKtoTOGL = this.mUOGLtoTOGL.times(this.mDESKtoUOGLpositiveScaleAndTranslateOnly);
		this.mDESKtoFB = this.mTOGLtoFB.times(this.mDESKtoTOGL);
		this.mDESKtoFBT = this.mFBtoFBT.times(this.mDESKtoFB);
		this.mFBTtoDESK = this.mDESKtoFBT.inverse();
		
		
		this.UOGLtoWOGLflipsHoriz = c.UOGLtoWOGLflipsHoriz;
		this.UOGLtoWOGLflipsVert = c.UOGLtoWOGLflipsVert;
		this.UOGLtoWOGLrotation = c.UOGLtoWOGLrotationClockwise;
		
	}
	
	public double[] d2hUOGLtoD3hTOGL(double[] d2hUOGL) {
		return
			JOGLHelper.getDFromM(
				this.mUOGLtoTOGL.times(					
					JOGLHelper.getMFromD(d2hUOGL)
				)
			);
	}
	
	public double[] dDESKtoDTOGL(double[] d2hUOGL) {
		return
			JOGLHelper.getDFromM(
				this.mUOGLtoTOGL.times(					
					JOGLHelper.getMFromD(d2hUOGL)
				)
			);
	}
	
	public double[][] dDESKsTodTOGLs(double[][] dDESKs) {
		double[][] dTOGLs = new double[dDESKs.length][];
		for(int i =0; i<dDESKs.length; i++) {
			dTOGLs[i] = d2hUOGLtoD3hTOGL(dDESKs[i]);
		}
		return dTOGLs;
	}	
		
	
	
	static private Matrix getmDESKtoUOGLpositiveScaleAndTranslateOnly(Rectangle2D rDESKdeskSpaceAlignedRect) {
		// translate
		Matrix tr = JOGLHelper.get2hmTranslation(
				-rDESKdeskSpaceAlignedRect.getX(), 
				-rDESKdeskSpaceAlignedRect.getY()
				);
		// scale so scale is right
		Matrix sc = JOGLHelper.get2hmScale(
				openGlView_width/rDESKdeskSpaceAlignedRect.getWidth(),
				openGlView_height/rDESKdeskSpaceAlignedRect.getHeight()
				);
		// translate
		Matrix tr2 = JOGLHelper.get2hmTranslation(
				openGlView_left, 
				openGlView_bottom
				); 
		// now combine and return
		return tr2.times(sc.times(tr));
	}
	
	public static Matrix[] getmWOGLcornersClockwise() {
		Matrix[] mWOGLcornersClockwise = new Matrix[4];
		mWOGLcornersClockwise[0] = JOGLHelper.getMFromD(new double[] {openGlView_left, openGlView_bottom, 1 });
		mWOGLcornersClockwise[1] = JOGLHelper.getMFromD(new double[] {openGlView_left, openGlView_bottom+openGlView_height, 1 });
		mWOGLcornersClockwise[2] = JOGLHelper.getMFromD(new double[] {openGlView_left+openGlView_width, openGlView_bottom+openGlView_height, 1 });
		mWOGLcornersClockwise[3] = JOGLHelper.getMFromD(new double[] {openGlView_left+openGlView_width, openGlView_bottom, 1 });
		return mWOGLcornersClockwise;
	}
	
	
    
	
	
	/*
	  	each projector has inners - bits that are excl that projectors and are not masked
		
		stage 0. start with 
			inners are quadrilaterals 5mm inside the extent of each projector
			outers are the rest
		stage 1. do inner intersections and maintain constraints on outers - ie outers don't intersect inners ever.
			start with projector i
				intersect it with each projector j st j>i
				if inner of i intersects inner of j
					inner of i stays same
					inner of j := inner of j - extent of i
					outer of i stays same since we have altered inner of j so it doesn't intersect outer of i
					outer of j needs to be made smaller:
						outer of j := outer of j - inner of i + (outer of j INT extent of i)
				else
					do nothing.
	 * 
	 */

	public static final boolean DEBUG_OUTPUT_BLEND_IMAGES = false;
    
	
	public static void setNonBlackedAreas(
		List<ProjectorTransforms> projectorTransformsInBlackingOrder,
        BlendOptions bo
	) {
        for(ProjectorTransforms pt: projectorTransformsInBlackingOrder) {
            pt.aDESKnonBlackedArea = null;
        }
        setaDESKnonBlackedAreas(projectorTransformsInBlackingOrder, bo);
        
        if(DEBUG_OUTPUT_BLEND_IMAGES) { 
            BufferedImage bi = new BufferedImage(2000,1000,BufferedImage.TYPE_INT_ARGB);
            BufferedImage bi2 = new BufferedImage(2000,1000,BufferedImage.TYPE_INT_ARGB);
            Graphics2D g = bi.createGraphics();
            g.translate(100,100);
            Graphics2D g2 = bi2.createGraphics();
            g2.translate(100,100);
            for(ProjectorTransforms pt: projectorTransformsInBlackingOrder) {
                //JOGLHelper.fillShapeInRandomColor(g, pt.aDESKnonBlackedArea);
                JOGLHelper.fillShapeInRandomColor(g, pt.gpDESKcornersOfProjectorLimitClockwiseInWOGL);
                
                JOGLHelper.fillAreaPolysInRandomColors(g2, pt.aDESKnonBlackedArea);
            }                
            try {
                ImageIO.write(bi,"png",new File("nonblackedareas-1.png"));
                ImageIO.write(bi2,"png",new File("nonblackedareas-2.png"));
            } catch(Exception e) {
                throw new RuntimeException(e);
            }
        }
        
        for(ProjectorTransforms pt: projectorTransformsInBlackingOrder) {
            assert pt.aDESKnonBlackedArea!=null;
            pt.rDESKdeskSpaceAlignedNonBlackedRect = pt.aDESKnonBlackedArea.getBounds2D();
            List<List<Matrix>> polys = JOGLHelper.get2hmPolysFromPolygonalArea(pt.aDESKnonBlackedArea);
            assert polys.size()>0;
            if(polys.size()>1) {
                throw new IllegalStateException("Non-blacked area consists of more than one polygon.");                
            }
            
            pt.mDESKcornersOfNonBlackedArea = JOGLHelper.getMatrixArrayFromMatrixList(polys.get(0));
        }
    }
    
    // total width of desired overlap region between two projectors
    // when projectors are positioned, all must overlap by at least this.
    private static final double widthOfOverlapRegions = 20.0; 
    
    
    
    private static void setaDESKnonBlackedAreas(
        List<ProjectorTransforms> projectorTransformsInBlackingOrder,
        BlendOptions bo
    ) {        
        if(bo.blackOut==bo.BLACKOUT_RECTONLY) {
            for(ProjectorTransforms pt: projectorTransformsInBlackingOrder) {
                pt.aDESKnonBlackedArea = new Area(JOGLHelper.getGeneralPathFrom2hmPoly(pt.mDESKcornersOfProjectorLimitClockwiseInWOGL));
                if(bo.aDESKvisibleAreaMaskOrNull!=null) {
                    pt.aDESKnonBlackedArea.intersect(bo.aDESKvisibleAreaMaskOrNull);
                }
            }
        } else {            
            assert bo.blackOut==bo.BLACKOUT_REDUCEDOVERLAP || bo.blackOut==bo.BLACKOUT_ALLOVERLAP;
        
    		LinkedList<Area> partsToSubtractByProjector = new LinkedList<Area>();
    		LinkedList<Area> nonBlackedByProjector = new LinkedList<Area>();
    		
    		
    		/************ stage 0 **************/
    		
    		for(ProjectorTransforms pc: projectorTransformsInBlackingOrder) {
    			
    			Area nonBlacked = new Area(pc.gpDESKcornersOfProjectorLimitClockwiseInWOGL);
                if(bo.aDESKvisibleAreaMaskOrNull!=null) {
                    nonBlacked.intersect(bo.aDESKvisibleAreaMaskOrNull);
                } else {
                    // do nothing
                }
                nonBlackedByProjector.add(nonBlacked);
    			
                Area subtract;
    			if(bo.blackOut==bo.BLACKOUT_REDUCEDOVERLAP) {
    				double centreX = nonBlacked.getBounds2D().getCenterX();
    				double centreY = nonBlacked.getBounds2D().getCenterY();
    				double w = nonBlacked.getBounds2D().getWidth();
    				double h = nonBlacked.getBounds2D().getHeight();
    				AffineTransform extentToSubtr = 
    					AffineTransform.getTranslateInstance(centreX, centreY);
                    // has to be 2*overlapwidth because top and bottom.
    				extentToSubtr.concatenate(
    						AffineTransform.getScaleInstance((w-2.0*widthOfOverlapRegions)/w, (h-2.0*widthOfOverlapRegions)/h)
    					);
    				extentToSubtr.concatenate(
    					AffineTransform.getTranslateInstance(-centreX, -centreY)
    				);
    				subtract = nonBlacked.createTransformedArea(extentToSubtr);
    			} else {
    				subtract = JOGLHelper.getCopyOfArea(nonBlacked);
    			}
                partsToSubtractByProjector.add(subtract);
    		}
    		
    		
    		/************* stage 1 *******************/
    		
    		for(int i=0; i<nonBlackedByProjector.size(); i++) {
    			Area iSubtract = partsToSubtractByProjector.get(i);
    			for(int j=i+1; j<nonBlackedByProjector.size(); j++) {
    				Area jNonBlacked = nonBlackedByProjector.get(j);
    				
                    jNonBlacked.subtract(iSubtract);
    			}
            }
            
    			

            for(int i=0; i<nonBlackedByProjector.size(); i++) {
                Area cleaned = JOGLHelper.cleanUpPolygonalArea(nonBlackedByProjector.get(i));
                List<Matrix> l = JOGLHelper.get2hmPolysFromPolygonalArea(cleaned).get(0);
                
                // limb removing is hacky but seems to work in pratice. wouldn't make claims about it! */                 
                JOGLHelper.removeLimbsWidthLessThanD(l, 1.05*widthOfOverlapRegions);
                cleaned = new Area(JOGLHelper.getGeneralPathFrom2hmPoly(l));
                projectorTransformsInBlackingOrder.get(i).aDESKnonBlackedArea = cleaned;
    		}
    		
    		
    	
        }
	}
	
    
    
    


	public void paintBlendImage(BlendOptions bo, Graphics2D g, List<Projector> projectorsInBlackingOrder,
            final boolean doInColorChannelsForDebugNotAlphaChannel) {
        /*
         * For colouring g, use new Color(0,0,0, zeroToTwoFiveFive).
         * If zeroToTwoFiveFive is 0 then full intensity, mask pixel is transparent black
         * If zeroToTwoFiveFive is 255 then no intensity, mask pixel is opaque black
         */
        if(bo.blend) {
            logger.info("Creating blend mask...");
            g.setComposite(java.awt.AlphaComposite.Src);
            if(projectorsInBlackingOrder.size()==1) {
                // cheat to avoid calculations for each pixel
                g.setColor( new Color(0,0,0,0));
                g.fillRect(0,0,this.fbWidth, this.fbHeight);
            } else {
                for(int y=0; y<this.fbHeight; y++) {
                    for(int x=0; x<this.fbWidth; x++) {
                        Matrix mFBTp = JOGLHelper.getMFromD(x+0.5,y+0.5,1.0);
                        Matrix mDESKp = this.mFBTtoDESK.times(mFBTp);
                        double DESKpx = mDESKp.get(0,0)/mDESKp.get(2,0);
                        double DESKpy = mDESKp.get(1,0)/mDESKp.get(2,0);
                                            
                        double top = paintBlendImageNewAlg_alpha(DESKpx, DESKpy, this);
                        double bot = 0.0;
                        if(top==0.0) {
                            // no need to bother!
                        } else {
                            for(Projector p: projectorsInBlackingOrder) { 
                                bot+= paintBlendImageNewAlg_alpha(DESKpx, DESKpy, p.transforms);
                            }
                        }
                        assert top==0.0 || bot!=0.0;
                        double topdivbot = (top==0.0) ? 0.0 : top/bot;
                        // if topdivbot is 1.0 then full intensity
                        // if topdivbot is 0.0 then no intensity
                        double a;
                        if(doInColorChannelsForDebugNotAlphaChannel) {
                            a = topdivbot;
                        } else {
                            a = paintBlendImageNewAlg_adjustAlpha(topdivbot);
                        }
                        // if a is 1.0 then full intensity
                        // if a is 0.0 then no intensity
                        int zeroToTwoFiveFive = (int)(255.0*(1.0-a));
                        if(doInColorChannelsForDebugNotAlphaChannel) {
                            g.setColor( new Color(255-zeroToTwoFiveFive,255-zeroToTwoFiveFive,255-zeroToTwoFiveFive, 255));
                        } else {
                            g.setColor( new Color(0,0,0, zeroToTwoFiveFive));
                        }
                        g.drawLine(x,y,x,y);
                    }
                }     
            }
            logger.info("Done creating blend mask.");           
        } else {
            g.setComposite(java.awt.AlphaComposite.Src);
            g.setColor( new Color(0,0,0,255) );
            g.fillRect(0,0,this.fbWidth,this.fbHeight);
            g.setColor( new Color(0,0,0,0));
            List<List<Matrix>> mDESKpolys = 
                JOGLHelper.get2hmPolysFromPolygonalArea(this.aDESKnonBlackedArea);
            
            for(List<Matrix> mDESKpoly: mDESKpolys) {                   
                g.fill(
                    JOGLHelper.getGeneralPathFrom2hmPoly(
                        JOGLHelper.applyMatrixToSetOfMatrices(
                            this.mDESKtoFBT, 
                            mDESKpoly
                        )
                    )
                );
            }
            g.setComposite(java.awt.AlphaComposite.SrcOver);            
        }
    }
	
    public static double paintBlendImageNewAlg_adjustAlpha(double a) {
        // if a is 1.0 then full intensity
        // if a is 0.0 then no intensity
        // do gamma correction. 0.4 seems to work well for our projectors
        return Math.pow(a,0.4);
    }
    
    public static final boolean BLEND_USES_HARVILLE=false;
    
    public static double paintBlendImageNewAlg_alpha(double DESKpx, double DESKpy, ProjectorTransforms i) {
        if(i.aDESKnonBlackedArea.contains(DESKpx, DESKpy)) {
            // need distance of p from nearest side of gp
            if(BLEND_USES_HARVILLE) {
                return JOGLHelper.getHarvilleDistOfPointFromPoly(
                        DESKpx, DESKpy,
                        i.mDESKcornersOfNonBlackedArea
                );
            } else {
                return JOGLHelper.getDistOfPointFromPoly(
                        DESKpx, DESKpy,
                        i.mDESKcornersOfNonBlackedArea
                );
            }
        } else {
            return 0.0;
        }
    }
	
}
