/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.renderer;

import java.util.LinkedList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import t3.hrd.input.InputDeviceException;
import t3.hrd.input.PointInputDevice;
import t3.hrd.input.ShapeInputDevice;
import t3.hrd.state.StateManager;
import t3.hrd.util.FPSTimer;

/*
 * Threading notes:
 * 
 * You may only update the statemanager from callback routines. This ensures a
 * single threaded model. 
 * 
 * Problems arise trying to do anything multithreaded becaause opengl rendering has to 
 * be done by the awt event processing thread, apparently. When we call GLCanvas.display()
 * it adds a runnable to the awt event queue that does the opengl rendering and blocks
 * until this runnable is completed. Crucially this means that we can't guarantee
 * that other AWT events won't get between our projector display requests. So we might
 * end up with, eg:
 * 		GLCanvas.display for projector 1
 * 		GLCanvas.display for projector 2
 * 		GLCanvas.display for projector 3
 * 		<Random AWT event>
 * 		GLCanvas.display for projector 4
 * and if the random event does anything with the statemanager then we have lost our
 * consistancy. We can't use locks either because if we hold a lock while we do all the
 * GLCanvas.displays() and try to get the lock when we process our random awt event then
 * we get deadlock!
 * 
 * And anyway, it's better to have our control code in another process because that way
 * we can use things like Swing keyboard focuses.
 * 
 * TODO store up awt generated inputdeviceevents and keyboardevents in a queue and process
 * them in the main thread, not the awt thread.
 */

/**
 * This class creates a multiprojector display that renders
 * tiles, cursors and links. Create the tiles, cursors and links
 * by calling methods on its stateManager object.
 * <p>
 * Threading notes: You may only update the statemanager from callback routines - 
 * this avoids race conditions and deadlock.  
 * 
 * @author pjt40
 *
 */
public class HRDRenderer {
	
	
	
	/**
	 * Call methods on the stateManager object to create and manipulate
	 * Tiles, Cursors and Links on the display. Threading notes: 
	 * You may only update the statemanager from callback routines.
	 * This ensures a single-threaded model and avoids deadlock.
	 */
	public final StateManager stateManager;
	
	public final HRDRendererCallBacks callBacks;
	public final boolean mouseAsPointInputDevice;

	public final int clientId;
	
	private static final Logger logger = Logger.getLogger("t3.hrd.renderer");

	final StateListenerForHRDRenderer stateListener;
	final List<Projector> projectors;
	final List<ProjectorConfig> projectorConfigs;
	
	
	// null if no sharing or, if so, the primary projector 
	Projector shareTexturesBetweenContextsPrimaryProjector;
    final BlendOptions blendOptions; 
	
	private LinkedList<PointInputDevice> pointInputDevices;
    private ShapeInputDevice shapeInputDevice;
	
	private volatile boolean requestClose = false;	
	private volatile boolean requestForceRedrawAll = false;
	public String infoString;
    
	private int closing = 0;
	
	
	
	/**
	 * Create a new HRDRenderer. This creates all the OpenGL windows but does not start the render loop.
	 * You should call the doRenderLoop method once you have created the HRDRenderer.
	 * 
     * @param clientId              Client ID for this client
	 * @param projectorConfigs		Configuration of each projector, in blacking order (ie head of list has no blacking applied)				
	 * @param callBacks				Callbacks allow you
	 * @param pointInputDevices		Pointing devices used with the display
     * @param shapeInputDevice      Shape device
     * @param shareTexturesBetweenContexts      Iff this is true then opengl textures are shared between the different opengl windows, 
     *      using the opengl features to share textures between contexts.
	 * @param mouseAsPointInputDevice	Iff this is true then mouse events are converted to point input device events. 
	 *      It has personId 0 (or 1 if ALT held down) and penType 0 (or 1 if CTRL held down).
     * @param aDESKvisibleAreaOrNull      A mask that can be applied to restrict the visible area eg to a rectangle. If null then no mask.
	 * @throws Projector.ProjectorOpeningException	If there are problems opening the window
	 */

	public HRDRenderer(
			int clientId,
			List<ProjectorConfig> projectorConfigs,
			HRDRendererCallBacks callBacks,
			LinkedList<PointInputDevice> pointInputDevices,
            ShapeInputDevice shapeInputDevice,
			boolean shareTexturesBetweenContexts,
			boolean mouseAsPointInputDevice,
            BlendOptions bo
	) throws Projector.ProjectorOpeningException {
		AWTErrorHandler.setHandler();
		this.clientId = clientId;
		this.stateListener = new StateListenerForHRDRenderer(this);
		this.stateManager =  new StateManager(stateListener, true, true, 2048, 512);
		this.projectorConfigs = projectorConfigs;
		this.callBacks = callBacks;
		this.pointInputDevices = pointInputDevices;
        this.shapeInputDevice = shapeInputDevice;
		this.mouseAsPointInputDevice = mouseAsPointInputDevice;
		this.projectors = new java.util.LinkedList<Projector>();
		this.blendOptions = bo;
        
		List<ProjectorTransforms> projectorTransforms = new LinkedList<ProjectorTransforms>();
		for(ProjectorConfig c: this.projectorConfigs) {
			projectorTransforms.add(new ProjectorTransforms(c));
		}
		        
		ProjectorTransforms.setNonBlackedAreas(projectorTransforms, bo);
		
		if(shareTexturesBetweenContexts) {
			boolean first = true;
			for(ProjectorTransforms dc:projectorTransforms) {
				if(first) {
					this.shareTexturesBetweenContextsPrimaryProjector
						= new Projector(dc, this, null);
					projectors.add( this.shareTexturesBetweenContextsPrimaryProjector );
					first=false;
				} else {
					projectors.add( new Projector(dc, this, this.shareTexturesBetweenContextsPrimaryProjector ) );
				}
			}
		} else {
			this.shareTexturesBetweenContextsPrimaryProjector = null;
			for(ProjectorTransforms dc:projectorTransforms) {
				projectors.add( new Projector(dc, this, null ) );
			}
		}
        
        // TESTING!! Just output masks only
        /*for(ProjectorTransforms pt: projectorTransforms) {
            BufferedImage bi = new BufferedImage(pt.fbWidth,pt.fbHeight,BufferedImage.TYPE_INT_ARGB);
            Graphics2D g = bi.createGraphics();
            pt.paintBlendImage(bo, g, this.projectors, true);
            try {
                ImageIO.write(bi,"png",new File("mask-"+pt.hashCode()+".png"));
            } catch(Exception e) {
                throw new RuntimeException(e);
            }
        }
        System.exit(0);*/
		
	}
	
	

	/**
	 * Run the HRDRenderer in single threaded mode. Go into a loop: 
	 * call callback_oncePerFrame; 
	 * then redraw each projector if it needs to be redrawn (1); 
	 * then call updateState on each PointInputDevice.
	 * <p>
	 * Returns only when the display is closed by pressing F12 or calling requestClose().
	 * It does not close the PointInputDevices.
	 * <p>
	 * Note that at point (1) the actual opengl rendering may well take place in a 
	 * different thread, with this thread blocked until it completes. The thread that
	 * does the OpenGL rendering is determined by the java.media.opengl.Threading class
	 * and depends on the platform but this can be overridden by specifying 
	 * -Dopengl.1thread=(option) at the command line. On our system it didn't make any
	 * difference to performance. See java.media.opengl.Threading for more information.
	 */
	public void doItSingleThreaded() {
		/* tried various threading options:
		 * 
		 * 1. running all this in the jogl opengl renderer thread
		 * but it didn't work - the jogl opengl renderer thread is the awt thread
		 * and it's not a good idea to block it with an infinite loop because you
		 * stop receiving eg mouse and keyboard events.
		 * 
		 * 2. running each projector in its own thread with java opengl multithreading
		 * enabled, but that made things slightly slower
		 * 
		 * also tried doing bufferswaps manually instead of automatically, but this
		 * just made things slightly slower.
		 */

		this.closing=0;
		logger.info("Starting HRDRenderer in single threaded mode.");
		FPSTimer fpsTimer = new FPSTimer("FRAME RATE = ","",1000000000.0,0.3,2000,8000);
		Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
        
		this.requestForceRedrawAll();
		
		
		while(!requestClose) {
			
			fpsTimer.oneFrame();	
			
			stateListener.clear();	
			this.updateAllPidsStatesIfNecessary(true);
            
            // shape input device!
            if(this.shapeInputDevice!=null) {
                try {
                    long t1 = System.currentTimeMillis();
                    if(this.shapeInputDevice.updateState()) {
                        this.callBacks.callback_shapeInputDeviceStateChanged(shapeInputDevice.state.clone(), this);
                        long t2 = System.currentTimeMillis();
                        //0System.out.println("Timing: "+(t2-t1));
                    } else {
                       // no state change
                    }
                } catch (InputDeviceException e) {
                    throw new RuntimeException(e);
                }
            } else {
                // we don't have a shape input device!
            }
            
			this.callBacks.callback_oncePerFrame();
			
			final boolean forceRedrawAll = this.requestForceRedrawAll;
			this.requestForceRedrawAll = false;
			
			boolean first = true;
			for(Projector d: projectors) {
				if(!first) {
					// we update pids states before each projector draws to ensure a high sampling rate
					this.updateAllPidsStatesIfNecessary(false);						
				} else {
					// don't update pids states if we've only just done it
				}				
				d.oncePerFrameRender(forceRedrawAll);
				first = false;
			}
			
			for(Projector d: projectors) {
				d.oncePerFrameSwapBuffersIfNecessary();
			}
			
		}
		
		this.requestClose = false;
		this.closing = 1;
		this.callBacks.callback_closingBeforeLastOGL();	
		for(Projector d: projectors) {
			d.oncePerFrameRender(true);
		}		
		for(Projector d: projectors) {
			d.close();		
		}					
		HRDRenderer.this.callBacks.callback_closingAfterWindowsClosed();
	}
	
	
	private long timeOfLastPidsStatesUpdate = 0;
	private FPSTimer pidTimer = new FPSTimer("PID samples per second = ","",100000000.0,0.3,2000,8000);
	
	private void updateAllPidsStatesIfNecessary(boolean forceUpdate) {
		if(forceUpdate || System.currentTimeMillis() > timeOfLastPidsStatesUpdate+40 ) {
			for(PointInputDevice pid: this.pointInputDevices) {
				try {
					if(pid.updateState()) {
						// we have to clone the state if we are going to serialize it later.
						this.callBacks.callback_pointInputDeviceStateChanged(pid.state.clone(), this);
					} else {
                        // no change               
                    }
				} catch (InputDeviceException e) {
					throw new RuntimeException(e);
				}
			}
			this.timeOfLastPidsStatesUpdate = System.currentTimeMillis();
			this.pidTimer.oneFrame();

		} else {
			// do nothing, was less than 40ms since last call.
		}
	}
	
	
	boolean isLastOpenGLRedrawBeforeClosing() {
		return this.closing==1;
	}
	
	
	/**
	 * Request that the display be closed and the render loop stopped.
	 * This method returns immediately and is thread-safe: it can be called by any thread.
	 */
	public void requestClose() {
		// may be called by any thread
		this.requestClose = true;		
	}
	
	
	
	/**
	 * Request that all projectors be redrawn.
	 * This method returns immediately and i thread-safe: it can be called by any thread.
	 */
	public void requestForceRedrawAll() {
		// may be called by any thread
		this.requestForceRedrawAll = true;		
	}
	
	boolean keyPressed(java.awt.event.KeyEvent e) {
		// called asynchronously
		if(e.getKeyCode()==e.VK_F12) {
			this.requestClose();
			return true;
		} else {
			return false;
		}
	}
	
	
	public static class AWTErrorHandler {

		public AWTErrorHandler() {
			// called only by awt.
		}

		public void handle(Throwable e) {
			if(logger!=null) {
				logger.log(Level.SEVERE,"Caught exception and about to exit",e);
			}
			e.printStackTrace();
			System.exit(1);
		}
		
		public static void setHandler() {
			System.setProperty("sun.awt.exception.handler",AWTErrorHandler.class.getName());
		}
	}
	
	
	

	
}
