/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.input;

import java.util.Properties;

import Jama.Matrix;

import t3.hrd.state.JOGLHelper;
import uk.org.pjt.jmaxellanotostream.PenListener;
import uk.org.pjt.jmaxellanotostream.PenStreamingConnection;

/**
 * PointInputDevice for use with Maxell Anoto streaming pens.
 * 
 * @author pjt40
 *
 */
public class MaxellAnotoPen extends PointInputDevice implements PenListener  {

	
	private final PenStreamingConnection psc;
	private final int cdbClickButton, cdbDragButton, cdbNotClickOrDragButtonOrMinusOne, cdbNotClickOrDragButtonsBitMap, noncdbPenDownButton;
    private final boolean clickDragButtonEmulation;
	private final static int divJump = 10;

	private static final double millimetresPerPenUnit = 0.03711263;
	
	private boolean changed = false;
	private int penOldX, penOldY;
	
	// click drag button stuff
	private int cdbState = 1;
	private long cdbTime = 0;
	private double cdbDESKox, cdbDESKoy;
	private static final long CDB_MAXTIMEDOWN=200;
	private static final long CDB_MAXTIMEUP=400;
	private static final double CDB_MAXDESKMOVE=2.0;
	
	private final boolean accurate;
	
	// filter spurious penups
	//private boolean filterPenUpState = false;
	//private long filterPenUpTime=0;
	//private static final long FILTERPENUP_TIME=50;
	
	
	public MaxellAnotoPen(int clientId, int personId, Matrix mINPUTtoDESK, boolean accurate, Properties p, String prefix) throws InputDeviceException {
		super(clientId,0,personId, mINPUTtoDESK);
		this.accurate=accurate;
		try {
			String port = JOGLHelper.props_getString(p,prefix+"comPort");
            this.clickDragButtonEmulation = JOGLHelper.props_getBoolean(p,prefix+"clickDragButtonEmulation");
            this.noncdbPenDownButton = JOGLHelper.props_getInt(p,prefix+"noncdbPenDownButton");
            this.cdbDragButton = JOGLHelper.props_getInt(p,prefix+"cdbDragButton");
			this.cdbClickButton = JOGLHelper.props_getInt(p,prefix+"cdbClickButton");
            this.cdbNotClickOrDragButtonOrMinusOne = JOGLHelper.props_getInt(p,prefix+"cdbNotClickOrDragButtonOrMinusOne");
            this.cdbNotClickOrDragButtonsBitMap = this.cdbNotClickOrDragButtonOrMinusOne==-1 ? 0 : (1<<this.cdbNotClickOrDragButtonOrMinusOne);
			this.psc = PenStreamingConnection.createInstanceFromPortName(port,this,true,true);
		} catch(Exception e) {
			throw new InputDeviceException(e);
		}
	}
	
	@Override
	public boolean updateState() throws InputDeviceException {
		
		/*if(this.filterPenUpState && System.currentTimeMillis()>FILTERPENUP_TIME+this.filterPenUpTime) {
			// pen has been in the pen up state too long. it wasn't a spurious 
		}*/
		// timings check
        if(this.clickDragButtonEmulation) {
    		if(this.cdbState==2 && System.currentTimeMillis()>CDB_MAXTIMEDOWN+this.cdbTime) {
    			// transition C
    			this.cdbState=3;
    			assert this.state.buttons==this.cdbNotClickOrDragButtonsBitMap;
    			assert this.state.positionAndButtonsKnown==true;
    			this.changed = false;
    			psc.poll();
    			return this.changed;				
    		} else if(this.cdbState==4 && this.state.buttons!=0) {
    			// transition E
    			this.state.buttons=0;
    			return true;
    			// needed to return here so that the system registers that 
    			// we now have no buttons pressed
    		} else if(this.cdbState==4 && System.currentTimeMillis()>CDB_MAXTIMEUP+this.cdbTime) {
    			// tranisition F
    			this.cdbState = 1;
    			assert this.state.buttons==0;
    			this.state.positionAndButtonsKnown=false;
    			psc.poll();
    			return true;
    		} else {
    			this.changed = false;
    			psc.poll();
    			return this.changed;			
    		}
        } else {
            this.changed = false;
            psc.poll();
            return this.changed;                    
        }
	}

	@Override
	public void close() throws InputDeviceException {
		psc.close();

	}
	
	private void actuallyPenUp() {
        if(this.clickDragButtonEmulation) {
    		if(this.cdbState==3) {
    			// transition J
    			this.cdbState=1;
    			this.state.positionAndButtonsKnown=false;
    			this.state.buttons=0;
    		} else if(this.cdbState==6) {
    			// transition H
    			this.cdbState=4;
    			this.state.positionAndButtonsKnown=true;
    			this.state.buttons=0;
    			this.cdbTime = System.currentTimeMillis();
    		} else if(this.cdbState==2) {
    			// time check is done elsewhere
    			if(
    				Math.abs(this.cdbDESKox-this.state.DESKx)<this.CDB_MAXDESKMOVE
    				&& Math.abs(this.cdbDESKoy-this.state.DESKy)<this.CDB_MAXDESKMOVE	
    			) {
    				// transition D
    				this.cdbState=4;
    				this.cdbTime = System.currentTimeMillis();
    				this.state.positionAndButtonsKnown=true;
    				this.state.buttons=1<<this.cdbClickButton;
    			} else {
    				// transition B
    				this.cdbState=1;
    				this.state.positionAndButtonsKnown=false;
    				this.state.buttons=0;				
    			}
    		} else {
    			// sometimes we get two penup signals. don't know why. 
    			// just leave it with same state.
    		}
    		this.changed = true;
        } else {
            this.state.positionAndButtonsKnown=false;
            this.state.buttons=0;
            this.changed = true;
        }
	}

	public void penUp() {
		// might want to filter out psurious penups...
		actuallyPenUp();
	}

	
	public void actuallySample(int x, int y) {

	    this.setStateDESKcoordsFromINPUTcoords(millimetresPerPenUnit*x, millimetresPerPenUnit*y);
        this.changed = true;
        
        if(this.clickDragButtonEmulation) { 
    		if(this.cdbState==1) {
    			// transition A
    			this.cdbState=2;
    			this.cdbTime = System.currentTimeMillis();
    			this.cdbDESKox = this.state.DESKx;
    			this.cdbDESKoy = this.state.DESKy;
    			this.state.positionAndButtonsKnown=true;
    			this.state.buttons = this.cdbNotClickOrDragButtonsBitMap;
    		} else if(this.cdbState==4) {
    			// transition G
    			this.cdbState=6;
    			this.state.positionAndButtonsKnown=true;
    			this.state.buttons = 1<<this.cdbDragButton;
    		} else {
    			// no transition
    			// pen was in a state and is still in a state
    			// don't alter buttons or posnandbuttonsknown.
    		}
        } else {
            this.state.positionAndButtonsKnown=true;
            this.state.buttons = 1<<this.noncdbPenDownButton;
        }
		
	}
	
	
	public void sample(int x, int y, int force) {
		if(Math.abs(x-this.penOldX)>=this.divJump || Math.abs(y-this.penOldY)>=this.divJump || !this.state.positionAndButtonsKnown) {
			this.penOldX = x;
			this.penOldY = y;
			actuallySample(x,y);
		} else {
			// no need
		}
	}

}
