/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.input;

import java.awt.geom.Rectangle2D;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;

import t3.hrd.state.Cursor;
import t3.hrd.state.JOGLHelper;
import t3.hrd.util.FPSTimer;
import Jama.Matrix;

/**
 * Experimental. No documentation yet.
 * 
 * This enables T3 to connect via a pipe to an executable file that performs computer
 * vision to segment arm outlines from an overhead camera.
 * 
 * @author pjt40
 *
 */

// protocol:
//mode = 0: waiting for size
// mode = 1: waiting for rest of packet

// protocol: 
//      each coordinate repr by 2 bytes, so point is 4 bytes, rect is 16 bytes

// frame is:
//      2 bytes: size as a number of bytes in entire frame
//      2 bytes: num polygons in this frame
//      4 bytes: zeros
//      
//      for each poly:
//          2 bytes: numPoints
//          8 bytes: brect
//
//      the rest:
//          byte buffer containing points, 2 bytes for each coord, in order





public class ExternalProgramShapeInput extends ShapeInputDevice {
    private final Process process;
    private boolean closing = false;
    private final Thread processErrMonitor;
    private final DataInputStream processBuffdOutStreamWrappered;
    private final CircularBufferedInputStream processBufferedOutStream;
    private static final Logger logger = Logger.getLogger("t3.hrd.input");
    
    private int curMode;
    private int curFrameSize;
    private final Matrix mSHAPEDATAtoDESK;
    private final Matrix[] tempSHAPEDATAboundingBox;
    private final FPSTimer fpst = new FPSTimer("Arm shadow input rate = ","",1000.0, 0.2, 2000, 8000);
	
	public ExternalProgramShapeInput(int clientId, Properties p, String prefix) throws InputDeviceException {
		super(clientId);
		try {
            
            String programInclFullPath = JOGLHelper.props_getString(p,prefix+"programInclFullPath"); 
                // eg "W:\\practical\\opencvstuff\\t3handseg\\Debug\\pjttest2.exe";
            String spaceSeperatedArgs = JOGLHelper.props_getString(p,prefix+"spaceSeparatedProgramArgs");  
                // eg "1 2 3";
            File dir = new File(JOGLHelper.props_getString(p,prefix+"fullPathToStartInDirectory"));
            
            Matrix[] mDESKpoints = new Matrix[4];
            Matrix[] mSHAPEDATApoints = new Matrix[4];
            for(int k=0; k<4; k++) {
                    mDESKpoints[k] =
                        JOGLHelper.getMFromD(
                            JOGLHelper.props_getDouble(p,prefix+"DESKpoint"+k+"x"),
                            JOGLHelper.props_getDouble(p,prefix+"DESKpoint"+k+"y"),
                            1
                        );   
                    mSHAPEDATApoints[k] =
                        JOGLHelper.getMFromD(
                            JOGLHelper.props_getDouble(p,prefix+"SHAPEDATApoint"+k+"x"),
                            JOGLHelper.props_getDouble(p,prefix+"SHAPEDATApoint"+k+"y"),
                            1
                        ); 
            }
            this.mSHAPEDATAtoDESK = JOGLHelper.getPlaneToPlaneHomogMatrixFromFour2hmPointCorrespondances(mSHAPEDATApoints,mDESKpoints);
            
            List<String> commandAndArgs = new LinkedList<String>();
            commandAndArgs.add(spaceSeperatedArgs);
            commandAndArgs.add(0,programInclFullPath);
            
            ProcessBuilder pb = new ProcessBuilder(commandAndArgs);
            pb.directory(dir);
            pb.redirectErrorStream(false);
            this.process = pb.start();
                        
            this.processErrMonitor = new Thread("Process error monitor for "+process.hashCode()) {
                public void run() {
                    try {
                        final BufferedReader errStream = 
                            new BufferedReader(
                                new InputStreamReader(
                                     ExternalProgramShapeInput.this.process.getErrorStream()
                               )
                        );
                        while(true) {
                            String s = errStream.readLine();
                            if(s==null) {
                                break;
                            } else {
                                logger.info("Process "+ExternalProgramShapeInput.this.process.hashCode()+" says: "+s);
                            }
                        }
                        int exitValue = ExternalProgramShapeInput.this.process.waitFor();
                        if(!closing && exitValue!=0) {
                            throw new RuntimeException("Process "+ExternalProgramShapeInput.this.process.hashCode()+" exited with code "+ExternalProgramShapeInput.this.process.exitValue());
                        } else {
                            logger.info("Process "+ExternalProgramShapeInput.this.process.hashCode()+" exited with code "+ExternalProgramShapeInput.this.process.exitValue());
                        }
                    } catch(IOException e) {
                        throw new RuntimeException(e);
                    } catch(InterruptedException e) {
                        throw new RuntimeException(e);
                    }
                }
            };
            
            this.processErrMonitor.start();
            this.processBufferedOutStream = new CircularBufferedInputStream(
                    process.getInputStream(),
                    24000,
                    200
                    );
            this.processBuffdOutStreamWrappered = new DataInputStream(this.processBufferedOutStream);
            this.curMode = 0;
            this.curFrameSize = 0;
            this.tempSHAPEDATAboundingBox = new Matrix[4];
            for(int i=0; i<4; i++) {
                this.tempSHAPEDATAboundingBox[i] = JOGLHelper.getMFromD(0.0,0.0,1.0);
            }
		    
		} catch(Exception e) {
			throw new InputDeviceException(e);
		}
	}
	
    
    
	@Override
	public boolean updateState() throws InputDeviceException {
        
        try {
            boolean readAtLeastOneFrame = false;
            while(true) {
                
                long t1=System.currentTimeMillis(); 
                    
                this.processBufferedOutStream.ifMinAutoReadSizeAvailableThenReadAsMuchAsPosIntoBufferWithoutBlocking();            

                long t2=System.currentTimeMillis();
                
                int nBytesAvailableInclBuffered = processBuffdOutStreamWrappered.available();
                
                if(curMode ==0 && nBytesAvailableInclBuffered>=2) {
                    curFrameSize=(int)processBuffdOutStreamWrappered.readShort();
                    //System.out.println("Received size="+size);
                    curMode=1;
                                        
                } else if(curMode==1 && nBytesAvailableInclBuffered>=2*curFrameSize-2) {
                    // we'll guess and say that there's another complete packet ready
                    //System.out.println("Skipping.");
                    int nBytesSkipped = processBuffdOutStreamWrappered.skipBytes(curFrameSize-2);
                    assert nBytesSkipped == curFrameSize-2;
                    curMode=0;
                    
                } else if(curMode==1 && nBytesAvailableInclBuffered>=curFrameSize-2) { 
                    
                    int nPolygonsInFrame = (int) processBuffdOutStreamWrappered.readShort();
                    //System.out.println("Received npolys="+nPolygonsInFrame);
                    if(processBuffdOutStreamWrappered.readInt()!=0) {
                        throw new RuntimeException("Bad stream");
                    }
                    
                    short[] nPointsInPolygons = new short[nPolygonsInFrame];
                    Rectangle2D.Double[] rDESKboundingRects = new Rectangle2D.Double[nPolygonsInFrame];
                    
                    // compute rect stuff
                    // todo need a method that takes Matrix[] and computes the brect. 
                    // should be vv easy
                    // and should be able to use it in Tile and in scarottrarectangle
                                        
                    for(int curPolyIndex=0; curPolyIndex<nPolygonsInFrame; curPolyIndex++) {
                        nPointsInPolygons[curPolyIndex] = processBuffdOutStreamWrappered.readShort();
                        
                        //System.out.println("Points in poly = "+nPointsInPolygons[curPolyIndex]);
                        
                        short rx = processBuffdOutStreamWrappered.readShort();
                        short ry = processBuffdOutStreamWrappered.readShort();
                        short rw = processBuffdOutStreamWrappered.readShort();
                        short rh = processBuffdOutStreamWrappered.readShort();
                        
                        // todo compute desk rect and store it
                        tempSHAPEDATAboundingBox[0].set(0,0,(double)rx);
                        tempSHAPEDATAboundingBox[0].set(1,0,(double)ry);
                        assert tempSHAPEDATAboundingBox[0].get(2,0)==1.0;
                        tempSHAPEDATAboundingBox[1].set(0,0,(double)rx);
                        tempSHAPEDATAboundingBox[1].set(1,0,(double)ry+rh);
                        assert tempSHAPEDATAboundingBox[1].get(2,0)==1.0;
                        tempSHAPEDATAboundingBox[2].set(0,0,(double)rx+rw);
                        tempSHAPEDATAboundingBox[2].set(1,0,(double)ry+rh);
                        assert tempSHAPEDATAboundingBox[2].get(2,0)==1.0;
                        tempSHAPEDATAboundingBox[3].set(0,0,(double)rx+rw);
                        tempSHAPEDATAboundingBox[3].set(1,0,(double)ry);
                        assert tempSHAPEDATAboundingBox[3].get(2,0)==1.0;
                        
                        rDESKboundingRects[curPolyIndex] = 
                            JOGLHelper.getBoundingRectFrom2hmPoly(
                               tempSHAPEDATAboundingBox,
                               mSHAPEDATAtoDESK
                            );
                    }
                    
                    int nBytesInBuffer = curFrameSize-8-10*nPolygonsInFrame;
                    //System.out.println("Byte buffer should be "+nBytesInBuffer);
                    byte[] byteBuffer = new byte[nBytesInBuffer];
                    int nBytesRead = processBuffdOutStreamWrappered.read(byteBuffer,0,nBytesInBuffer);
                    assert nBytesRead==nBytesInBuffer;
                    
                    // now construct new state object
                    this.state.polygonsForShapeCursor = new Cursor.CursorShapePolygons(
                            byteBuffer,
                            this.mSHAPEDATAtoDESK,
                            nPointsInPolygons,
                            rDESKboundingRects
                    );
                    
                    // reset the mode etc
                    curMode=0;
                    readAtLeastOneFrame = true;
                    
                } else {
                    
                    // not enough data available to do anything
                    // we don't want to block waiting so we'll just ignore
                    break;
                }   
                
                long t3=System.currentTimeMillis();
                
                //System.out.println("Timing: "+(t2-t1)+" "+(t3-t2));
            }
            
            if(readAtLeastOneFrame) {
                fpst.oneFrame();
            }
            
            return readAtLeastOneFrame;
        } catch(IOException e) {
            throw new InputDeviceException(e);
        }
    }
        

	@Override
	public void close() throws InputDeviceException {
        this.closing=true;
		if(!processHasTerminated(this.process)) {
		    this.process.destroy();
        } else {
            // no need to kill the process as it isn't running any more
        }
	}
    
    private static boolean processHasTerminated(Process p) {
        try {
            int i = p.exitValue();
            return true;
        } catch(IllegalThreadStateException e) {
            return false;
        }        
    }
	

}
