/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.remotehrd.server;


import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.Serializable;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import t3.hrd.input.KeyboardInput;
import t3.hrd.input.PointInputDeviceState;
import t3.hrd.input.ShapeInputDeviceState;
import t3.hrd.state.Cursor;
import t3.hrd.state.Link;
import t3.hrd.state.OrderedElement;
import t3.hrd.state.StateManager;
import t3.hrd.state.Tile;
import t3.hrd.state.UnwarpedRect;
import t3.remotehrd.protocol.OpCreateCursor;
import t3.remotehrd.protocol.OpCreateLink;
import t3.remotehrd.protocol.OpCreateTile;
import t3.remotehrd.protocol.OpCreateUnwarpedRect;
import t3.remotehrd.protocol.OpCursorPos;
import t3.remotehrd.protocol.OpReorderTilesAndLinks;
import t3.remotehrd.protocol.OpSetCursorOptions;
import t3.remotehrd.protocol.OpSetTileAff;
import t3.remotehrd.protocol.OpSetTileVisibility;
import t3.remotehrd.protocol.OpUpdateTileContents;


/* 

 */

/**
 * This class represents a RemoteHRD server. It has a HRD StateManager, and changes to
 * the state are broadcast to any connected RemoteHRDClients.
 * <p>
 * You can use its StateManager as usual but with some restrictions: you must synchronize on it before
 * you use it, or its Tiles, Cursors or Links; and you must call endOfBurst to ensure that
 * changes get sent on to the clients.
 * <p>
 * We have one listener thread that listens for new connections. It does:
 * <ul>
 * 		<li>while true
 * 		<ul>
 * 			<li>block waiting for new connection
 * 			<li>call callback
 * 			<li>create new client object and add to clients set
 * 			<li>start client's receiver thread
 * 		</ul>
 * </ul>
 * <p> 
 * We have one thread per client that receives and processes messages from that client. * 
 * It does:
 * <ul>
 * 		<li>lock statemanager to prevent anyone updating tiles or cursors or sending messages to all clients
 * 		<li>send entire state to client
 * 		<li>add client to clientslinkedin set
 * 		<li>unlock
 * 		<li>while true
 * 			<ul>
 * 				<li>block waiting for new message from client
 * 				<li>process that message
 * 			</ul>
 * 		<li>and if the client leaves or has an error:
 * 			<ul>
 * 			<li>notify the clientsmanager, which removes the client from the clients set and the linkedinclients set
 * 			<li>close the socket
 * 			<li>call the callback
 * 			</ul>
 * 	</ul>
 * 
 * Processing is therefore multithreaded and uses locks on StateManager. 
 * You need to be careful with this. If you use the StateManager, or one of its
 * Tiles, Links or Cursors then you will need to synchronize on StateManager first.
 * <p>
 * To send a message to all clients you call the manager's addToSendQueue method. 
 * If you perform operations on the StateManager or one if its Tiles, Links or Cursors then
 * a message will automatically be generated and the addToSendQueue method invoked.
 * <p>
 * When you call addToSendQueue, the message is not sent immediately; instead it is
 * added to a queue and might not be sent until you call endOfBurst. 
 * Messages within a burst might be amalgamated to save processing time and network bandwidth. 
 * <p>
 * There are two modes for sending:
 * <ul>
 * <li>In alwaysServerPush mode we send messages to the client as soon as you call hindEndOfBurst.
 * <li>In the other mode, we buffer messages until the client sends a ready message.
 * At that point we send whatever is waiting in the queues, regardless of whether you
 * have called endOfBurst. Any calls to endOfBurst will be ignored - 
 * we wait for the ready message. If the client we receieve a ready but there is nothing
 * waiting in the queues then we send messages when you next call endOfBurst.
 * </ul>
 * 
 * @author pjt40
 *
 */

public class RemoteHRDServer {
	
	// clients is clients connected
	// linked in is clients who're connected and who also receive state updates. some
	// clients will be connected but won't yet have received the entire state so they
	// aren't yet linked in.
	private Map<Integer,Client> allClientsClientIdToClients = new HashMap<Integer,Client>();
	public final Map<Integer,Client> allClientsClientIdToClientsImmutable = Collections.unmodifiableMap(allClientsClientIdToClients);
	private Set<Client> clientsLinkedIn = new HashSet<Client>();
	private StateListenerForServer stateListener;
    
	/*private Object addingRemovingIteratingClientsOrLinkedIn = new Object();
     we used ot have a lock to protect this but it deadlocked because this lock
     and statemanager lock were acquired in different orders in place, so we now
     just use statemanager to protect iterating, adding and removing clients or clients 
     linked in.*/
    
	private final ServerSocket serverSocket;
	private final ServerSocket inputSourceServerSocket;
	
	public final RemoteHRDServerCallBacks callBacks;

	private static final Logger logger = Logger.getLogger("t3.remotehrd.server");
	
	/**
	 * See notes above.
	 */
	public final StateManager stateManager;
	final boolean alwaysServerPush;
	
	private Thread listenThread = new Thread("RemoteHRDServer Listener Thread") {
		public void run() {
			logger.info("Listening on "+serverSocket+" and ready for clients to connect...");
			try {
				while(true) {
					Socket newSocket = RemoteHRDServer.this.serverSocket.accept();
					newSocket.setTcpNoDelay(true);
                    newSocket.setSendBufferSize(1024*1024*8);
                    System.out.println("Send buffer size is "+newSocket.getSendBufferSize());
					Client newClient;
					try {
						newClient = new Client(newSocket, RemoteHRDServer.this, alwaysServerPush);
					} catch(IOException e) {
						// couldn't open connection to client.
						// no reason to stop server.
						e.printStackTrace();
						continue;
					}
					RemoteHRDServer.this.callBacks.remoteServerCallBack_clientJoined(newClient.clientId);
					synchronized(stateManager) {   // used to be seperate lock
						if(allClientsClientIdToClients.containsKey(newClient.clientId)) {
							throw new RuntimeException("Client already exists with id: "+newClient.clientId);
						}
						allClientsClientIdToClients.put(newClient.clientId,newClient);
					}
					newClient.startReceiverAndProcessorThread();
				}
			} catch(Throwable e) {
				//includes exceptions and assertionerrors
				RemoteHRDServer.this.stateManager.fatalError(e);
			}
		}
	};
	
	
	private Thread inputSourceListenThread = new Thread("RemoteHRDServer Inputsource Listener Thread") {
		public void run() {
			logger.info("Listening on "+inputSourceServerSocket+" and ready for input sources to connect...");
			try {
				while(true) {
					Socket newSocket = RemoteHRDServer.this.inputSourceServerSocket.accept();
					newSocket.setTcpNoDelay(true);
					InputSource inputSource;
					try {
						inputSource = new InputSource(newSocket, RemoteHRDServer.this);
					} catch(IOException e) {
						// couldn't open connection to inputSource.
						// no reason to stop server.
						e.printStackTrace();
						continue;
					}
					inputSource.startReceiverAndProcessorThread();
				}
			} catch(Throwable e) {
				//includes exceptions and assertionerrors
				RemoteHRDServer.this.stateManager.fatalError(e);
			}
		}
	};
	
	/**
	 * Creates a RemoteHRDServer and starts the listening thread. This constructor
	 * returns immediately.
	 * 
	 * @param serverSocket
	 * @param callBacks
	 * @param alwaysServerPush
	 */
	public RemoteHRDServer(ServerSocket serverSocket, ServerSocket inputSourceServerSocket, RemoteHRDServerCallBacks callBacks, boolean alwaysServerPush) {
		// starts the listener thread!
		this.serverSocket = serverSocket;
		this.inputSourceServerSocket = inputSourceServerSocket;
		this.callBacks = callBacks;
		this.alwaysServerPush = alwaysServerPush;
		this.stateListener = new StateListenerForServer(this);
		this.stateManager = new StateManager(this.stateListener, false, false, 0,0);
		this.listenThread.start();		
		this.inputSourceListenThread.start();
	}
	
	/**
	 * Add a message to the send queue. The message might not be sent until you call
	 * endOfBurst(). See notes above for more details. This method is thead-safe; it
	 * can be called from any thread.
	 * @param msg
	 */
	public void addToSendQueue(Serializable msg) {
		// called in any thread
        synchronized(stateManager) {   // used to be seperate lock
			for(Client c:clientsLinkedIn) {
				c.addToSendQueue(msg);
			}
		}
	}
	
	/**
	 * Indicates that queued messages can be sent through to the client. This method is thead-safe; it
	 * can be called from any thread.
	 */
	public void endOfUpdateFrame() {
		// called in any thread
        synchronized(stateManager) {   // used to be seperate lock
			for(Client c:clientsLinkedIn) {
				c.endOfUpdateFrame();
			}
		}
	}
	
	
	void clientHasStopped(Client c) {
		// called in any thread by Client.close()
        synchronized(stateManager) {   // used to be seperate lock
			this.clientsLinkedIn.remove(c);
			this.allClientsClientIdToClients.remove(c.clientId);
		}
	}
	
	
	void receivedMessageFromClientOrInputSource(Object msg) {
		// called in client's thread
		if(msg instanceof PointInputDeviceState) {
			this.callBacks.remoteServerCallBack_pointInputDeviceStateChanged((PointInputDeviceState)msg);
		} else if(msg instanceof KeyboardInput) {
			this.callBacks.remoteServerCallBack_keyboardInput((KeyboardInput)msg);
		} else if(msg instanceof ShapeInputDeviceState) {
			this.callBacks.remoteServerCallBack_shapeInputDeviceStateChanged((ShapeInputDeviceState)msg);
		} else {
			callBacks.remoteServerCallBack_receivedOtherMessageFromClientOrInputSource(msg);
		}
	}
	
	
	void sendEntireStateToClientAndThenLinkIn(Client client) {
        synchronized(stateManager) {   // used to be seperate lock
			// we synchronize while we send the entire state to the client
			// because we don't want anyone else to send messages to all linked in
			// clients between finishing the call to sendEntireStateToClient
			// and linking in the client
			
			synchronized(stateManager) {
				int[] order = new int[stateManager.tilesAndLinksInOrderReadOnly.size()];
				int i = 0;
				for(OrderedElement el: stateManager.tilesAndLinksInOrderReadOnly) {
					order[i++] = el.elementId;
					if(el instanceof Tile) {
						Tile tile = (Tile)el;
						client.addToSendQueue(
							new OpCreateTile(tile.elementId, tile.tileWidth, tile.tileHeight, tile.flags)
						);
						BufferedImage tileImage = tile.getImageForReadingOnlyPaintIfNecessary();
						client.addToSendQueue(
							new OpUpdateTileContents(
								tile.elementId,
								0,0,
								tileImage,
								this.callBacks.remoteServerCallBack_decideShouldCompressInitial(tileImage)
                                )
							);
						for(UnwarpedRect unwarpedRect: tile.unwarpedRectsLinesReadOnly) {
							client.addToSendQueue(
								new OpCreateUnwarpedRect(tile.elementId, unwarpedRect)
							);
						}
						for(UnwarpedRect unwarpedRect: tile.unwarpedRectsWordsReadOnly) {
							client.addToSendQueue(
								new OpCreateUnwarpedRect(tile.elementId, unwarpedRect)
							);
						}
						client.addToSendQueue(
							new OpSetTileAff(
								tile.elementId, 
								tile.getDESKcentreX(), 
								tile.getDESKcentreY(), 
								tile.getDESKwidth(), 
								tile.getDESKheight(), 
								tile.getTILEtoDESKrotationClockwise()
							)
						);
						client.addToSendQueue(
							new OpSetTileVisibility(tile.elementId, tile.isVisible())
						);
					} else {
						assert el instanceof Link;
						Link link = (Link)el;
						client.addToSendQueue(
							new OpCreateLink(
								link.elementId,
								link.displayType,
								link.color,
								link.getTstandardRectToDESKrectA().getMatrix2dHomogReadOnly(),
								link.getTstandardRectToDESKrectB().getMatrix2dHomogReadOnly() 
							)
						);
							
					}
					
					
				}		
				client.addToSendQueue(
					new OpReorderTilesAndLinks(order)
					);
				
				for(Cursor cursor: stateManager.cursorsReadOnly) {
					int cursorId = cursor.getCursorId();
					client.addToSendQueue(new OpCreateCursor(cursorId));
					client.addToSendQueue(new OpSetCursorOptions(cursorId, cursor.getColor(), cursor.getDisplayType()));
					if(cursor.getDisplayType()==cursor.DISPLAYTYPE_TRAIL && cursor.getMostRecentPosition().visible) {
						client.addToSendQueue(new OpCursorPos(cursorId, cursor.getMostRecentPosition().mDESKpoint.get(0,0), cursor.getMostRecentPosition().mDESKpoint.get(1,0), cursor.getMostRecentPosition().visible));
					}
				}
			}	
				
			client.endOfUpdateFrame();
			this.clientsLinkedIn.add(client);
		}
	}
}
