/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.remotehrd.server;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.concurrent.LinkedBlockingQueue;

import t3.remotehrd.protocol.BurstTerminator;
import t3.remotehrd.protocol.OpCreateTile;

public class NonBlockingObjectSender {
    private final ObjectOutputStream obos;
    
    private final LinkedBlockingQueue<Serializable> queue;
    private final OnThrowable ot;
    private final boolean nonblocking;
    
    public NonBlockingObjectSender(boolean nonblocking, ObjectOutputStream obos, OnThrowable ot) {
        this.obos = obos;
        this.nonblocking = nonblocking;
        this.ot = ot;
        this.queue = new LinkedBlockingQueue<Serializable>(1000);
        if(nonblocking) {
            this.senderThread.setPriority(Thread.MAX_PRIORITY);
            this.senderThread.start();
        } else {
            // don't!
        }
    }
    
    public void sendAndFlushIfBTNonBlocking(Serializable s) {
        if(nonblocking) {
            this.queue.add(s);
        } else {
            long t1 = System.currentTimeMillis();
            actuallySynchronouslySendAndFlushIfBT(s);
            System.out.println("Sending took "+(System.currentTimeMillis()-t1));
        }
    }

    private final Thread senderThread = new Thread() {
      public void run() {
          try {
              while(true) {
                  Serializable s = NonBlockingObjectSender.this.queue.take();
                  actuallySynchronouslySendAndFlushIfBT(s);
              }
          } catch(Throwable t) {
              ot.hasThrown(t);
          }
      }
    };
    
    private final void actuallySynchronouslySendAndFlushIfBT(Serializable s) {
        try {
            if(s instanceof BurstTerminator) {
                // have to reset BEFORE burst terminator sent
                this.obos.reset();
            } else {
                // do nothing
            }
            System.out.println("Writing "+s+" num in queue="+queue.size());
            this.obos.writeObject(s);
            if(s instanceof BurstTerminator) {
                obos.flush();
            }
        } catch(Throwable t) {
            ot.hasThrown(t);
        }
    }
    
    
    public static interface OnThrowable {
        public void hasThrown(Throwable t);
    }
}
