/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.remotehrd.server;

import java.io.EOFException;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.net.Socket;
import java.net.SocketException;
import java.util.logging.Level;
import java.util.logging.Logger;


/**
 * Represents, at the server end, an InputSource connected to the server.
 * @author pjt40
 *
 */
public class InputSource {
	private ObjectInputStream ois;
	private Socket socket;
	private RemoteHRDServer server;

	private static final Logger logger = Logger.getLogger("t3.remotehrd.server");
	
	InputSource(Socket s, RemoteHRDServer server) throws IOException{
		this.server = server;
		this.socket = s;
		this.ois = new ObjectInputStream( this.socket.getInputStream());		
	}
	
	private Thread receiverAndProcessorThread = new Thread("Input source's receiver and processor thread") {
		public void run() {
			logger.info("Connected to input source "+InputSource.this.socket+" as "+InputSource.this);
			try {
				while(true) {
					Object msg = ois.readObject();
					logger.fine("Received message from input source: "+msg);					
					InputSource.this.server.receivedMessageFromClientOrInputSource(msg);
				}
			} catch(EOFException e) {
				// client left
				InputSource.this.close();
			} catch(Throwable e) {
				//includes exceptions and assertionerrors
				InputSource.this.errorWithThisInputSource(e);
			}
		}
	};
	
	
	/**
	 * Reports a specified exception to do with this client, and closes
	 * the client without stopping the server. This method is thread-safe; 
	 * it can be called from any thread.
	 * @param e
	 */
	public void errorWithThisInputSource(Throwable e) {
		// indicates an error with this InputSource, not with whole server
		// should be callable from any thread
		if(
			(e instanceof SocketException)
			&& e.getMessage().equals("Connection reset")
		) {
			if(logger!=null) {
				logger.log(Level.INFO,"Inputsource terminated connection "+this);
			}
		} else {
			if(logger!=null) {
				logger.log(Level.SEVERE,"Caught exception and about to close inputsource ",e);
			}
			e.printStackTrace();
		}
		this.close();
	}
	
	/**
	 * Ensures that no more messages are sent to this client and closes the
	 * client's socket. This method can be called more than once with no ill
	 * effects. This method is thread-safe; it can be called from any thread.
	 */
	public synchronized void close() {
		try {
			if(!this.socket.isClosed()) {
				this.socket.close();
			}
		} catch(Throwable e) {
			this.server.stateManager.fatalError(e);
		}
	}
	

	void startReceiverAndProcessorThread() {
		this.receiverAndProcessorThread.start();
	}
}
