/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.util;


import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.geom.Area;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStreamReader;
import java.util.LinkedList;
import java.util.Properties;

import t3.hrd.input.PointInputDevice;
import t3.hrd.renderer.BlendOptions;
import t3.hrd.renderer.ProjectorConfig;
import t3.hrd.state.JOGLHelper;


/**
 * Utility which loads a properties file, allows you to calibrate the display, 
 * test the alignment and save it back to the same properties file.
 * <p>
 * The properties file must be set up correctly before hand but with any values in
 * the DESKcalibPoints fields. Calibration takes place using pointInputDevice 0 in the
 * properties file.
 * <p>
 * The first argument should be the name of the properties file.
 * 
 * @author pjt40
 *
 */
public class RunCalibrate {

	private static LinkedList<ProjectorConfig> projectorConfigs;
	private static LinkedList<PointInputDevice> pids;
    private static Area aDESKvisibleAreaMask;
	private static String fileName;
	
	private static void testInput(int s) throws Throwable {
		System.out.println("Enter the PID index of the input you wish to test:");
		String ss= new BufferedReader(new InputStreamReader(System.in)).readLine();
		try {
			int i = Integer.parseInt(ss);
			PointInputDevice pid = pids.get(i);
			System.out.println("Testing point input device "+pid);
			System.out.println("Will run at approx 30Hz for "+s+"s...");
			Thread.sleep(1000);
			pid.updateState();
			System.out.println("Current: "+pid.state);
			long t = System.currentTimeMillis()+s*1000;
			while(System.currentTimeMillis()<t) {
				Thread.sleep(33);
				if(pid.updateState()) {
					System.out.println("Updated: "+pid.state);
				} else {
					// no change
				}
			}
		} catch(NumberFormatException e) {
			System.out.println("Bad input "+ss);
		} catch(IndexOutOfBoundsException e) {
			System.out.println("No such pid with pid index "+ss);
		}
	}
	
	private static void showMenu() {
		GraphicsDevice[] ds = GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices();
		System.out.println("=========================================");
		System.out.println("");
		System.out.println("Current configuration was originally loaded from file "+fileName);
		System.out.println("");
		System.out.println("Contains "+projectorConfigs.size()+" projectors, using graphics devices:");
		for(ProjectorConfig projectorConfig: projectorConfigs ) {
			System.out.println(
					"    GraphicsDevice Index: "
					+projectorConfig.graphicsDeviceIndex
					+"   GraphicsDevice ID String: "
					+ds[projectorConfig.graphicsDeviceIndex].getIDstring());
		} 
		System.out.println("");		
		System.out.println("Contains "+pids.size()+" PointInputDevices:");
		for(PointInputDevice pid: pids) {
			System.out.println("    (PID index "+pids.indexOf(pid)+") "+pid.toString());
		}
		System.out.println("");
		System.out.println("We will calibrate projectors using the first PointInputDevice "+pids.getFirst().toString());
		System.out.println("");
	}
	
	public static void main(String[] args) {

		try {	
			
			fileName = args[0];
			
			System.out.println("===================");
			System.out.println("Calibration utility");
			System.out.println("===================");
			System.out.println("");
			System.out.println("Use this to calibrate once you have created a configuration file.");
			System.out.println("");
					
			Properties properties = new Properties();
			properties.load(new FileInputStream(new File(args[0])));

			System.out.println("Loading projector configurations from file "+args[0]);
			projectorConfigs = LoadConfigsFromProperties.loadProjectorConfigsFromProperties(properties);
            
            BlendOptions boOverlapped = new BlendOptions();
            boOverlapped.blackOut = BlendOptions.BLACKOUT_RECTONLY;
            boOverlapped.blend = false;
            boOverlapped.aDESKvisibleAreaMaskOrNull = null;
            
            BlendOptions boBlacked = new BlendOptions();
            boBlacked.blackOut = BlendOptions.BLACKOUT_ALLOVERLAP;
            boBlacked.blend = false;
            boBlacked.aDESKvisibleAreaMaskOrNull = LoadConfigsFromProperties.loadDESKvisibleAreaMaskFromProperties(properties); 
            
            BlendOptions boBlended = new BlendOptions();
            boBlended.blackOut = BlendOptions.BLACKOUT_RECTONLY;
            boBlended.blend = true;
            boBlended.aDESKvisibleAreaMaskOrNull = LoadConfigsFromProperties.loadDESKvisibleAreaMaskFromProperties(properties); 
            
			System.out.println("Loading point input device configurations from file "+args[0]);
			pids = LoadConfigsFromProperties.loadPointInputDevicesFromProperties(0,true, properties);
			if(pids.size()==0) {
				throw new Exception("No point input devices defined in config file. You must define point input devices to calibrate with."); 
			}
			
			showMenu();
			
			while(true) {
				System.out.println("");
				System.out.println("Choose:");
                System.out.println("P=calibrate a projector");
                System.out.println("C=checkerboard, overlapped (F12)");
                System.out.println("D=checkerboard, blacked (F12)");
                System.out.println("E=checkerboard, blended (F12)");
                System.out.println("W=white, overlapped (F12)");
                System.out.println("X=white, blacked (F12)");;
                System.out.println("Y=white, blended (F12)");
                System.out.println("R=view with rect mask applied (F12)");
                System.out.println("G=grid for 60s");
                System.out.println("S=save");
                System.out.println("Q=quit");
                System.out.println("I=test input for 20s");
                System.out.println("J=test input for 120s");
                System.out.println("M=show menu again");
                System.out.println("A=calibrate all projectors, view, save, quit");
				String s= new BufferedReader(new InputStreamReader(System.in)).readLine();
				
				if(s.equalsIgnoreCase("m")) {
					showMenu();
				} else if(s.equalsIgnoreCase("p")) {
					System.out.println("Enter graphics device index of the projector you wish to calibrate:");
					String ss= new BufferedReader(new InputStreamReader(System.in)).readLine();
					try {
						int i = Integer.parseInt(ss);
						ProjectorConfig pc = LoadConfigsFromProperties.getProjectorConfigForGraphicsDeviceIndex(projectorConfigs,i);
						if(pc==null) {
							System.out.println("Projector does not exist in configuration file for graphics device index "+i);
						} else {
							System.out.println("Calibrating projector with graphics device index "+i+"...");
							System.out.println("You may need to change window focus manually.");
							Calibrate.calibrate(pc,pids.getFirst());
							System.out.println("Calibrated projector with graphics device index "+i);
						}
					} catch(NumberFormatException e) {
						System.out.println("Bad input "+ss);
					}
				} else if(s.equalsIgnoreCase("g")) {
					System.out.println("Viewing grid for 60s...");
					System.out.println("You may need to change window focus manually.");
					Calibrate.showGrid(projectorConfigs);
				} else if(s.equalsIgnoreCase("i")) {
					testInput(20);
				} else if(s.equalsIgnoreCase("j")) {
					testInput(120);
				} else if(s.equalsIgnoreCase("c")) {
					System.out.println("Viewing results of calibration - can hang or exit if bad calibration data...");
					System.out.println("You may need to change window focus manually. Press F12 while in window to exit.");
					Calibrate.showTestChecker(projectorConfigs,pids, false,boOverlapped);	
                } else if(s.equalsIgnoreCase("d")) {
                    System.out.println("Viewing results of calibration - can hang or exit if bad calibration data...");
                    System.out.println("You may need to change window focus manually. Press F12 while in window to exit.");
                    Calibrate.showTestChecker(projectorConfigs,pids, false,boBlacked);  
                } else if(s.equalsIgnoreCase("e")) {
                    System.out.println("Viewing results of calibration - can hang or exit if bad calibration data...");
                    System.out.println("You may need to change window focus manually. Press F12 while in window to exit.");
                    Calibrate.showTestChecker(projectorConfigs,pids, false,boBlended);         		
				} else if(s.equalsIgnoreCase("w")) {
					System.out.println("Viewing results of calibration - can hang or exit if bad calibration data...");
					System.out.println("You may need to change window focus manually. Press F12 while in window to exit.");
					Calibrate.showTestWhite(projectorConfigs,pids, false,boOverlapped);	               
                } else if(s.equalsIgnoreCase("x")) {
                    System.out.println("Viewing results of calibration - can hang or exit if bad calibration data...");
                    System.out.println("You may need to change window focus manually. Press F12 while in window to exit.");
                    Calibrate.showTestWhite(projectorConfigs,pids, false,boBlacked);             
                } else if(s.equalsIgnoreCase("y")) {
                    System.out.println("Viewing results of calibration - can hang or exit if bad calibration data...");
                    System.out.println("You may need to change window focus manually. Press F12 while in window to exit.");
                    Calibrate.showTestWhite(projectorConfigs,pids, false,boBlended); 
				} else if(s.equalsIgnoreCase("q")) {
					System.out.println("Goodbye.");
					System.exit(0);
				} else if(s.equalsIgnoreCase("a")) {
					for(ProjectorConfig projectorConfig: projectorConfigs ) {
						System.out.println("Calibrating projector. You may need to change window focus manually.");
						Calibrate.calibrate(projectorConfig,pids.getFirst());							
					}
					System.out.println("Viewing results of calibration - can hang or exit if bad calibration data...");
					System.out.println("You may need to change window focus manually. Press F12 while in window to exit.");
					Calibrate.showTestChecker(projectorConfigs,pids, false,boBlacked);	
					LoadConfigsFromProperties.saveProjectorConfigsToProperties(projectorConfigs, properties);
					JOGLHelper.storePropertiesSorted(properties, new FileOutputStream(args[0]),"");
					System.out.println("Saved into "+args[0]);
				} else if(s.equalsIgnoreCase("s")) {
					LoadConfigsFromProperties.saveProjectorConfigsToProperties(projectorConfigs, properties);
					JOGLHelper.storePropertiesSorted(properties, new FileOutputStream(args[0]),"");
					System.out.println("Saved into "+args[0]);
					
				} else {
					System.out.println("Wrong input "+s);
				}
			}
			
		} catch(Throwable e) {
			//includes exceptions and assertionerrors
			e.printStackTrace();
			System.exit(1);
		}
	}	
}
