/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.state;


import java.awt.Color;
import java.awt.geom.Rectangle2D;
import java.io.Serializable;
import java.util.LinkedList;
import java.util.List;
import java.util.logging.Logger;

import t3.hrd.renderer.Projector;
import Jama.Matrix;


/**
 * A Cursor represents a cursor on the display surface. Create new Cursors by calling the appropriate
 * method on the a StateManager object. 
 * <p>
 * Threading notes: this class is not thread-safe. You must use some kind of locking scheme
 * if you use it in a multithreaded environment.
 * 
 * @author pjt40
 *
 */
public class Cursor {

	private static final Logger logger = Logger.getLogger("t3.hrd.state");
	
	private CursorTrailPoint[] cursorTrailPoints;
	private int cursorTrailPointPosn;
	private CursorShapePolygons polygonsForShapeCursor;
	
	private int displayType;
	private Color color;
	private final int cursorId;
	
	private static final int NUM_TRAIL_POINTS = 100;
	public static final long TRAIL_PERIOD = 1000;
	public static final int DISPLAYTYPE_INVISIBLE = 0;
	public static final int DISPLAYTYPE_TRAIL = 1;
	public static final int DISPLAYTYPE_SHAPE = 2;
	
	private StateListener cursorsListener;
	boolean destroyed;

	/**
	 * @param stateListener
	 * @param cursorId
	 * @param color
	 * @param displayType
	 */
	Cursor(StateListener stateListener, int cursorId, Color color, int displayType)  {
		this.cursorsListener = stateListener;
		this.destroyed = false;
		this.cursorTrailPoints = new CursorTrailPoint[NUM_TRAIL_POINTS];
		this.color = color;
		this.cursorTrailPointPosn = 0;
		this.displayType = displayType;
		this.cursorId = cursorId;
	}
	
	public int getCursorId() { return this.cursorId; }
	public CursorTrailPoint getMostRecentPosition() { return cursorTrailPoints[cursorTrailPointPosn]; }
	public Color getColor() { return this.color; }
	public int getDisplayType() { return this.displayType; }
	public boolean isDestroyed() { return this.destroyed; }
	

	
	/**
	 * Sets the cursor position and visibility.
	 * @param DESKx
	 * @param DESKy
	 * @param visible
	 */
	public void opCursorPos(double DESKx, double DESKy, boolean visible){
		checkNotDestroyed();
		CursorTrailPoint newctp = new CursorTrailPoint(JOGLHelper.getMFromD(DESKx,DESKy,1.0), visible);
		cursorTrailPointPosn++;
		while(cursorTrailPointPosn>=cursorTrailPoints.length) {
			cursorTrailPointPosn-=cursorTrailPoints.length;
		}
		cursorTrailPoints[cursorTrailPointPosn] = newctp;	
		this.cursorsListener.callback_cursorPos(this, DESKx, DESKy,visible);
	}

	public void opCursorShape(CursorShapePolygons polygonsForShapeCursor){
		checkNotDestroyed();
		this.polygonsForShapeCursor = polygonsForShapeCursor;
		this.cursorsListener.callback_cursorShape(this,polygonsForShapeCursor);
	}
		
	/**
	 * Sets the cursor colour and display type.
	 * @param col
	 * @param displayType
	 */
	public void opSetCursorOptions(Color col, int displayType)  {
		checkNotDestroyed();
		this.color = col;
		this.displayType = displayType;		
		this.cursorsListener.callback_updatedCursorOptions(this);
	}	
	
	
	void destroyed() {
		this.destroyed = true;
	}
	
	private void checkNotDestroyed() {
		if(this.destroyed) {
			throw new IllegalStateException("Cursor is destroyed");
		}
	}
	
	
	
	
	public void oglDrawCursor(javax.media.opengl.GL gl, Projector p) {
		// draws cursor on all projectors
		
		if(this.displayType == this.DISPLAYTYPE_TRAIL) {
		
			long curTime = System.currentTimeMillis();
			
			int i = cursorTrailPointPosn;
			int count = 0;
			
			boolean first = true;
			float curLineWidth = 0.0f;
			short curStippleStyle = 0;
			Matrix mTOGLlastPoint = null;
			boolean lastVisible = false;
			
			gl.glDisable(gl.GL_TEXTURE_2D);
			JOGLHelper.oglColor(gl, this.color);
			gl.glEnable(gl.GL_LINE_STIPPLE);		
			
			while(count<cursorTrailPoints.length && cursorTrailPoints[i]!=null && curTime-cursorTrailPoints[i].time<=TRAIL_PERIOD) {
				double proportion = 1.0-(curTime-cursorTrailPoints[i].time) / (double)TRAIL_PERIOD;
				float lineWidth = 
					proportion>0.60 ? 3.0f :
					(proportion>0.45 ? 2.0f : 1.0f);
				short stippleStyle = 
					proportion>0.30 ? (short)0xffff :
					(proportion>0.15 ? (short)0x00ff : (short)0x0101);
				if(first || lineWidth!=curLineWidth) {
					// causes problems
					gl.glLineWidth(lineWidth);
					curLineWidth = lineWidth;
				} else {
					// do nothing			
				}
				if(first || stippleStyle != curStippleStyle) {
					gl.glLineStipple(1,(short)stippleStyle);
					curStippleStyle = stippleStyle;
				} else {
					// do nothing				
				}
				boolean thisVisible = this.cursorTrailPoints[i].visible;
				Matrix mTOGLthisPoint;
				if(thisVisible) {
					mTOGLthisPoint = p.transforms.mDESKtoTOGL.times(this.cursorTrailPoints[i].mDESKpoint);
				} else {
					mTOGLthisPoint = null;
				}
				if(thisVisible && lastVisible  && !first) {
					gl.glBegin(gl.GL_LINES);
					JOGLHelper.oglVertex4dFromM(gl,mTOGLlastPoint);
					JOGLHelper.oglVertex4dFromM(gl,mTOGLthisPoint);
					gl.glEnd();				
				} else {
					// can't draw line
				}
				mTOGLlastPoint = mTOGLthisPoint;
				lastVisible = thisVisible;
				first = false;
				i--;
				if(i<0) { i+= cursorTrailPoints.length; }
				count++;
			}
			gl.glDisable(gl.GL_LINE_STIPPLE);
			
		} else if (this.displayType == this.DISPLAYTYPE_SHAPE) {
            if(this.polygonsForShapeCursor!=null) {
    			
    			// todo!
    			// v easy, just transform points and draw polygon.
    			// we'll assume they're convex for now
    			
    			gl.glDisable(gl.GL_TEXTURE_2D);
    			JOGLHelper.oglColor(gl, this.color);
    			
    			Matrix mSHAPEDATAtoTOGL = p.transforms.mDESKtoTOGL.times(
    				this.polygonsForShapeCursor.mSHAPEDATAtoDESK);
    			
                List<Matrix> mSHAPEDATApoints = new LinkedList<Matrix>();
                int startPosOfCurPolygon = 0;
    			
                /*
                BufferedImage bi = new BufferedImage(640,480,BufferedImage.TYPE_INT_ARGB);
                Graphics2D g = bi.createGraphics();
                g.setColor( Color.BLUE);
                int opx=0, opy=0;
                for(int numPointsInCurPolygon: this.polygonsForShapeCursor.nPointsForPolygons) {
                    boolean first=true;
                    g.setColor( new Color((float)Math.random(),(float)Math.random(),(float)Math.random()));
                    for(int i=0; i<numPointsInCurPolygon; i++) {
                        byte upperX = this.polygonsForShapeCursor.byteBuffer[startPosOfCurPolygon+4*i];
                        byte lowerX = this.polygonsForShapeCursor.byteBuffer[startPosOfCurPolygon+4*i+1];
                        byte upperY = this.polygonsForShapeCursor.byteBuffer[startPosOfCurPolygon+4*i+2];
                        byte lowerY = this.polygonsForShapeCursor.byteBuffer[startPosOfCurPolygon+4*i+3];
                        int pointX = (int)((upperX << 8) | (lowerX & 0xff));
                        int pointY = (int)((upperY << 8) | (lowerY & 0xff));
                        if(!first) {
                            g.drawLine(opx,opy,pointX,pointY);
                        }
                        opx=pointX;
                        opy = pointY;
                        first=false;
                        
                    }
                    startPosOfCurPolygon += 4*numPointsInCurPolygon;
                }
                try {
                ImageIO.write(bi,"png",new File("handoutline.png"));
                } catch(Exception e) { throw new RuntimeException(e); }
                System.exit(0);
                */
                
                for(int numPointsInCurPolygon: this.polygonsForShapeCursor.nPointsForPolygons) {
                    mSHAPEDATApoints.clear();
                    for(int i=0; i<numPointsInCurPolygon; i++) {
                        byte upperX = this.polygonsForShapeCursor.byteBuffer[startPosOfCurPolygon+4*i];
                        byte lowerX = this.polygonsForShapeCursor.byteBuffer[startPosOfCurPolygon+4*i+1];
                        byte upperY = this.polygonsForShapeCursor.byteBuffer[startPosOfCurPolygon+4*i+2];
                        byte lowerY = this.polygonsForShapeCursor.byteBuffer[startPosOfCurPolygon+4*i+3];
                        int pointX = (int)((upperX << 8) | (lowerX & 0xff));
                        int pointY = (int)((upperY << 8) | (lowerY & 0xff));
                        mSHAPEDATApoints.add(
                            JOGLHelper.getMFromD(
                                (double)pointX,
                                (double)pointY,
                                1.0
                            )
                        );
                        
                    }
                    startPosOfCurPolygon += 4*numPointsInCurPolygon;
                    //JOGLHelper.oglColor(gl, new Color((float)Math.random(),(float)Math.random(),(float)Math.random()));
                    //gl.glBegin(GL.GL_POLYGON);
                    //JOGLHelper.oglVertex4dFromMs(gl,mSHAPEDATApoints,mSHAPEDATAtoTOGL);
                    //gl.glEnd();
                    JOGLHelper.oglTesselateAndDraw2hmPoly(gl, mSHAPEDATApoints,mSHAPEDATAtoTOGL);
    			}
            } else {
                // nothing to draw
            }
		} else {
			assert false;
		}
	}
	
	/**
	 * Data structure representing a point in time on the cursor's trail.
	 * @author pjt40
	 *
	 */
	public static class CursorTrailPoint {
		public CursorTrailPoint(Matrix mDESKpoint, boolean visible) {
			this.time = System.currentTimeMillis();
			this.visible = visible;
			this.mDESKpoint = mDESKpoint;
		}
		public final long time;
		public final boolean visible;
		public final Matrix mDESKpoint;
	}
	
	
	/**
     * Experimental.
	 * @author pjt40
	 *
	 */
	public static class CursorShapePolygons implements Serializable {
		// SHAPEDATA is 2d homog space that varies from packet to packet
		// makes things a lot easier
        // needs java 6 in order for rect2d.double to be serializable
		
		/**
         * 
         */
        private static final long serialVersionUID = 615968386876395000L;

        // this transformation can be null if there are no polygons.		
		public final Matrix mSHAPEDATAtoDESK;
        
        // contains the points in order, no gaps, 4 bytes per point consisting of:
        // upperx, lowerx, uppery, lowery
		public final byte[] byteBuffer;
		public final Rectangle2D.Double[] rDESKboundingBoxes;
		public final short[] nPointsForPolygons;
        public CursorShapePolygons(byte[] byteBuffer, Matrix mSHAPEDATAtoDESK, short[] nPointsForPolygons, Rectangle2D.Double[] rDESKboundingBoxes) {
            // TODO Auto-generated constructor stub
            this.byteBuffer = byteBuffer;
            this.mSHAPEDATAtoDESK = mSHAPEDATAtoDESK;
            this.nPointsForPolygons = nPointsForPolygons;
            this.rDESKboundingBoxes = rDESKboundingBoxes;
        }
		
	}
    /*
    public static class SerializableRectangle2DDouble implements Serializable {
        public final double x, y, w, h;
        public SerializableRectangle2DDouble(Rectangle2D.Double r) {
            this.x = r.x; this.y=r.y; this.w=r.width; this.h=r.height;
        }
    }*/
	
	
	public String toString() {
		return super.toString()+" cursorId="+this.cursorId;
	}
	
}

/*

case CURSOR_CROSS:
	{
		Vector3<float> p;
		int            i,j;
		int			   x,y;
		int            a,b;
		const float    S     = 0.02f; //width/height of the cursor in metres
		const DWORD    black = RGB(0,0,0);
		const DWORD    white = RGB(255,255,255);
		int            disp  = g_fovealDisplays[d.fovealDisplayNum]->serverDisplay;
				
		for( i=0; i<MAX_CURSORS; i++ )
		if( this->cursors[i].active )
		{
			j = cursors[i].buffer_pos;
			// Is the cursor hidden?
			if( !cursors[i].visible[j] ) continue;
			// Is the cursor on this display?
			if( disp!=cursors[i].display[j] ) continue;
			// Get the centre of the cursor
			p.set( cursors[i].desk_x[j], cursors[i].desk_y[j], 1 );
			d.deskToFramebuffer( p );
			
			x = int(p[0]/p[2]);
			y = int(p[1]/p[2]);

			
			
			// Get the top left of the cursor
			p.set( cursors[i].desk_x[j] - S/2.0f, cursors[i].desk_y[j] - S/2.0f, 1 );
			d.deskToFramebuffer( p );
			// Calculate a and b which give the cursor an appropriate scale, according
			// to how big it should look on the desk. I've put min and max values on a and b.
			a = min( max( abs(int( (p[0]/p[2]-x)*0.3 )), 1), 20);
			b = min( max( abs(int(  p[0]/p[2]-x      )), 3), 60);			
			// Draw the black lines
			GraphxLine(hdc, x-1,y-b,x-1,y-a, black);
			GraphxLine(hdc, x+1,y-b,x+1,y-a, black);
			GraphxLine(hdc, x-1,y+b,x-1,y+a, black);
			GraphxLine(hdc, x+1,y+b,x+1,y+a, black);
			GraphxLine(hdc, x-b,y-1,x-a,y-1, black);
			GraphxLine(hdc, x-b,y+1,x-a,y+1, black);
			GraphxLine(hdc, x+b,y-1,x+a,y-1, black);
			GraphxLine(hdc, x+b,y+1,x+a,y+1, black);
			// Draw the white lines
			GraphxLine(hdc, x  ,y-b,x  ,y-a, white);
			GraphxLine(hdc, x  ,y+b,x  ,y+a, white);
			GraphxLine(hdc, x-b,y  ,x-a,y  , white);
			GraphxLine(hdc, x+b,y  ,x+a,y  , white);
		}
		return false;
	}
	
*/