/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.renderer;



import java.awt.Frame;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Point;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.logging.Logger;

import javax.media.opengl.GLCanvas;
import javax.media.opengl.GLCapabilities;
import javax.media.opengl.GLContext;
import javax.media.opengl.GLEventListener;

import t3.hrd.input.KeyboardInput;
import t3.hrd.input.PointInputDeviceState;
import t3.hrd.state.JOGLHelper;
import t3.hrd.state.Link;
import t3.hrd.state.Tile;
import Jama.Matrix;



/**
 * Class representing a single projector.
 * 
 * @author pjt40
 *
 */
public class Projector{
	

	private static final Logger logger = Logger.getLogger("t3.hrd.renderer");
	
	/**
	 * Transforms used by this projector
	 */
	public final ProjectorTransforms transforms;
	/**
	 * Parent HRDRenderer
	 */
	public final HRDRenderer hrdRenderer;
	
	Frame frame;
	GLCanvas canvas;
	final ProjectorOpenGLCanvasRepainter openGLcanvasRepainter;
	//ProjectorConfig projectorConfig;
	private final GraphicsDevice graphicsDevice; 

	private final Set<Tile> tilesVisibleInThisProjectorsDeskSpace;
	private final Set<Link> linksVisibleInThisProjectorsDeskSpace; 
	public final Set<Link> linksVisibleInThisProjectorsDeskSpaceImmutable; 
	public final Set<Tile> tilesVisibleInThisProjectorsDeskSpaceImmutable;


	private boolean needToSwapBuffers = false;
	
	
	Projector(ProjectorTransforms transforms, HRDRenderer hrd, Projector shareContextWith) throws ProjectorOpeningException {
		
		this.hrdRenderer = hrd;	
	
		this.transforms = transforms;
		this.openGLcanvasRepainter = new ProjectorOpenGLCanvasRepainter(this);			
		
		this.tilesVisibleInThisProjectorsDeskSpace = new HashSet<Tile>();
		this.linksVisibleInThisProjectorsDeskSpace = new HashSet<Link>();
		this.tilesVisibleInThisProjectorsDeskSpaceImmutable = Collections.unmodifiableSet(this.tilesVisibleInThisProjectorsDeskSpace);
		this.linksVisibleInThisProjectorsDeskSpaceImmutable = Collections.unmodifiableSet(this.linksVisibleInThisProjectorsDeskSpace);
		
		this.frame = createOGLWindow(
				transforms.projectorConfig, 
				this.openGLcanvasRepainter, 
				shareContextWith!=null 
					? shareContextWith.canvas.getContext()
					: null
			);
		this.graphicsDevice = this.frame.getGraphicsConfiguration().getDevice();
		this.canvas = (GLCanvas) frame.getComponent(0);
		
		frame.addWindowListener( new WindowAdapter() {
			public void windowClosed(WindowEvent e){
			}
			public void windowClosing(WindowEvent e) {
				Projector.this.hrdRenderer.requestClose();
				windowClosed(e);
			}
			public void windowDeiconified(WindowEvent e) {
				// this doesn't seem to work
				Projector.this.hrdRenderer.requestForceRedrawAll();
			}
		});
		this.canvas.addFocusListener( new FocusAdapter() {
			// but this works!
			public void focusGained(FocusEvent e){
				Projector.this.hrdRenderer.requestForceRedrawAll();
			}
		});
		// add input listeners to the canvas, not the frame, since the frame stops
		// generating input events in full screen mode but the canvas does not.
		canvas.addKeyListener( new KeyListener() {
			public void keyPressed(KeyEvent e) {
				boolean processed = Projector.this.hrdRenderer.keyPressed(e);	
				if(!processed) {
					int personId = ((e.getModifiersEx() & e.ALT_DOWN_MASK)!=0) ? 1 : 0; 
					int modifiers = e.getModifiersEx() & ~ e.ALT_DOWN_MASK & ~ e.ALT_MASK;
					Projector.this.hrdRenderer.callBacks.callback_keyboardInput( new KeyboardInput(Projector.this.hrdRenderer.clientId, personId, e.getKeyChar(), e.getKeyCode(), modifiers, KeyboardInput.MESSAGE_TYPE_PRESSED), Projector.this.hrdRenderer );
				}
			}	
			public void keyTyped(KeyEvent e) { 	
				int personId = ((e.getModifiersEx() & e.ALT_DOWN_MASK)!=0) ? 1 : 0; 
				int modifiers = e.getModifiersEx() & ~ e.ALT_DOWN_MASK & ~ e.ALT_MASK;
				Projector.this.hrdRenderer.callBacks.callback_keyboardInput( new KeyboardInput(Projector.this.hrdRenderer.clientId, personId, e.getKeyChar(), e.getKeyCode(), modifiers, KeyboardInput.MESSAGE_TYPE_TYPED), Projector.this.hrdRenderer ); 
			}
			public void keyReleased(KeyEvent e) { 
				int personId = ((e.getModifiersEx() & e.ALT_DOWN_MASK)!=0) ? 1 : 0; 
				int modifiers = e.getModifiersEx() & ~ e.ALT_DOWN_MASK & ~ e.ALT_MASK;
				Projector.this.hrdRenderer.callBacks.callback_keyboardInput( new KeyboardInput(Projector.this.hrdRenderer.clientId,personId, e.getKeyChar(), e.getKeyCode(), modifiers, KeyboardInput.MESSAGE_TYPE_RELEASED), Projector.this.hrdRenderer );  
			}
		});
		canvas.addMouseListener( new MouseListener() {
			public void mouseClicked(MouseEvent e) { 
				if(!dispatchMouseEventIfAsPointInputDevice(e)) {
					Projector.this.hrdRenderer.callBacks.callback_mouseClicked(e,Projector.this);
				}
			}
	        public void mouseEntered(MouseEvent e) { 
				if(!dispatchMouseEventIfAsPointInputDevice(e)) {
					Projector.this.hrdRenderer.callBacks.callback_mouseEntered(e,Projector.this);
	        	}
	        }
	        public void mouseExited(MouseEvent e) { 
				if(!dispatchMouseEventIfAsPointInputDevice(e)) {
					Projector.this.hrdRenderer.callBacks.callback_mouseExited(e,Projector.this);
				}
	        }
	        public void mousePressed(MouseEvent e) { 
				if(!dispatchMouseEventIfAsPointInputDevice(e)) {
					Projector.this.hrdRenderer.callBacks.callback_mousePressed(e,Projector.this);
				}
	        }
	        public void mouseReleased(MouseEvent e) { 
				if(!dispatchMouseEventIfAsPointInputDevice(e)) {
					Projector.this.hrdRenderer.callBacks.callback_mouseReleased(e,Projector.this);
				}
	        }
	      
		});
		canvas.addMouseMotionListener(new MouseMotionListener() {
			public void mouseMoved(MouseEvent e) { 
				if(!dispatchMouseEventIfAsPointInputDevice(e)) {
					Projector.this.hrdRenderer.callBacks.callback_mouseMoved(e,Projector.this);
				}
			}
	        public void mouseDragged(MouseEvent e) { 
				if(!dispatchMouseEventIfAsPointInputDevice(e)) {
					Projector.this.hrdRenderer.callBacks.callback_mouseDragged(e,Projector.this);}
				}
		});

	}
	
	private boolean dispatchMouseEventIfAsPointInputDevice(MouseEvent e)  {	
		if(Projector.this.hrdRenderer.mouseAsPointInputDevice) {
			double[] dDESK = Projector.this.getDESKfromAWTMouseEvent(e);
			int buttons = 0; 			
			if((e.getModifiersEx() & e.BUTTON1_DOWN_MASK )!=0) { buttons |= 1<<0; }
			if((e.getModifiersEx() & e.BUTTON2_DOWN_MASK )!=0) { buttons |= 1<<2; }
			if((e.getModifiersEx() & e.BUTTON3_DOWN_MASK )!=0) { buttons |= 1<<1; }
			int penType = ((e.getModifiersEx() & e.CTRL_DOWN_MASK )!=0) ? 1 : 0; 
			int personId = ((e.getModifiersEx() & e.ALT_DOWN_MASK)!=0) ? 1 : 0; 
			PointInputDeviceState msg = 
				new PointInputDeviceState(
					Projector.this.hrdRenderer.clientId, 
					penType, 
					personId,
					true,
					dDESK[0]/dDESK[2],
					dDESK[1]/dDESK[2],
					buttons,
                    null
				);
			Projector.this.hrdRenderer.callBacks.callback_pointInputDeviceStateChanged(msg,this.hrdRenderer);
			return true;
		} else {
			return false;
		}
	}
	
	
	/**
	 * Exception that can be thrown if we cannot open the OpenGL window
	 * 
	 * @author pjt40
	 *
	 */
	public static class ProjectorOpeningException extends Exception {
		public ProjectorOpeningException(String s) { super(s); }
	};
	
	
	

	
	
	void oncePerFrameRender(boolean forceRedraw) {
		
		boolean needToRedrawBecauseOfElVisOrAffChangeInOurDeskSpace =
			recalcElsInThisProjectorsDeskSpaceOncePerFrame();
		
		boolean needToRedraw =
			forceRedraw
			|| needToRedrawBecauseOfElVisOrAffChangeInOurDeskSpace 
			|| this.hrdRenderer.stateListener.projectorNeedsRefreshingBecauseOfCursors(this)
			|| this.isNecessaryToRedrawBecauseOfTileContentUpdate()
			|| this.isNecessaryToRedrawBecauseSharingTexturesAndPrimary()			
			|| this.hrdRenderer.callBacks.callback_needToRepaintOverlay(this);
		
		if(needToRedraw) {
			this.canvas.display();
			this.needToSwapBuffers = true;
		} else {
			// no need to redraw
			this.needToSwapBuffers = false;
		}
	}

	void oncePerFrameSwapBuffersIfNecessary() {
		if(this.needToSwapBuffers) {
			this.canvas.swapBuffers();
			this.needToSwapBuffers = false;			
		} else {
			// no need!
		}
	}
	
	
	/**
	 * This MUST be called exactly once per frame. Returns true iff the projector must
	 * be redrawn because of tiles or links moving/vischanging on this projector's desk space
	 * @return
	 */
	private boolean recalcElsInThisProjectorsDeskSpaceOncePerFrame() {
		boolean needToRedraw = false;
		for(Tile tile: hrdRenderer.stateListener.backedTilesWhoseAffineTransformsOrVisibilityOrUnwRectsHaveChanged) {
			
			boolean tileIsNowInThisProjectorsDeskSpace = 
				tile.isVisible() && this.transforms.rDESKdeskSpaceAlignedNonBlackedRect.intersects(tile.getrDESKboundingBox());
			boolean tileWasOnThisProjectorsDeskSpace = this.tilesVisibleInThisProjectorsDeskSpace.contains(tile);
			
			// we need to redraw if the tile was or is in our desk space
			if(tileIsNowInThisProjectorsDeskSpace || tileWasOnThisProjectorsDeskSpace) {
				needToRedraw = true;
			}
			
			// now work out which tiles are in our desk space
			if(tileIsNowInThisProjectorsDeskSpace && !tileWasOnThisProjectorsDeskSpace) {
				this.tilesVisibleInThisProjectorsDeskSpace.add(tile);
			} else if(!tileIsNowInThisProjectorsDeskSpace && tileWasOnThisProjectorsDeskSpace) {
				this.tilesVisibleInThisProjectorsDeskSpace.remove(tile);
			} else {
				// no change
			}			
		}		
		
		for(Link link: hrdRenderer.stateListener.linksWhoseAffineTransformsOrVisibilityHaveChanged) {
			
			boolean wasOnThisProjector = this.linksVisibleInThisProjectorsDeskSpace.contains(link);
			boolean isNowOnThisProjector = this.transforms.rDESKdeskSpaceAlignedNonBlackedRect.intersects(link.getrDESKboundingBox());;

			// 	we need to redraw if the link was or is in our desk space
			if(wasOnThisProjector || isNowOnThisProjector) {
				needToRedraw = true;
			}
			
			// now work out which links are in our desk space
			if(!wasOnThisProjector && isNowOnThisProjector) {
				linksVisibleInThisProjectorsDeskSpace.add(link);
			} else if(wasOnThisProjector && !isNowOnThisProjector) {
				linksVisibleInThisProjectorsDeskSpace.remove(link);
			} else {
				// no change
			}
		}	
		return needToRedraw;
	}
	
	
	
	
	private boolean isNecessaryToRedrawBecauseSharingTexturesAndPrimary() {
		return 
			this.hrdRenderer.shareTexturesBetweenContextsPrimaryProjector == this
			&& (
				this.hrdRenderer.stateListener.backedTilesWhoseVisibilityHasChanged.size()!=0
				|| hrdRenderer.stateListener.tileContentUpdatesForBackedTiles.size()!=0
			);				
	}	
	
	private boolean isNecessaryToRedrawBecauseOfTileContentUpdate() {
		// primary or non-primary
		for(StateListenerForHRDRenderer.TileContentUpdate tileContentUpdate:hrdRenderer.stateListener.tileContentUpdatesForBackedTiles) {
			if(this.tilesVisibleInThisProjectorsDeskSpace.contains(tileContentUpdate.tile)) {
				// if tile is visible on this projector then whether we're texture sharing or not
				// we will have to redraw.
				return true;
			} else {
				// continue
			}
		}
		return false;		
	}
	

	
	
	void close() {
		// will have previously been told to remove tiles.
		frame.setVisible(false);
		frame.dispose();
	}
	
	private double[] getDESKfromAWTMouseEvent(MouseEvent e) {
		Point eventComponentOrigin = e.getComponent().getLocationOnScreen();
		int xInOneBigScreenSpace = e.getX()+eventComponentOrigin.x;
		int yInOneBigScreenSpace = e.getComponent().getHeight()-(e.getY()+eventComponentOrigin.y);
		int xInFBspace = xInOneBigScreenSpace - 
			this.graphicsDevice.getDefaultConfiguration().getBounds().x;
		int yInFBspace = yInOneBigScreenSpace - 
			this.graphicsDevice.getDefaultConfiguration().getBounds().y;
		Matrix mFBeventCoords = JOGLHelper.getMFromD(
			xInFBspace,
			yInFBspace,
			1.0
		);
		return JOGLHelper.getDFromM(this.transforms.mFBTtoDESK.times(mFBeventCoords));
		
	}
	

	
	
	
	
	/**
	 * Creates an OpenGL window
	 * 
	 * @param c		Desired location of the window
	 * @param r		Event listener for the JOGL canvas
	 * @param shareWith	If not null, then JOGL canvas's OpenGL context must share with this context.
	 * @return		Frame containting the JOGL canvas
	 * @throws ProjectorOpeningException		If we cannot create the window as expected
	 */
	public static Frame createOGLWindow(ProjectorConfig c, GLEventListener r, GLContext shareWith) throws ProjectorOpeningException {
		GraphicsDevice graphicsDevice;
		try {
			graphicsDevice = GraphicsEnvironment.getLocalGraphicsEnvironment().getScreenDevices()[c.graphicsDeviceIndex];
		} catch(ArrayIndexOutOfBoundsException e) {
			throw new IllegalArgumentException("No such GraphicsDevice with index "+c.graphicsDeviceIndex);
		}
		
	
		if(System.getProperty("sun.java2d.noddraw")==null || !System.getProperty("sun.java2d.noddraw").equals("true")) {
			throw new ProjectorOpeningException("Need to run jvm with -Dsun.java2d.noddraw=true");
		}
		
		Frame frame = new Frame("Display", graphicsDevice.getDefaultConfiguration());
		
		frame.setUndecorated(true); 
		
		GLCanvas canvas = new GLCanvas(new GLCapabilities(),null, shareWith, graphicsDevice);

		// need to ignore repaint events from OS so that we don't run into thread problems
		// whereby the awt thread is repainting while we're updating the state.
		canvas.setIgnoreRepaint(true);
		
		canvas.setAutoSwapBufferMode(false);
		canvas.addGLEventListener(r);
				
		// we want to trap tab key presses
		canvas.setFocusTraversalKeysEnabled(false);
		
		frame.add(canvas);
		frame.setLocation(
				graphicsDevice.getDefaultConfiguration().getBounds().x, 
				graphicsDevice.getDefaultConfiguration().getBounds().y
				);
		frame.setSize(
				graphicsDevice.getDisplayMode().getWidth(),
				graphicsDevice.getDisplayMode().getHeight());
		if(c.window_fullScreenExclusive) {
			graphicsDevice.setFullScreenWindow(frame);
			if(graphicsDevice.getFullScreenWindow() != frame) {
				throw new ProjectorOpeningException(
						"Asked for full screen exclusive and didn't get it on device "
						+graphicsDevice.getIDstring()
					);
			}
		}
		frame.setVisible(true);
		if(canvas.getGraphicsConfiguration().getDevice()!=graphicsDevice) {
			throw new ProjectorOpeningException(
					"We asked for device "
					+graphicsDevice.getIDstring()
					+" and got device "
					+canvas.getGraphicsConfiguration().getDevice().getIDstring()
				);
		}
		

		// trap key presses and make the focus.
		frame.requestFocus();
		canvas.requestFocusInWindow();
		
		
		
		return frame;
	}

	
		
}	
	
	
	
	
	
	
	
	
	
	

