/**
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios.swing;

import java.awt.Rectangle;
import java.util.HashMap;
import java.util.logging.Logger;

import javax.swing.JComponent;
import javax.swing.JRootPane;
import javax.swing.RepaintManager;
import javax.swing.SwingUtilities;

/*
 * This class is completely thread safe
 */


class VerboseRepaintManager extends RepaintManager {
	private HashMap<JRootPane,Rectangle> rootPaneToDirtyRegion = new HashMap<JRootPane,Rectangle>();

	private static final Logger logger = Logger.getLogger("t3.hrd.portfolios.swing");
	VerboseRepaintManager() {
		super();
		// turn off double buffering
		// if we don't then it tries to copy a huge back buffer into our
		// little buffered image :-( and throws an array out of bounds exception
        this.setDoubleBufferingEnabled(false);
	}

	
    public synchronized void addDirtyRegion(JComponent c, int x, int y, int w, int h) {
		// synchronize on the paintmanager 
    	JRootPane rootPane = c.getRootPane();
    	
    	Rectangle r = rootPaneToDirtyRegion.get(rootPane);
    	Rectangle newRect = new Rectangle(x,y,w,h);
    	correctForNegativeWidthHeight(newRect);

    	if(!SwingFramePortfolio.adjustCoordsToFrameOfRefOfRootPane(newRect, c)) {
    		return;
    	}
    	
    	if(r==null) {
    		r = newRect;
    	} else {
    		r = r.union(newRect);
    	}
    	rootPaneToDirtyRegion.put(rootPane, r);
        super.addDirtyRegion(c,x,y,w,h);
        
    }
    
	public void paintDirtyRegions() {	
		// danger of deadlock. mustn't be synchronized on THIS while we actually repaint
		// because repainting might require locks that are held by threads that are 
		// waiting for the lock on THIS so that they can enter addDirtyRegion.
		// instead we'll do an atomic swap and then release the lock.
		
		HashMap<JRootPane,Rectangle> rootPaneToDirtyRegionCopy; 
		
		synchronized(this) {
			rootPaneToDirtyRegionCopy = this.rootPaneToDirtyRegion;
			this.rootPaneToDirtyRegion = new HashMap<JRootPane,Rectangle>();
		}
		
        super.paintDirtyRegions();
        
        // still need to synchronize on it when iterating
		synchronized(SwingFramePortfolio.swingFramePortfoliosThreadSafe) {
			for(final SwingFramePortfolio p: SwingFramePortfolio.swingFramePortfoliosThreadSafe) {
				
                if(!p.isDestroyed()) {
                    
    				// get the rectangle in the portfolio's rootframe's coords
    				Rectangle redrawRectInPortfoliosRootframe = rootPaneToDirtyRegionCopy.get(p.frame.getRootPane());
    				if(redrawRectInPortfoliosRootframe==null) {
    					redrawRectInPortfoliosRootframe = new Rectangle(); 
    				}
    				
    				// expand the rectangle to include any popups
    				for(MyPopup pop:SwingFramePortfolio.mypf.getPopupsForSourceRootPane(p.frame.getRootPane())) {
    					
    					// check is there a repaint event for the popup
    					Rectangle rr = rootPaneToDirtyRegionCopy.get(pop.contents.getRootPane());
    					if(rr!=null) {
    													
    						// convert rr into frame of ref of portfolio's rootframe
    						rr = SwingUtilities.convertRectangle(pop.contents.getRootPane(), rr, p.frame.getRootPane());
    						
    						// expand redraw rect to include the new area
    						redrawRectInPortfoliosRootframe = redrawRectInPortfoliosRootframe.union(rr);
    						
    					} else {
    						// the popup didn't need redrawing!
    					}
    					
    				}
    				
    				// intersect with whole of tile area
    				// we do this because often we get update rectangles like (-3,0,100,100)
    				redrawRectInPortfoliosRootframe = redrawRectInPortfoliosRootframe.intersection(new Rectangle(0,0,p.getTileWidthInTILE(),p.getTileHeightInTILE()));
    				
    				if(redrawRectInPortfoliosRootframe.height>0 && redrawRectInPortfoliosRootframe.width>0) {
    					final Rectangle r = redrawRectInPortfoliosRootframe;
    					p.portfolioServer.performActionAsynchronouslyByCurrentThread(
                                new Runnable() { public void run() {
                                    if(!p.isDestroyed()) {
                                        p.triggerRepaintTile(r);
                                    }
                                }}
    					);							
    				} else {
    					// the portfolio didn't need updating! great.
    				}
                } else {
                    // p is destroyed
                }				
			}
		}
		
		rootPaneToDirtyRegionCopy = null;

		// no need to do rootPaneToDirtyRegion.clear();

    }
	
	
	

	
	

	
	// see http://java.sun.com/products/jfc/tsc/articles/painting/
	
	public static void correctForNegativeWidthHeight(Rectangle r) {
		if(r.height<0) {
			r.y = r.y+r.height;
			r.height = -r.height;
		}
		if(r.width<0) {
			r.x = r.x+r.width;
			r.width = -r.width;
		}
	}
}