/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.portfolios;


import java.io.Serializable;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import t3.hrd.state.Cursor;

/**
 * Data structure class that stores behind-the-scenes data for T3 about a point input
 * device. You should not normally need to use this class. This class is not thread safe; 
 * if you need to use it then you should synchronize on the stateManager first.
 * @author pjt40
 *
 */
public class PointInputDeviceData {
	final Cursor cursor;
	
	// no need to have synchronized set methods for these
	// they're only modified/read by the thread for the client with the pen
	int buttonsWhenLastEventCreated;
	boolean positionAndButtonsKnownWhenLastEventCreated;
	double DESKxWhenLastEventCreated, DESKyWhenLastEventCreated;	
	double DESKxWhenLastFDOPeventCreated, DESKyWhenLastFDOPeventCreated;	
	double PORTxWhenFDOPmodeEntered, PORTyWhenFDOPmodeEntered;	
	int buttonsWhenLastFDOPeventCreated;
	Set<Portfolio> portfoliosInsideWhenLastEventCreated;
    Set<Portfolio> portfoliosToNotifyWhenPressedOutside;
    Serializable extraWhenLastEventCreated;
    Serializable extraWhenLastFDOPeventCreated;
	
	// if this is null then not in fdop mode
	Portfolio fdopModePortfolio;
	int fdopModeButton;
	
	PointInputDeviceData(Cursor cursor) {
		// TODO Auto-generated constructor stub
		this.buttonsWhenLastEventCreated = 0;
		this.positionAndButtonsKnownWhenLastEventCreated = false;
		this.DESKxWhenLastEventCreated = 0.0;
		this.DESKyWhenLastEventCreated = 0.0;
		this.portfoliosInsideWhenLastEventCreated = new HashSet<Portfolio>();
        this.portfoliosToNotifyWhenPressedOutside = new HashSet<Portfolio>();
		this.cursor = cursor;
		this.fdopModePortfolio = null;
		this.fdopModeButton = 0;
        this.extraWhenLastEventCreated = null;
        this.extraWhenLastFDOPeventCreated = null;
	}

	public int getButtonsWhenLastEventCreated() {
		return buttonsWhenLastEventCreated;
	}

	public int getButtonsWhenLastFDOPeventCreated() {
		return buttonsWhenLastFDOPeventCreated;
	}

	public Cursor getCursor() {
		return cursor;
	}

	public double getDESKxWhenLastEventCreated() {
		return DESKxWhenLastEventCreated;
	}

	public double getDESKxWhenLastFDOPeventCreated() {
		return DESKxWhenLastFDOPeventCreated;
	}

	public double getDESKyWhenLastEventCreated() {
		return DESKyWhenLastEventCreated;
	}

	public double getDESKyWhenLastFDOPeventCreated() {
		return DESKyWhenLastFDOPeventCreated;
	}

	public Portfolio getFdopModePortfolio() {
		return fdopModePortfolio;
	}

	public Set<Portfolio> getPortfoliosInsideWhenLastEventCreated() {
		return Collections.unmodifiableSet(portfoliosInsideWhenLastEventCreated);
	}

	public double getPORTxWhenFDOPmodeEntered() {
		return PORTxWhenFDOPmodeEntered;
	}

	public double getPORTyWhenFDOPmodeEntered() {
		return PORTyWhenFDOPmodeEntered;
	}

	public boolean isPositionAndButtonsKnownWhenLastEventCreated() {
		return positionAndButtonsKnownWhenLastEventCreated;
	}
	
	
	
}
