/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */
package t3.hrd.input;


import java.util.Properties;
import java.util.logging.Logger;

import Jama.Matrix;

import t3.hrd.state.JOGLHelper;
import t3.hrd.state.JOGLHelper.PropertiesException;
import uk.org.pjt.jwintab.Jwintab;

/**
 * Wintab PointInputDevice class for use with graphics tablets.
 * <p>
 * Uses the following properties:
 * <ul>
 * <li>hrd.pointInputDevices.n.xDivJump: number of x divisions the pen must move before a new event is triggered.
 * <li>hrd.pointInputDevices.n.yDivJump: number of y divisions the pen must move before a new event is triggered.
 * </ul>
 * <p>
 * For our tablet: 1 Inch = 25.40mm; tablet is 48"*36"; has 48000 x divisions and 36000 y divisions;
 * works out around 40 divisions per mm; so xScale = 48.0 * 25.40 / 48000.0 = 0.025400; 
 * and yScale = 36.0 * 25.40 / 36000.0 = 0.025400; and xDivJump=yDivJump=10.
 * 
 * @author pjt40
 *
 */
public class Wintab extends PointInputDevice {

	private static final Logger logger = Logger.getLogger("t3.hrd.input");
	
	private final int xDivJump, yDivJump;
	private final boolean accurate;
	
	private int wintabArg[] = new int[4];
	private int wintabOldx=0;
	private int wintabOldy=0;
	private int wintabOldButtons=0;
	private long noPacketTimer = 0;
    private int noPacketTimerFrames = 0;
    private boolean skipNextPacketAsJunk = false;
    private boolean checkNextNotJunkedPacketForButtonHeldWhileInAir = false;
	
	public Wintab(int clientId, int personId, Matrix mINPUTtoDESK, boolean accurate, Properties p, String prefix) throws InputDeviceException { 
		super(clientId, 0,personId, mINPUTtoDESK);
		this.accurate=accurate;
		try {
			this.xDivJump = JOGLHelper.props_getInt(p,prefix+"xDivJump");
			this.yDivJump = JOGLHelper.props_getInt(p,prefix+"yDivJump");			
		} catch(PropertiesException e) {
			throw new InputDeviceException(e.getMessage());
		}
		if(Jwintab.open()==-1) {
			throw new InputDeviceException("Couldn't open wintab");
		}
	}
	public boolean updateState()  throws InputDeviceException{
		logger.fine("Updated state");
		int res = Jwintab.getPacket(wintabArg, 1);
		if(res==0) {
			// no packet
			if(this.noPacketTimerFrames>10 && (System.currentTimeMillis()>this.noPacketTimer+300)) {
				// if no packet for 10 frames and 300ms then pen is off the table or very still.
				// if it is off the table we need to junk the next packet as wintab 
                // seems to get confused. if it's still on the table then there's no real
                // harm in doing this anyway. we also need to see where the pen is being
                // put down again - if it's a long way from where it previously was then
                // if a button was being held down we need to clear it.
                this.skipNextPacketAsJunk = true;
                this.checkNextNotJunkedPacketForButtonHeldWhileInAir = true;
			} else {
                this.noPacketTimerFrames++;
			}
           // System.out.println("No packet");
            return false;
		} else if(res==-1) {
			throw new InputDeviceException("Couldn't get wintab packet");
		} else if(res==1) {
            
            this.noPacketTimer = System.currentTimeMillis();
            this.noPacketTimerFrames = 0;            
            
            if(this.skipNextPacketAsJunk) {
                
                this.skipNextPacketAsJunk = false;
                //System.out.println("Junked packet");            
                return false;
                
            } else {
            
    			if(Math.abs(wintabArg[0]-this.wintabOldx)>=this.xDivJump || Math.abs(wintabArg[1]-this.wintabOldy)>=this.yDivJump || this.wintabOldButtons!=wintabArg[2] || accurate) {
    				
                    this.state.positionAndButtonsKnown = true;
    				this.setStateDESKcoordsFromINPUTcoords((double)wintabArg[0], (double)wintabArg[1]);
    				
                    if(this.checkNextNotJunkedPacketForButtonHeldWhileInAir) {
                        if(Math.abs(wintabArg[0]-this.wintabOldx)>=5*this.xDivJump || Math.abs(wintabArg[1]-this.wintabOldy)>=5*this.yDivJump) {
                            // reset button state as we were holding a button down in the air and
                            // now we've come down somewhere else.
                            this.state.buttons=0;
                            this.checkNextNotJunkedPacketForButtonHeldWhileInAir=false;
                        } else {
                            this.state.buttons = wintabArg[2];
                            this.checkNextNotJunkedPacketForButtonHeldWhileInAir=false;
                        }
                    } else {
    					this.state.buttons = wintabArg[2];
    				}
        
                    this.wintabOldx=wintabArg[0];
                    this.wintabOldy=wintabArg[1];               
                    this.wintabOldButtons=wintabArg[2];
                     
    				logger.fine("State: x="+this.state.DESKx+" y="+this.state.DESKy+" buttons="+this.state.buttons);
                    
                    //System.out.println(this.state);
    				return true;
                    
    			} else {
                    
                    //System.out.println("Skiped as didn't move a pixels worth");
    				// didn't move anywhere near a screen pixel's worth.
    				return false;                    
    			}
            }
		} else {
			assert false;
			return false;
		}
	}
	public void close()  throws InputDeviceException{
		Jwintab.close();
	}

}
