
/*
 * Copyright (c) 2008 Philip Tuddenham
 * 
 * This work is licenced under the 
 * Creative Commons Attribution-NonCommercial-ShareAlike 2.5 License. 
 * To view a copy of this licence, visit 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/ or send a letter to 
 * Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.
 */

package t3.hrd.input;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * Has a circular buffer.
 * Whenever it reads from the underlying stream, it reads as much as it can.
 * Whenever you call a read operation on it, if there is ever more than
 * minAutoReadSize available in the underlying stream, then it reads as much
 * as it can.
 * You can also invoke this operation directly. 
 * @author pjt40
 *
 */
class CircularBufferedInputStream extends FilterInputStream {
    
    public static void main(String[] args) throws Throwable {
        CircularBufferedInputStream c = new CircularBufferedInputStream(new Tester(), 20, 5);
        int i=0;
        while(i<2000) {
            i++;
            System.out.println(c.read());
        }
    }
    
    public static class Tester extends InputStream {
        
        int c = 0;
        public int available() throws IOException {
            // TODO Auto-generated method stub
            return 20;
        }

        @Override
        public int read() throws IOException {
            // TODO Auto-generated method stub
            System.out.println("Read!");
            c++;
            if(c>255) c-=255;
            return c;
        }

        @Override
        public long skip(long n) throws IOException {
            // TODO Auto-generated method stub
            c+=n;
            c = c % 255;
            return n;
        }
        
    }
    
   
    private byte[] buf;
    private final int buflength;
    private int nextwritepos=0, nextreadpos=0;  
    // positions range from 0 up to and including buf.length-1
    // when nextwritepos==nextreadpos this means the buffer is empty, not full
    // so actually the buffer can only hold buflength-1 elements.
    

    private final int minAutoReadSize;
    private boolean endOfStreamReached = false;
    // MIN_AUTO_READ_SIZE<=buf.length
    
    public CircularBufferedInputStream(InputStream in, int bufsize, int minAutoReadSize) {
        super(in);
        if (bufsize <= 0) {
            throw new IllegalArgumentException("Buffer size <= 0");
        }
        if (minAutoReadSize <= 0 || minAutoReadSize>bufsize) {
            throw new IllegalArgumentException("minAutoReadSize <= 0 or >bufsize");
        }
        this.buflength =bufsize+1;
        this.buf = new byte[this.buflength];
        this.minAutoReadSize = minAutoReadSize;
        
    }
    
    
    private InputStream getInIfOpen() throws IOException {
        if (in == null)
            throw new IOException("Stream closed");
        return in;
    }
        
    private byte[] getBufIfOpen() throws IOException {
        if (buf== null)
            throw new IOException("Stream closed");
        return buf;
    }
    
    /**
     * if the underlying input stream has at least MIN_AUTO_READ_SIZE available
     * and there is space in the buffer then read in as much as we can without blocking. 
     * At most two read operations are performed on the underlying stream.
     * @throws IOException
     */
    public void ifMinAutoReadSizeAvailableThenReadAsMuchAsPosIntoBufferWithoutBlocking() throws IOException {
        int canBeReadFromInputWithoutBlocking = Math.min(this.getInIfOpen().available(), this.getFreeSpaceInBuffer()); 
        if(canBeReadFromInputWithoutBlocking>=this.minAutoReadSize) {
            this.forceFillExactSize(canBeReadFromInputWithoutBlocking);
        }
    }
    
    
    /**
     * if the underlying input stream has at least MIN_AUTO_READ_SIZE available
     * and there is space in the buffer then read in as much as we can without blocking. 
     * Furthermore, we need to make sure that size bytes are available in the buffer 
     * when we're done, even if it means blocking while we wait for the input. 
     * At most two read operations are performed on the underlying stream.
     * @param size
     * @throws IOException
     */
    private void readIntoBufferIfMinAutoReadSizeAvailableOrSoAtLeastAvailable(int size) throws IOException {
        assert size<=this.buflength-1;
        int canBeReadFromInputWithoutBlocking = Math.min(this.getInIfOpen().available(), this.getFreeSpaceInBuffer());
        int availInBuffer = this.availableBuffered();
        if(canBeReadFromInputWithoutBlocking>=this.minAutoReadSize || availInBuffer<size) {
            int numBytesToRead = Math.max(canBeReadFromInputWithoutBlocking, size-availInBuffer);
            this.forceFillExactSize(numBytesToRead);            
        } else {
            // no need to do anything
        }
    }
    
    
    
    /**
     * Try to read size bytes from underlying input stream
     * We do at most two reads.
     * It might not actually read size bytes.
     * Might block.
     * @param size
     */
    private void forceFillExactSize(int size) throws IOException {
        if(size>getFreeSpaceInBuffer()) {
            throw new IllegalArgumentException("Not enough room in buffer");
        }
        if(nextwritepos+size>buflength) {
            int numBytesToReadA = buflength-nextwritepos;
            int numBytesToReadB = size-numBytesToReadA;
            forceFillExactSize(numBytesToReadA);
            forceFillExactSize(numBytesToReadB);
            return;
        } else {
            int actuallyRead = this.getInIfOpen().read(this.getBufIfOpen(), nextwritepos, size);
            if(actuallyRead!=-1) {
                nextwritepos+=actuallyRead;
                if(nextwritepos>=buflength) {
                    nextwritepos-=buflength;
                }
            } else {
                endOfStreamReached=true;
            }
            return;
        }
    }
    
    private void readFromBuffer(byte[] b, int off, int len) {
        assert len<=availableBuffered();
        if(nextreadpos+len>buflength) {
            int numBytesToReadA = buflength-nextreadpos;
            int numBytesToReadB = len-numBytesToReadA;
            readFromBuffer(b,off,numBytesToReadA);
            readFromBuffer(b,off+numBytesToReadA,numBytesToReadB);
        } else {
            System.arraycopy(this.buf,nextreadpos,b, off, len );
            nextreadpos+=len;
            if(nextreadpos>=buflength) {
                nextreadpos-=buflength;
            }
        }
    }
    
    
    private int availableBuffered() {
        if(nextreadpos<=nextwritepos) {
            return nextwritepos-nextreadpos;
        } else {
            return buflength-(nextreadpos-nextwritepos);
        }
    }
    
    /**
     * Always minus 1 because can never actually fill the buffer
     * @return
     */
    private int getFreeSpaceInBuffer() {
        return buflength-availableBuffered()-1;
    }

    public int available() throws IOException {
        return availableBuffered()+this.getInIfOpen().available();
    }

    @Override
    public void close() throws IOException {
        this.getInIfOpen().close();
        this.in = null;
        this.buf = null;
    }


    @Override
    public int read() throws IOException {   
        this.readIntoBufferIfMinAutoReadSizeAvailableOrSoAtLeastAvailable(1);
        if(this.availableBuffered()==0) {
            assert this.endOfStreamReached;
            return -1;
        } else {
            int returnVal = this.getBufIfOpen()[this.nextreadpos];
            nextreadpos++;
            if(nextreadpos>=buflength) {
                nextreadpos-=buflength;
            }
            return returnVal & 0xff;
        }
    }

    public int read(byte[] b, int off, int len) throws IOException {
        if(len>buflength-1) {
            throw new IllegalArgumentException("Cannot read a block bigger than the circular buffer size");
        }
        this.readIntoBufferIfMinAutoReadSizeAvailableOrSoAtLeastAvailable(len);
        int canActuallyRead = Math.min(len,this.availableBuffered());
        if(canActuallyRead==0) {
            if(this.endOfStreamReached) {
                return -1;                
            } else {
                return 0;
            }
        }
        readFromBuffer(b, off, canActuallyRead);
        return canActuallyRead;       
    }

    @Override
    public int read(byte[] b) throws IOException {
        return read(b, 0, b.length);
    }

    @Override
    public long skip(long n) throws IOException {
        if(availableBuffered()>=n) {
            this.nextreadpos += n;
            if(this.nextreadpos>=buflength) {
                this.nextreadpos-=buflength;
            }
            return n;
        } else {
            int skippedInBuffer = availableBuffered();
            this.nextreadpos = this.nextwritepos;
            return skippedInBuffer + this.getInIfOpen().skip(n-skippedInBuffer);
        }
    }
    
    
}
