// Copyright (c) 2016, Noa Zilberman
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// * Redistributions of source code must retain the above copyright notice, this
//   list of conditions and the following disclaimer.
//
// * Redistributions in binary form must reproduce the above copyright notice,
//   this list of conditions and the following disclaimer in the documentation
//   and/or other materials provided with the distribution.
//
// * Neither the name of the project, the copyright holder nor the names of its
//  contributors may be used to endorse or promote products derived from
//   this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#define _GNU_SOURCE
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/ip_icmp.h>
#include <arpa/inet.h>
#include <sys/select.h>
#include <sched.h>
#include "tsc.h"

//note, to allow root to use icmp sockets, run:
//sysctl -w net.ipv4.ping_group_range="0 0"


#define CPU_MHZ_FILE        "/proc/cpuinfo"
#define CPU_MHZ_PREFIX        "cpu MHz\t\t: "

#define DEFAULT_CPU 2
#define SAMPLES_MAX 100000
#define OUTPUT_FILE "out.txt"

//read clock frequency from cpuinfo
unsigned int get_mhz(void)
{
    FILE *fp;
    char line[1024];
    unsigned int clock_mhz = 0;
    
    fp = fopen("/proc/cpuinfo", "r");
    if (!fp)
        return 0;

    while (fgets(line, sizeof(line), fp))
    {
        if (strncmp(line, CPU_MHZ_PREFIX, sizeof(CPU_MHZ_PREFIX)-1) == 0)
        {
            clock_mhz = atoi(line+sizeof(CPU_MHZ_PREFIX)-1);
            break;
        }
    }
    fclose(fp);
    return clock_mhz;
}

//pin the task to a specific core
static void pin_cpu(int cpu)
{
        cpu_set_t cpu_set;
        CPU_ZERO(&cpu_set);
        CPU_SET(cpu, &cpu_set);
        if (sched_setaffinity(0, sizeof(cpu_set), &cpu_set) == -1)
                perror("sched_setaffinity");
}


static int compare_ull(const void *pa, const void *pb)
{
        unsigned long long a = *(const unsigned long long *)pa;
        unsigned long long b = *(const unsigned long long *)pb;
        return (a < b) ? -1 : (a > b) ? +1 : 0;
}



//handle the icmp message
int icmp_msg(struct icmphdr icmp_hdr, struct sockaddr_in addr, int sock, unsigned char *data )
{
         int rc;
        struct timeval timeout = {2, 0}; //wait max 3 seconds for a reply
        fd_set read_set;
        socklen_t slen;
        struct icmphdr rcv_hdr;
       rc = sendto(sock, data, sizeof icmp_hdr + 12,
                        0, (struct sockaddr*)&addr, sizeof addr);
        if (rc <= 0) {
            perror("Sendto");
            return -1;
        }
        memset(&read_set, 0, sizeof read_set);
        FD_SET(sock, &read_set);
        
        //wait for a reply with a timeout
        //Yes, busy polling is faster but not representing
        rc = select(sock + 1, &read_set, NULL, NULL, &timeout);
        if (rc == 0) {
            puts("Got no reply\n");
           return -1;
        } else if (rc < 0) {
            perror("Select");
            return -1;
        }

        //we don't care about the sender address in this example..
        slen = 0;
        rc = recvfrom(sock, data, sizeof data, 0, NULL, &slen);
        if (rc <= 0) {
            perror("recvfrom");
            return -1;
        } else if (rc < sizeof rcv_hdr) {
            printf("Error, got short ICMP packet, %d bytes\n", rc);
            return -1;
        }
        memcpy(&rcv_hdr, data, sizeof rcv_hdr);
        if (rcv_hdr.type == ICMP_ECHOREPLY) {
        } else {
            printf("Got ICMP packet with type 0x%x ?!?\n", rcv_hdr.type);
        }
    return 0;
}

//measure the round trip time
void measure_rtt(struct in_addr *dst,unsigned long long *rtt,unsigned int samples_num, unsigned int cpu)
{
    int i;
    struct icmphdr icmp_hdr;
    struct sockaddr_in addr;
    unsigned long long end, start;
    int sock = socket(AF_INET,SOCK_DGRAM,IPPROTO_ICMP);
    if (sock < 0) {
        perror("socket");
        return ;
    }

    memset(&addr, 0, sizeof addr);
    addr.sin_family = AF_INET;
    addr.sin_addr = *dst;

    memset(&icmp_hdr, 0, sizeof icmp_hdr);
    icmp_hdr.type = ICMP_ECHO;
    icmp_hdr.un.echo.id = 1984;//arbitrary id

    for (i=0;i<samples_num;i++) {
        unsigned char data[2048];
        int rc;
        struct timeval timeout = {2, 0}; //wait max 3 seconds for a reply
        fd_set read_set;
        socklen_t slen;
        struct icmphdr rcv_hdr;
       
        icmp_hdr.un.echo.sequence = i;
        memcpy(data, &icmp_hdr, sizeof icmp_hdr);
        memcpy(data + sizeof icmp_hdr, "latency_test", 12); //icmp payload
        rdtscpll(start,cpu);
        icmp_msg(icmp_hdr,addr,sock,data);
        rdtscpll(end,cpu);
        rtt[i]=end-start;
    }
}


int main(int argc, char **argv)
{
    int flag,i;
    struct in_addr dst;
    char *fname;
    unsigned int samples_num; 
    unsigned int cpu;
    unsigned long long *rtt;
    unsigned long long start, end;
    unsigned int clock_mhz;

     // Setting default values
     cpu= DEFAULT_CPU;
     samples_num = SAMPLES_MAX;
     fname - OUTPUT_FILE;
  while ((flag = getopt (argc, argv, "i:f:n:c:")) != -1)
    switch (flag)
      {
      case 'i':
        if (inet_aton(optarg, &dst) == 0) {
	        perror("inet_aton");
        	printf("%s isn't a valid IP address\n", optarg);
        return 1;
         }
        break;
      case 'f':
        fname = optarg;
        break;
      case 'n':
         samples_num = atoi(optarg);
         break;
      case 'c':
         cpu = atoi(optarg);
         break;
      default:
          printf("\nusage: ./icmp_test -i <ip address> -f <output file> -n <number of samples> ");
          abort; 
      }

        rtt=malloc(samples_num*sizeof(unsigned long long*));
    if (mlock(rtt,sizeof(rtt)))
           {
            free(rtt);
            return (0);
           }


    clock_mhz = get_mhz();
    if (!clock_mhz)
    {
        fprintf(stderr, "Could not determine clock rate from " CPU_MHZ_FILE "\n");
        return 0;
    }

    pin_cpu(cpu);
    measure_rtt(&dst,rtt,samples_num,cpu);
    FILE *f=fopen(fname,"w"); 
    for (i=0;i<samples_num;i++)
	{
		fprintf(f,"%llu\n",rtt[i]/clock_mhz);
	} 
    qsort(rtt, samples_num, sizeof(rtt[0]), compare_ull);
        fprintf(f,"cpu clock frequency\t\t%u\n",clock_mhz);
        fprintf(f,"min\t\t%llu\t\t%llu\n" "1%%\t\t%llu\t\t%llu\n" "5%%\t\t%llu\t\t%llu\n" "10%%\t\t%llu\t\t%llu\n" "25%%\t\t%llu\t\t%llu\n" "median\t\t%llu\t\t%llu\n" "75%%\t\t%llu\t\t%llu\n" "90%%\t\t%llu\t\t%llu\n" "95%%\t\t%llu\t\t%llu\n" "99%%\t\t%llu\t\t%llu\n" "99.9%%\t\t%llu\t\t%llu\n"  "max\t\t%llu\t\t%llu\n",
                rtt[0]/clock_mhz, rtt[0]*1000/clock_mhz, 
                rtt[samples_num*1/100]/clock_mhz, rtt[samples_num*1/100]*1000/clock_mhz,
                rtt[samples_num*5/100]/clock_mhz,rtt[samples_num*5/100]*1000/clock_mhz,
                rtt[samples_num*10/100]/clock_mhz, rtt[samples_num*10/100]*1000/clock_mhz, 
                rtt[samples_num*25/100]/clock_mhz, rtt[samples_num*25/100]*1000/clock_mhz,
                rtt[samples_num*50/100]/clock_mhz, rtt[samples_num*50/100]*1000/clock_mhz,
                rtt[samples_num*75/100]/clock_mhz, rtt[samples_num*75/100]*1000/clock_mhz,
                rtt[samples_num*90/100]/clock_mhz, rtt[samples_num*90/100]*1000/clock_mhz, 
                rtt[samples_num*95/100]/clock_mhz, rtt[samples_num*95/100]*1000/clock_mhz,
                rtt[samples_num*99/100]/clock_mhz, rtt[samples_num*99/100]*1000/clock_mhz,
                rtt[samples_num*999/1000]/clock_mhz,rtt[samples_num*999/1000]*1000/clock_mhz, 
                rtt[samples_num-1]/clock_mhz,  rtt[samples_num-1]*1000/clock_mhz);
                munlock(rtt, sizeof(rtt));
	fclose(f);
         return 0;
}


