// Copyright (c) 2015, Matthew Chapman <matthew.chapman@exablaze.com> 
// Copyright (c) 2016, Noa Zilberman, Matthew Grosvenor
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// * Redistributions of source code must retain the above copyright notice, this
//   list of conditions and the following disclaimer.
//
// * Redistributions in binary form must reproduce the above copyright notice,
//   this list of conditions and the following disclaimer in the documentation
//   and/or other materials provided with the distribution.
//
// * Neither the name of the project, the copyright holder nor the names of its
//  contributors may be used to endorse or promote products derived from
//   this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <asm/msr.h>
#include <math.h>

#define USE_RDTSCP
#define THRESHOLD_USEC  1
#define TIME_SEC        1
#define MAXSAMPLES      1000000
#define DEFAULT_FILE    "out.txt"
#define DEFAULT_TYPE    "accurate"
#define CNT_TH		10

#ifndef rdtscll
#define rdtscll(val) do { \
     unsigned int __a,__d; \
     asm volatile("rdtsc" : "=a" (__a), "=d" (__d)); \
     (val) = ((unsigned long)__a) | (((unsigned long)__d)<<32); \
} while(0)
#endif
#ifndef rdtscpll
#define rdtscpll(val, aux) do { \
        unsigned long __a, __d; \
        asm volatile (".byte 0x0f,0x01,0xf9" : "=a" (__a), "=d" (__d), "=c" (aux)); \
        (val) = (__d << 32) | __a; \
} while (0)
#endif


#ifdef USE_RDTSCP
#define do_rdtscp(val, cpu) rdtscpll(val, cpu)
#else
#define do_rdtscp(val, cpu) do { rdtscll(val); cpu=0; } while (0)
#endif


#define CPU_MHZ_FILE        "/proc/cpuinfo"
#define CPU_MHZ_PREFIX        "cpu MHz\t\t: "

unsigned int get_mhz(void)
{
    FILE *fp;
    char line[1024];
    unsigned int clock_mhz = 0;

    fp = fopen("/proc/cpuinfo", "r");
    if (!fp)
        return 0;

    while (fgets(line, sizeof(line), fp))
    {
        if (strncmp(line, CPU_MHZ_PREFIX, sizeof(CPU_MHZ_PREFIX)-1) == 0)
        {
            clock_mhz = atoi(line+sizeof(CPU_MHZ_PREFIX)-1);
            break;
        }
    }
    fclose(fp);
    return clock_mhz;
}

volatile int done = 0;
void handle_sigalrm(int signo)
{
    done = 1;
}

static int compare_ull(const void *pa, const void *pb)
{
        unsigned long long *a = *(unsigned long long **)pa;
        unsigned long long *b = *(unsigned long long **)pb;
        return (a[0] < b[0]) ? -1 : (a[0] > b[0]) ? +1 : 0;
}

int main(int argc, char **argv)
{
    unsigned long long tsc, tsc2, last, start, threshold;
    unsigned int i, samples = 0, clock_mhz;
    unsigned int cpu, cpu2, lastcpu;
    unsigned long long **buffer;
    unsigned int samples_num,time_sec,threshold_usec,threshold_99;
    unsigned long long samples_th[10];
    unsigned long long samples_count[10],samples_time[10];
    unsigned long long global_count=0,global_time=0;
    int flag;
    char *fname,*jtype;

    clock_mhz = get_mhz();
    if (!clock_mhz)
    {
        fprintf(stderr, "Could not determine clock rate from " CPU_MHZ_FILE "\n");
        return 0;
    }

     time_sec=TIME_SEC;
     threshold_usec=THRESHOLD_USEC;
     samples_num=MAXSAMPLES;
     fname=DEFAULT_FILE;
     jtype=DEFAULT_TYPE;
     samples_th[0]=CNT_TH*clock_mhz/1000;
     samples_count[0]=0;
     samples_time[0]=0;
if (argc == 1) {
  printf("\nusage: jitter -t <accurate/full/novice> -o <output file> -u <threshold in usec> -d <runtime in sec> -s <max samples>\n>> accurate - measure events accurately, but may miss events\n>> full - measure all events, but include code-induced events\n>> novice - same as full, but without locking memory (mlock)");
  abort;
}
      


while ((flag = getopt (argc, argv, "t:o:d:u:s:n:")) != -1)
    switch (flag)
      {
      case 't':
        jtype = optarg;
        break;
      case 'o':
        fname = optarg;
        break;
      case 'd':
        time_sec = atoi(optarg);
        break;
      case 'u':
	threshold_usec = atoi(optarg);
        break;
      case 's':
         samples_num = atoi(optarg);
         break;
      case 'n':
         samples_th[0]= atoi(optarg)*clock_mhz/1000;
      default:
          printf("\nusage: jitter -t <accurate/full/novice> -o <output file> -u <threshold in usec> -d <runtime in sec> -s <max samples>\n >> accurate - measure events accurately, but may miss events\n>> full - measure all events, but include code-induced events\n>> novice - same as full, but without locking memory (mlock)");
          abort; 
      }

    printf("Test parameters: output %s type %s Threshold %d time %d samples %d\n",fname,jtype,threshold_usec,time_sec,samples_num);

    for (i=1;i<10;i++) {
        samples_th[i]=samples_th[0]+samples_th[0]*pow(10,i-1)/10;
        samples_count[i]=0;
        samples_time[i]=0;
    }
    buffer = malloc (samples_num * sizeof(long long int*));
    if (strcmp(jtype,"full")==0) {
           /* lock this buffer into RAM */
           if (mlock(buffer,sizeof(buffer)))
           {
            free(buffer);
            return (0);
           }
        } 
    for (i=0; i<samples_num; i++) {
    	buffer[i] = malloc (2 * sizeof(long long int));
        buffer[i][0] = 0;
        buffer[i][1] = 0;
        
    }

    threshold = threshold_usec * clock_mhz;

    signal(SIGALRM, handle_sigalrm);
    alarm(time_sec);

    do_rdtscp(tsc, cpu);
    start = tsc;
    last=tsc;
    lastcpu=cpu;
   printf("here we go!\n");  
  if (strcmp(jtype,"accurate")==0) {
    //Polling mode
    	while (!done)
    	{
    
	        do_rdtscp(tsc, cpu);
	        do_rdtscp(tsc2,cpu2);
 	       if ((tsc2 - tsc > threshold) && (cpu == cpu2))
	        {
   	   	        if ((samples >= MAXSAMPLES) | (samples>=samples_num))
        	        break;
	            buffer[samples][0] = tsc2-tsc;
    	        buffer[samples++][1] = tsc2-start;
        	}
                global_count++;
                global_time+=tsc2-tsc;
                for (i=0;i<10;i++){
                      if (tsc2-tsc > samples_th[i]) {
                          samples_count[i]++;
                          samples_time[i]+=tsc2-tsc;
                      }
                }
        }
     }
     else //full or novice mode
     {
           while (!done)
	    {  
			do_rdtscp(tsc, cpu);
        	if ((tsc - last > threshold) && (cpu == lastcpu))
        	{
            	if ((samples >= MAXSAMPLES) | (samples >= samples_num))
                	break;
		        buffer[samples][0] = tsc-last;
		        buffer[samples++][1] = tsc-start;
        	}
      		last = tsc;
        	lastcpu = cpu;
        }
    }

    printf("now save...\n");
    FILE *f=fopen(fname,"w");
    fprintf(f,"Test parameters: type %s Threshold %d time %d samples %d\n",jtype,threshold_usec,time_sec,samples_num);



   for (i = 0; i < samples; i++){
                fprintf(f,"%llu\t%llu\t%llu\n", buffer[i][0] / clock_mhz,buffer[i][0]*1000/clock_mhz, buffer[i][1] / clock_mhz);    
       }
        qsort(buffer, samples, sizeof(buffer[0]), compare_ull);
        fprintf(f,"\ncpu clock frequency\t\t%u\n",clock_mhz);
        fprintf(f,"number of events:\t%u\n",samples);
        fprintf(f,"run time:	\t%llu seconds\n\n",(tsc-start)/(clock_mhz*1000000));
        if (samples>0) {
            fprintf(f,"min\t\t%llu\t\t%llu\n" "1%%\t\t%llu\t\t%llu\n" "5%%\t\t%llu\t\t%llu\n" "10%%\t\t%llu\t\t%llu\n" "25%%\t\t%llu\t\t%llu\n" "median\t\t%llu\t\t%llu\n" "75%%\t\t%llu\t\t%llu\n" "90%%\t\t%llu\t\t%llu\n" "95%%\t\t%llu\t\t%llu\n" "99%%\t\t%llu\t\t%llu\n" "99.9%%\t\t%llu\t\t%llu\n"  "max\t\t%llu\t\t%llu\n",
                buffer[0][0]/clock_mhz, buffer[0][0]*1000/clock_mhz, 
                buffer[samples*1/100][0]/clock_mhz, buffer[samples*1/100][0]*1000/clock_mhz,
                buffer[samples*5/100][0]/clock_mhz, buffer[samples*5/100][0]*1000/clock_mhz,
                buffer[samples*10/100][0]/clock_mhz, buffer[samples*10/100][0]*1000/clock_mhz, 
                buffer[samples*25/100][0]/clock_mhz, buffer[samples*25/100][0]*1000/clock_mhz,
                buffer[samples*50/100][0]/clock_mhz, buffer[samples*50/100][0]*1000/clock_mhz,
                buffer[samples*75/100][0]/clock_mhz, buffer[samples*75/100][0]*1000/clock_mhz,
                buffer[samples*90/100][0]/clock_mhz, buffer[samples*90/100][0]*1000/clock_mhz, 
                buffer[samples*95/100][0]/clock_mhz, buffer[samples*95/100][0]*1000/clock_mhz,
                buffer[samples*99/100][0]/clock_mhz, buffer[samples*99/100][0]*1000/clock_mhz,
                buffer[samples*999/1000][0]/clock_mhz, buffer[samples*999/1000][0]*1000/clock_mhz,
                buffer[samples-1][0]/clock_mhz,  buffer[samples-1][0]*1000/clock_mhz);
          }

          fprintf(f,"\n\n Global statistics: Total Events\t%llu\tRun time[us]\t%llu \n",global_count,global_time/clock_mhz);
          for (i=0;i<10;i++) {
           fprintf(f,"Threshold[ns]\t%llu\t\tEvents\t%llu\t\tTime[us]\t%llu\n",1+samples_th[i]*1000/clock_mhz,samples_count[i],samples_time[i]/clock_mhz);
          }
          fprintf(f,"successfully ended\n\n");
          
          fclose(f);
          if (strcmp(jtype,"full")==0) {
                   munlock(buffer, sizeof(buffer));
          }
          return 0;
}

