// Copyright (c) 2015, Matthew Chapman <matthew.chapman@exablaze.com> 
// Copyright (c) 2016, Noa Zilberman, Matthew Grosvenor
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// * Redistributions of source code must retain the above copyright notice, this
//   list of conditions and the following disclaimer.
//
// * Redistributions in binary form must reproduce the above copyright notice,
//   this list of conditions and the following disclaimer in the documentation
//   and/or other materials provided with the distribution.
//
// * Neither the name of the project, the copyright holder nor the names of its
//  contributors may be used to endorse or promote products derived from
//   this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

#define _GNU_SOURCE
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <sched.h>
#include <pthread.h>
#include "tsc.h"

#include "impl.h"


#define CPU_MHZ_FILE        "/proc/cpuinfo"
#define CPU_MHZ_PREFIX        "cpu MHz\t\t: "

unsigned int get_mhz(void)
{
    FILE *fp;
    char line[1024];
    unsigned int clock_mhz = 0;

    fp = fopen("/proc/cpuinfo", "r");
    if (!fp)
        return 0;

    while (fgets(line, sizeof(line), fp))
    {
        if (strncmp(line, CPU_MHZ_PREFIX, sizeof(CPU_MHZ_PREFIX)-1) == 0)
        {
            clock_mhz = atoi(line+sizeof(CPU_MHZ_PREFIX)-1);
            break;
        }
    }
    fclose(fp);
    return clock_mhz;
}

static void pin_cpu(int cpu)
{
        cpu_set_t cpu_set;
        CPU_ZERO(&cpu_set);
        CPU_SET(cpu, &cpu_set);
        if (sched_setaffinity(0, sizeof(cpu_set), &cpu_set) == -1)
                perror("sched_setaffinity");
}

static int compare_ull(const void *pa, const void *pb)
{
        unsigned long long a = *(const unsigned long long *)pa;
        unsigned long long b = *(const unsigned long long *)pb;
        return (a < b) ? -1 : (a > b) ? +1 : 0;
}

int main(int argc, char *argv[])
{
        unsigned long long *pingtimes;
        unsigned long long start, end;
        unsigned int clock_mhz;
#if THREADS
        pthread_t thread;
#endif
        int child_pid;
        unsigned long long i;
	int parent_cpu = 0;
	int child_cpu = 1;

        pingtimes=malloc(SAMPLES*sizeof(unsigned long long*));
    if (mlock(pingtimes,sizeof(pingtimes)))
           {
            free(pingtimes);
            return (0);
           }


    clock_mhz = get_mhz();
    if (!clock_mhz)
    {
        fprintf(stderr, "Could not determine clock rate from " CPU_MHZ_FILE "\n");
        return 0;
    }


	if (argc >= 3)
	{
		parent_cpu = atoi(argv[1]);
		child_cpu = atoi(argv[2]);
	}

        init();

#if THREADS
        pthread_create(&thread, NULL, (void *(*)(void *))pong_thread, NULL);
#else
        child_pid = fork();
        switch (child_pid)
        {
                case -1:
                        perror("fork");
                        exit(1);

                case 0: /* child */
                        pin_cpu(child_cpu);
                        pong_thread();

                default:
                        break;
        }
#endif

        pin_cpu(parent_cpu);
        ping_thread_init();
        ping();

        for (i = 0; i < SAMPLES; i++)
        {
                rdtscll(start);
                ping();
                rdtscll(end);
                pingtimes[i] = end - start;

#if USLEEP
                usleep(USLEEP);
#endif
#if 0
                write(3, pingtimes, sizeof(pingtimes));
                {
                int j;
                for (j = 0; j < 10000000; j++)
                        asm volatile("");
                }
#endif
        }

#if !THREADS
        kill(child_pid, SIGTERM);
#endif

        qsort(pingtimes, SAMPLES, sizeof(pingtimes[0]), compare_ull);
        printf("cpu clock frequency\t\t%u\n",clock_mhz);
        printf("min\t\t%llu\t\t%llu\n" "1%%\t\t%llu\t\t%llu\n" "5%%\t\t%llu\t\t%llu\n" "10%%\t\t%llu\t\t%llu\n" "25%%\t\t%llu\t\t%llu\n" "median\t\t%llu\t\t%llu\n" "75%%\t\t%llu\t\t%llu\n" "90%%\t\t%llu\t\t%llu\n" "95%%\t\t%llu\t\t%llu\n" "99%%\t\t%llu\t\t%llu\n" "99.9%%\t\t%llu\t\t%llu\n"  "max\t\t%llu\t\t%llu\n",
                pingtimes[0]/clock_mhz, pingtimes[0]*1000/clock_mhz, 
                pingtimes[SAMPLES*1/100]/clock_mhz, pingtimes[SAMPLES*1/100]*1000/clock_mhz,
                pingtimes[SAMPLES*5/100]/clock_mhz, pingtimes[SAMPLES*5/100]*1000/clock_mhz,
                pingtimes[SAMPLES*10/100]/clock_mhz, pingtimes[SAMPLES*10/100]*1000/clock_mhz, 
                pingtimes[SAMPLES*25/100]/clock_mhz, pingtimes[SAMPLES*25/100]*1000/clock_mhz,
                pingtimes[SAMPLES*50/100]/clock_mhz, pingtimes[SAMPLES*50/100]*1000/clock_mhz,
                pingtimes[SAMPLES*75/100]/clock_mhz, pingtimes[SAMPLES*75/100]*1000/clock_mhz,
                pingtimes[SAMPLES*90/100]/clock_mhz, pingtimes[SAMPLES*90/100]*1000/clock_mhz, 
                pingtimes[SAMPLES*95/100]/clock_mhz, pingtimes[SAMPLES*95/100]*1000/clock_mhz,
                pingtimes[SAMPLES*99/100]/clock_mhz, pingtimes[SAMPLES*99/100]*1000/clock_mhz,
                pingtimes[SAMPLES*999/1000]/clock_mhz,pingtimes[SAMPLES*999/1000]*1000/clock_mhz, 
                pingtimes[SAMPLES-1]/clock_mhz,  pingtimes[SAMPLES-1]*1000/clock_mhz);
                munlock(pingtimes, sizeof(pingtimes));

         return 0;
}

