/*  -*- Mode: C;  -*-
 * File: jroute.c
 * Author: James Hall (jch1003@cl.cam.ac.uk)
 * Copyright (C) University of Cambridge Computer Laboratory, 1998
 **~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ** PACKAGE:
 **
 ** FUNCTION:
 **
 ** HISTORY:
 ** Created: Tue Aug 11 16:05:22 1998 (jch1003)
 ** Last Edited: Tue Nov  3 16:19:51 1998 By James Hall
 **
    $Log: jroute.c,v $
    Revision 1.14  1998/11/11 13:50:28  iap10
    *** empty log message ***

    Revision 1.13  1998/11/11 12:52:52  jch1003
    *** empty log message ***

    Revision 1.12  1998/10/27 10:54:27  iap10
    *** empty log message ***

    Revision 1.11  1998/10/21 10:16:03  jch1003
    *** empty log message ***

    Revision 1.10  1998/10/20 09:35:24  iap10
    *** empty log message ***

    Revision 1.9  1998/10/19 20:26:07  jch1003
    *** empty log message ***

    Revision 1.8  1998/10/12 10:30:42  iap10
    *** empty log message ***

    Revision 1.7  1998/10/09 15:07:52  iap10
    *** empty log message ***

    Revision 1.6  1998/10/09 14:38:21  iap10
    ..

    Revision 1.5  1998/10/09 12:11:27  jch1003
    *** empty log message ***

    Revision 1.4  1998/09/01 15:39:46  iap10
    *** empty log message ***

    Revision 1.3  1998/08/14 11:46:32  jch1003
    *** empty log message ***

 **~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 */

#ifdef linux
#include "linux-tweaks.h"
#endif

#include <pthread.h>
#include <stdio.h>
#include <pwd.h>
#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <errno.h>  
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

#include <getopt.h>

#include "list.h"
#include "jroute.h"
#include "jroute_threads.h"

char *prog;
char errbuf[132];
int verbose = 0;
int simple_output = 0;
int sort = 0;
int naddr = 0;			/* # addresses if in input preamble */
int naddr_done = 0;		/* #addresses actually getrouted */

int s;			        /* (icmp) socket file descriptor */  
int main_pid;

int probe_type = PROBE_TCP;             

int (*get)();			/* input function */

void 
usage()
{
  fprintf(stderr, "%s [-v(erboser)] [-q(uieter)] [-Q(uite like traceroute)] [-t(hreads)]\n\t [-m(ax-ttl)] [-f(irst-ttl)] [-r(oute)] [-o(utput file] [-s(ort)] \n\t[-S(et src address)] [-I set interface] [-g(ive up after n timeouts)] \n\t[-b generate probes and replies at most nKb/s]\n\t[-B generate src routed packets at most n/s] [-h(elp)] <file/host>\n", prog);
}

void 
help()
{
  usage();
  printf("\n");
  printf("Input lines have form \"(addr1+addr2+...+)host(/f/m$label) <CR>\"\n");
  printf("\twhere:\taddrs form a list of loose source routed requirements\n");
  printf("\t\tf/m = min/max probe ttl to employ\n");
  printf("\t\tand optional label will be quoted with result\n");
  printf("\t\tlsr list, either/both of /f /m and label are optional\n");
  printf("\tif any option is not supplied the current default is used\n\n");

  printf("flags \"-vqmfrbBh<CR>\" may be inserted anywhere in the input stream and will \n\tact/set current defaults from that point\n");
  printf("\t (if preceeded by a double \"--\" will take effect immediately)\n");
  printf("Current defaults can be cleared by entering flag followed by \"-\"\n");
  printf("\ti.e. to set source routing from current point in input:\n");
  printf("\t\t\"-raddr1+addr2+...\"\n");
  printf("\tand to clear it:\n");
  printf("\t\t\"-r-\"\n");
  printf("for more look at the source\n\n");
  printf("\n");
  return;
}



void 
opterror(char *msg)
{
  fprintf(stderr, "%s: %s\n", prog, msg);
  usage();
  exit (1);
}

void 
error(char *msg)
{
  fprintf(stderr, "%s: ", prog);
  if (errno)
    perror(msg);
  else
    fprintf(stderr, "%s\n", msg);
  exit (1);
}

void 
error2(char *msg1, char *msg2)
{
  fprintf(stderr, "%s: %s: ", prog, msg1);
  if (errno)
    perror(msg2);
  else
    fprintf(stderr, "%s\n", msg2);
  exit (1);
}

FILE 
*openf(char *fnm)
{
  FILE *f;

  if ((f = fopen(fnm, "r")) == NULL)
    error2(fnm, "input file - open error");

  return f;
}

/*
 * Interpret cl type arg. found in the input stream. 
 */
void 
command(char *comm, char *allowed)
{
  int i;
  TRC(fprintf(stderr, "command %s\n", comm);)
  if (*comm == '-')		/* double dash - already done */
    return;
  while (strlen(comm))
    {
      if (!strchr(allowed, *comm))
	{
	  comm++;
	  continue;
	}
      switch (*comm)
	{
	case 'v': 
	  if (nthreads == 1)
	    {
	      verbose++; 
	      simple_output = 0; 
	    }
	  comm++; 
	  break;
	case 'q': 
	  if (nthreads == 1)
	    {
	      verbose = MAX(0, verbose-1); 
	      simple_output = 0;
	    }
	  comm++; 
	  break;
	case 'Q': 
	  if (nthreads == 1)
	    { 
	      verbose = 0; 
	      simple_output = 1; 
	      }
	  comm++; 
	  break;
	case 'm': 
	  con_def.ttl_max = strtol(comm+1, &comm, 0);
	  con_def.ttl_max = MIN(con_def.ttl_max, TTL_MAX);
	  if (!con_def.ttl_max)
	    con_def.ttl_max = TTL_MAX_DEF;
	  break;
	case 'f': 
	  con_def.ttl_min = strtol(comm+1, &comm, 0);
	  con_def.ttl_min = MAX(con_def.ttl_min, TTL_MIN_DEF);
	  if (!con_def.ttl_min)
	    con_def.ttl_min = TTL_MIN_DEF;
	  break;
	case 'g':
	  giveup = strtol(comm+1, &comm, 0);
	  break;
	case 'r':
	  comm++;
	  if (*comm == '-')
	    {
	      con_def.route.num = 0;
	    }
	  else
	    {
	      if (!parseroute(&con_def.route, comm))
		{
		  fprintf(stderr, "# ignoring default lsr setting: %s\n", 
			  comm);
		  con_def.route.num = 0;
		}
	      while (*comm != ' ')
		comm++;
	    }
	  break;
	case 'b': maxrate = strtol(comm+1, &comm, 0); break;
	case 'B': maxrate = -strtol(comm+1, &comm, 0); break;
	//case 'c': add_context(comm+1); break;
	case 'h': help(); comm++; break;
		
	}
    }
  return;
}

/* 
 * Get input on line basis - used when input from file or console.
 */
int 
get_stdio(char *buf, FILE *i)
{
  char *nlp;

  if (feof(i))
    {
      return 0;
    }
  else
    {
      if (fgets(buf, MAX_INDATA_SZ, i) == NULL)
	{
	  if (feof(i))
	    {
	      return 0;
	    }
	  else
	    {
	      fprintf(stderr, "%s: input read error\n", prog);
	      exit (1);
	    }
	}
      nlp = strchr(buf, '\n');
      *nlp = '\0';
    }
  return 1;
  
}

/* 
 * read from telnet connection 
 */
int tread(int f, char *buf, int in, int out, int block) 
{
  int i;
  int to_read;
  int nread;
  int got = 0;
  int end = TNET_INBUF_SZ;
  int mark = in;
  int on = 1;

  static state_t state = STATE_NORMAL;
  
  if (!block && ioctl(f, FIONBIO, (char*)&on) < 0)
    error("FIONBIO unblock");
  
  to_read = in < out ? out - in : end - in;
  
  nread  = read(f, &buf[in], to_read);  
  if (nread < 0)
    {
      if (! block && errno == EWOULDBLOCK)
	nread = 0;
      else
	error("read");
    }
  else if (nread == 0)
    {
      //error("lost client");
;
    }
  else
    {
      got += nread;
      in = (in + nread)%TNET_INBUF_SZ;
    }

  on = 0;
  if (!block && ioctl(f, FIONBIO, (char*)&on) < 0)
    error("FIONBIO block");

  /* scan input for telnet state machine */

  for (i = got; i > 0; mark = (++mark)%TNET_INBUF_SZ, i--) 
    {
      unsigned char c = buf[mark];
      switch(state) 
	{
	case STATE_NORMAL:
	  if (c != IAC)
	    {
	      TRC(printf("%02x ", c);)
	      fflush(stdout);
	      /* check for commands to apply immediately */
	      if (c == '-' && buf[mark+1] == '-')
		{
		  char cbuf[TNET_INBUF_SZ], *cp = cbuf;
		  int cindx = mark+2; /* start after dashes */
		  while (buf[cindx] != '\x0d')
		    *(cp++) = buf[cindx++];
		  *cp = '\0';
		  command(cbuf, "mfvqQrbBhg");
		}
	    }
	  else
	    {
	      /* shift into the command state */
	      state = STATE_IAC;
	    }
	  break;
	  
	case STATE_IAC:
	  switch(c) 
	    {
	    case IP:
	      //printf("interrupt!\n");
	      exit(0);
	      break;
	      
	    case SE:
	    case NOP:
	    case DMARK:
	    case BRK:
	    case AO:
	    case AYT:
	    case EC:
	    case EL:
	    case GA:
	    case SB:
	      printf("ignoring option %02x\n", c);
	      break;
	      
	    case WILL:
	      printf("WILL ");
	      state = STATE_OPT;
	      break;
	    case WONT:
	      printf("WONT ");
	      state = STATE_OPT;
		  break;
	    case DO:
	      printf("DO   ");
	      state = STATE_OPT;
	      break;
	    case DONT:
	      printf("DONT  ");
	      state = STATE_OPT;
	      break;
	      
	    default:
	      printf("unhandled IAC %02x\n", c);
	      break;
	    }
	  if (state == STATE_IAC)
	    state = STATE_NORMAL;
	  break;
	  
	  
	case STATE_OPT:
	  switch(c) 
	    {
	    case TMARK:
	      {
		unsigned char reply[] = {IAC, WILL, TMARK};
		/* send WILL TMARK */
		write(2, reply, sizeof(reply));
		printf("tmark\n");
		state = STATE_NORMAL;
		break;
	      }
	      
	    default:
	      printf("ignoring option code %02x\n", c);
	      state = STATE_NORMAL;
	      break;
	    }
	}
      
    }
  return got;
} 

/* 
 * Get input on raw byte basis - used when running as inetd. driven server.
 * Handle telnet codes and implement basic telnet state machine, deliver 
 * input to caller on line basis. Telnet machine courtesy of Austin D.
 */
int 
get_tnet(char *obuf, FILE *infile)
{
  int i = 0;
  int nread;
  int gotline = 0;
  int block = BLOCK;

  static unsigned char ibuf[TNET_INBUF_SZ];
  static int in = 0;
  static int out = 0;
  static int bytes = 0;

  do 
    {
      
      /* First attempt to get a line to caller */
      
      for (; bytes; out = (++out)%TNET_INBUF_SZ, --bytes)
	{
	  switch (ibuf[out])
	    {
	    case '\x0a':	/* LF */
	      gotline++;
	      obuf[i] = '\0';
	    case '\x0d':	/* DROP THROUGH - clear CR */
	      break;
	    case '\x04':		/* EOF */
	      printf("#Closing connection\n");
	      return 0;
	      break;
	    default:
	      obuf[i++] = ibuf[out];
	      if (i == MAX_INDATA_SZ)
		error("telnet input line too long");
	      break;
	    }
	  if (gotline)
	    {
	      out = (++out)%TNET_INBUF_SZ;
	      --bytes;
	      break;
	    }
	}
      
      /* now attempt a read in, don't block if already got a line for caller */
      
      nread = tread(0, ibuf, in, out, gotline ? NO_BLOCK : BLOCK);
      bytes += nread;
      in = (in + nread)%TNET_INBUF_SZ;

    } while (!gotline);

  return 1;
}

  

int 
main(int argc, char **argv)
{
  char *fnm;
  char *cp, opt;
  FILE *infile = NULL;
  char inbuf[MAX_INDATA_SZ];
  char ofilenm[PATH_MAX];
  int n, i;
  char *source = NULL, *device = NULL;
  struct sockaddr_in wherefrom;	/* Who we are */
  struct ifaddrlist *al;
  struct hostinfo *hi, *hi_ss;
  uint32 *ap;
  addr_un_t to;
  src_route_t route_try;
  char clbuf[CL_BUFSZ], *p = clbuf;
  struct passwd *pwd;
  time_t tm;
  char hname[50];
  struct timeval finish;
  struct protoent *pe;

  if ((cp = strrchr(argv[0], '/')) != NULL)
    prog = cp + 1;
  else
    prog = argv[0];

  main_pid = getpid();

  *p = '\0';
  for (i = 1; i < argc; i++)
    p+= sprintf(p, "%s ", argv[i]);
  printf("# %s: invoked %s\n", prog, clbuf);

  pwd = getpwuid(getuid());
  if (pwd)
    {
      printf("# by %s ", pwd->pw_gecos);
       if (!gethostname(hname, 50))
	 {
	    printf("@%s ", hname);
	  }    
     }
  else
    {
      printf("# ");
    }
  
  if ((tm = time(NULL)) == -1)
    error("time error");
  printf("%s", ctime(&tm));

  opterr = 0;
  route.num = 0U;
  get = get_stdio;
  //clist_init(clist);

  ofile = stdout;		/* default */

  while ((opt = getopt(argc, argv, "vqQsDXt:m:f:r:o:S:I:g:b:B:h")) != EOF)
    switch (opt)
      {
      case 'v': verbose++; simple_output = 0; break;
      case 'q': verbose--; simple_output = 0; break;
      case 'Q': simple_output = 1; break;
      case 's': sort++; break;
      case 't': nthreads = atoi(optarg); break;
      case 'm': con_def.ttl_max = atoi(optarg); break;
      case 'f': con_def.ttl_min = atoi(optarg); break;
      case 'r': 
	if (!parseroute(&con_def.route, optarg))
	error2("lsr parse fail", optarg); 
	break;
      case 'o': strcpy(ofilenm, optarg); ofile = open_ofile(ofilenm, 0); break;
      case 'g': giveup = atoi(optarg); break;
      case 'X':			/* drop through */
	  giveup = 1;

      case 'D': get = get_tnet; break;
      case 'S':
	/*
	 * set the ip source address of the outbound
	 * probe (e.g., on a multi-homed host).
	 */
	source = optarg;
	break;
      case 'I':
	device = optarg;
	break;
      case 'b': maxrate = atoi(optarg); break;
      case 'B': maxrate = -atoi(optarg); break;
      case 'h': help(); return 0; break;
      case '?': opterror("Unknown option"); break;
      case ':': opterror("Missing parameter"); break;
      default: opterror(""); break;
      }

  if (getuid() && geteuid())
    error("Sorry can't help you - must have root privileges.");

  /* get icmp socket */
  if ((pe = getprotobyname("icmp")) == NULL)
    error("icmp protocol unknown!");
  if ((s = socket(AF_INET, SOCK_RAW, pe->p_proto)) < 0)
    error("icmp socket");

  if (argc  == optind)
    error("no input file or host specified");

  strncpy(inbuf, argv[optind], MAX_INDATA_SZ);
  cp = strchr(inbuf, ':');
  if (cp)
    *cp = '\0';

  if (!parseroute(&route_try, inbuf)) /* input must be a file */
    {
      //fprintf(stderr, "# - assuming \"%s\" is an input file\n", inbuf);
      fnm = argv[optind];
      if (*fnm == '-')
	{
	  fprintf(stderr, "# - using std in\n");
	  infile = stdin;
	  //if (get != get_tnet)
	    //nthreads = 1;
	}
      else
	{
	  fprintf(stderr, "# - assuming \"%s\" is an input file\n", inbuf);
	  infile = openf(fnm);
	}
    }
  else				/* one dest. input as c.l. arg. */
    {
      naddr = 1;
      nthreads = 1;
      if (simple_output)
	verbose = 0;
      else	
	verbose = MAX(verbose, 1);
      if (route_try.num == 1)
	{
	  to.addrint = route_try.rt[0].addrint;
	  route_try.num = 0;
	}
      else
	{
	  to.addrint = route_try.rt[route_try.num-1].addrint;
	  route_try.rt[route_try.num-1].addrint = 0U;
	  route_try.num--;
	  if (route_try.num > con_def.route.num)
	    con_def.route = route_try;
	}
    }

  /* don't produce gibberish */
  if (verbose || simple_output)
    nthreads = 1;

  if (con_def.ttl_min < TTL_MIN_DEF || con_def.ttl_min > TTL_MAX)
    {
      fprintf(stderr, "%s: minimum ttl %d specified\n",
	      prog, con_def.ttl_min);
      return 1;
    }

  if (con_def.ttl_max > TTL_MAX || con_def.ttl_max < TTL_MIN_DEF)
    {
      fprintf(stderr, "%s: maximum ttl %d specified - max is %d\n",
	      prog, con_def.ttl_max, TTL_MAX);
      return 1;
    }

  if (nthreads > N_THREADS_MAX)
    {
      fprintf(stderr, "%s: %d threads specified - max is %d\n",
	      prog, nthreads, N_THREADS_MAX);
      return 1;
    }
  else if (nthreads <= 0)
    {
      fprintf(stderr, "%s: %d threads specified\n",
	      prog, nthreads);
      return 1;
    }

  /* Get the interface address list */
  n = ifaddrlist(&al, errbuf);
  if (n < 0) 
    error2("ifaddrlist", errbuf);
  if (n == 0) 
    error("Can't find any network interfaces");
  
  /* Look for a specific device */
  if (device != NULL) 
    {
      for (i = n; i > 0; --i, ++al)
	if (strcmp(device, al->device) == 0)
	  break;
      if (i <= 0)
	error2("Can't find interface", device);
    }
  
  /* Determine our source address */
  if (source == NULL) 
    {
      /*
       * If a device was specified, use the interface address.
       * Otherwise, use the first interface found.
       * Warn if there are more than one.
       */
      setsin(&wherefrom, al->addr);
      if (n > 1 && device == NULL) 
	{
	  fprintf(stderr,
		  "# %s: Warning: Multiple interfaces found; using %s @ %s\n",
		  prog, inet_ntoa(wherefrom.sin_addr), al->device);
	}
    } 
  else 
    {
      hi = gethostinfo(source);
      if (hi == NULL)
	error2("no such host as", source);
      source = hi->name;
      hi->name = NULL;
      if (device == NULL) 
	{
	  /*
	   * Use the first interface found.
	   * Warn if there are more than one.
	   */
	  setsin(&wherefrom, hi->addrs[0]);
	  if (hi->n > 1)
	    fprintf(stderr,
		    "%s: Warning: %s has multiple addresses; using %s\n",
		    prog, source, inet_ntoa(wherefrom.sin_addr));
	} 
      else
	{
	  /*
	   * Make sure the source specified matches the
	   * interface address.
	   */
	  for (i = hi->n, ap = hi->addrs; i > 0; --i, ++ap)
	    if (*ap == al->addr)
	      break;
	  if (i <= 0)
	    {
	      fprintf(stderr,
		      "%s: %s is not on interface %s\n",
		      prog, source, device);
	      exit(1);
	    }
	  setsin(&wherefrom, *ap);
	}
      freehostinfo(hi);
    }

  fflush(stdout);
  setlinebuf(stdout);

  if (gettimeofday(&start, NULL) != 0)
    error("gettimeofday - start");
  
  if (infile)
    {
      if (sort)
	do_sorted(&wherefrom, infile, nthreads);
      else 
	do_unsorted(&wherefrom, infile, nthreads);
    }
  else
    {
      do_singleshot(&wherefrom, to, cp, nthreads);
    }

  if (gettimeofday(&finish, NULL) != 0)
    error("gettimeofday - finish");

  if (ofile != stdout)
    {
      char b[MAX_INDATA_SZ];

      fclose(ofile);
      ofile = open_ofile(ofilenm, 1);
      while (get_stdio(b, ofile))
	{
	  printf("%s\n", b);
	}
    }

  report(&start, &finish);

  fflush(stdout);

  return 0;
}
	  

int 
do_singleshot(struct sockaddr_in *wherefrom, addr_un_t to, char *cp, 
	       int nthreads)
{
  addr_train_t at;
  all_addrs = &at;
  train_bufsz = 1;

  at.dest.addrint = to.addrint;
  if (cp == NULL) 
    {
      fprintf(stderr, "%s: No port specified - default to 80\n", 
	      prog);
      cp = ":80";
    }
  at.port = atoi(cp+1);
  at.con = con_def;
  at.label[0] = '\0';

  sort = 1;
  threads_init(wherefrom, nthreads, NULL);
  rthread_inform();
  
  threads_finish(nthreads);

  return 0;
}

#define STDIN (infile == stdin)

int 
do_unsorted(struct sockaddr_in *wherefrom, FILE *infile, int nthreads)
{
  int read;
  int started = 0;
  uint line = 0;
  char inbuf[MAX_INDATA_SZ];
  
  train_bufsz = nthreads*ADDR_POOL_SZ_FACT;
  
  if ((all_addrs = (addr_train_t *)malloc(train_bufsz*sizeof(addr_train_t))) 
      == NULL)
    error("addr train malloc");
  
  threads_init(wherefrom, nthreads, infile);
  
  while (read = get(inbuf, infile))
    {
// fprintf(stderr, "INBUF %s\n", inbuf); sleep(1);
      
      line++;
      if (!strlen(inbuf))
	continue;

      if (*inbuf == '#')
	{
	  fprintf(ofile, "%s\n", inbuf);
	  continue; 
	}
      else if (*inbuf == '-')

	{
	  command(inbuf+1, "vqQmfrbBhg");
	  TRC(fprintf(stderr, "# command \"%s\" line %u\n", inbuf+1, line);)
	  continue;
	}
      else if (strstr(inbuf, "addresses"))
	{
	  fprintf(stderr, "# %s\n", inbuf);
	  if (naddr)
	    fprintf(stderr,
		    "# another notification of #addresses found (was %d now %d) line %u\n", naddr, naddr + atoi(inbuf), line);
	  naddr += atoi(inbuf);
	  continue;
	}
      else if (add_addr(inbuf, line))
	{
	  started = 1;
	}
      else
	{
	  fprintf(stderr, "# ignored input \"%s\" line %u\n", inbuf, line);
	}
    }
  threads_finish(nthreads);

  return 0;
}
  


int 
do_sorted(struct sockaddr_in *wherefrom, FILE *infile, int nthreads)
{
  char *cp, opt;
  char inbuf[MAX_INDATA_SZ];
  int n, i;
  int naddr_in = 0;
  int started = 0;
  int port;

  while (get(inbuf, infile))
    {
      if (*inbuf == '#')
	{
	  if (!started)
	    fprintf(ofile, "%s", inbuf);
	  continue; 
	}
      if (!isdigit(*inbuf))
	{
	  if (*inbuf == '-')
	    command(inbuf+1, "vqbBq");
	  else if (!started)
	    fprintf(ofile, "#%s", inbuf);

	  continue; 
	}
      if (strstr(inbuf, "addresses"))
	{
	  if (naddr)
	    {
	      error("two notifications of #addresses found");
	    }
	  else
	    {
	      naddr = atoi(inbuf);
	      train_bufsz = naddr;
	      nthreads = MIN(nthreads, naddr);
	      fprintf(ofile, "# %d addresses\n", naddr);
	      if ((all_addrs = (addr_train_t *)malloc(train_bufsz*sizeof(addr_train_t))) 
		  == NULL)
		error("addr train malloc");
	      threads_init(wherefrom, nthreads, NULL);
	      continue;
	    }
	}
      else
	{
	  char *pp;
	  started = 1;
	  if (all_addrs == NULL)
	    error("no address count found");
	  pp = strchr(inbuf, ':');
	  
	  if (pp)
	    {
	      all_addrs[addr_indx].port = atoi(pp+1);
	      *pp = '\0';
	    }
	  else
	    all_addrs[addr_indx].port = 80;
	  
	  if ((all_addrs[addr_indx].dest.addrint = inet_addr(inbuf)) == INADDR_NONE)
	    {
	      fprintf(stderr, "can't resolve address %s\n", inbuf);
	      continue;
	    }
	  
	  all_addrs[addr_indx].con = con_def;
	  addr_indx++;
	  if (++naddr_in > naddr)
	    break;
	  rthread_inform();
	  
//	      if( addr_indx % 64 == 0 )
//		  fprintf(stderr,"%d/%d (%d) complete\n",
//			  addr_indx, naddr, (addr_indx*100)/naddr );
	}
    }
  
  threads_finish(nthreads);

  fprintf(stderr, "sorting ");
  qsort(all_addrs, naddr, sizeof(addr_train_t), cmp_train);
  fprintf(stderr, "\n");
  /*resolve_to(all_addrs, naddr);*/
  pretty_print_trains(stdout, all_addrs, naddr, sort);

  return;
}


      
