/*  -*- Mode: C;  -*-
 * File: jroute_threads.c
 * Author: James Hall (jch1003@cl.cam.ac.uk)
 * Copyright (C) University of Cambridge Computer Laboratory, 1998
 **~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 ** PACKAGE:
 **
 ** FUNCTION:
 **
 ** HISTORY:
 ** Created: Tue Aug 11 16:04:52 1998 (jch1003)
 ** Last Edited: Wed Oct 28 15:39:40 1998 By James Hall
 **
    $Log: jroute_threads.c,v $
    Revision 1.12  2001/10/24 13:19:29  iap10
    *** empty log message ***

    Revision 1.11  1998/11/11 13:50:28  iap10
    *** empty log message ***

    Revision 1.10  1998/11/11 12:52:52  jch1003
    *** empty log message ***

    Revision 1.9  1998/10/27 10:54:27  iap10
    *** empty log message ***

    Revision 1.8  1998/10/21 10:16:03  jch1003
    *** empty log message ***

    Revision 1.7  1998/10/20 09:54:34  iap10
    *** empty log message ***

    Revision 1.6  1998/10/19 20:26:07  jch1003
    *** empty log message ***

    Revision 1.5  1998/10/12 10:30:42  iap10
    *** empty log message ***

    Revision 1.4  1998/10/09 12:11:27  jch1003
    *** empty log message ***

    Revision 1.3  1998/08/14 11:46:32  jch1003
    *** empty log message ***

 **~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 */


#ifdef linux
#include "linux-tweaks.h"
#endif

#ifdef __osf__
#include <kern/macro_help.h>
#define HAS_PTHREAD_ATTR
#endif


#include <pthread.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#include <sys/socket.h>
#include <sys/time.h>

#include <netinet/in_systm.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <netinet/ip_icmp.h>
#include <netinet/udp.h>
#include <netinet/tcp.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <memory.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>

#include "list.h"
#include "jroute.h"
#include "jroute_threads.h"

addr_train_t *all_addrs = NULL;	/* pool of addrs to do */
int train_bufsz;		/* its size */
int nthreads = N_THREADS_DEF;

/* guarded by routem mutex */
int ww = 0;			/* #router threads waiting */
int inthread_waiting = 0;	/* input thread blocked */
int pool = 0;			/* #addresses waiting */
int addr_indx = 0;		/* next address to put in pool */
int nextaddrt_indx = 0;  	/* next address to trace */

pthread_mutex_t routem;		/* housekeeping guard */
pthread_cond_t routec;		/* associated cv */
/**/

pthread_attr_t attr;		/* thread attributes */
int addrs_to_do;		/* flag - still more to do */

threadata_t *rthreads;		/* route threads data array */

/* guarded by icmpm mutex */
int responses_outstanding = 0;	/* expect icmp responses */
int ithread_waiting = 1;
int done = 0;
pthread_mutex_t icmpm;
pthread_cond_t icmpc;		/* associated cv */

/**/

/* guarded by writem mutex */
int writing = 0;
int write_waiting = 0;

pthread_mutex_t writem;
pthread_cond_t writec;

/**/

pthread_t icmpthread;		/* icmp receiver thread */

/* global counters */
int nreached = 0;
int nunreachable = 0;
int nprohibited = 0;
int nttlexpired = 0;
int nnotagreed = 0;
int *nhop_dist;

int probes_sent = 0;
int srsent = 0;
long probes_sentb = 0L;
int facks_sent = 0;
long facks_sentb = 0L;
int facks_recd = 0;
long facks_recdb = 0L;
int syn_recd = 0;
long syn_recdb = 0L;
long totb = 0L;
int icmp_recd = 0;		/* not guarded */
long icmp_recdb = 0L;		/* do */

int res_sent = 0;
long res_sentb = 0L;

/* and their mutex */
pthread_mutex_t globem;



void 
thread_error(int me, char *what, char *msg)
{
  fprintf(stderr, "%s: thread %d %s ", prog, me, what);
  if (errno)
    perror(msg);
  else
    fprintf(stderr, "%s\n", msg);
  mlock(&routem, "thread_error");
  if (--nthreads == 0)
    exit (1);
  munlock(&routem, "thread_error");
  pthread_exit ((pthread_addr_t)1);
}

void 
thread_error2(int me, char *msg1, char *msg2)
{
  fprintf(stderr, "%s: thread %d %s: %s", prog, me, msg1, msg2);
  mlock(&routem, "thread_error");
  if (--nthreads == 0)
    exit (1);
  munlock(&routem, "thread_error");
  pthread_exit ((pthread_addr_t)1);
}

void 
threads_init(struct sockaddr_in *wherefrom, 
	     int nthreads, FILE *inf)
{
  int i;

  /*set up mutexes etc. */
  mutinit(&routem, pthread_mutexattr_default, "routem");
  condinit(&routec, pthread_condattr_default, "routec");
  mutinit(&icmpm, pthread_mutexattr_default, "icmpm");
  condinit(&icmpc, pthread_condattr_default, "icmpc");
  mutinit(&globem, pthread_mutexattr_default, "globem");
  condinit(&writec, pthread_condattr_default, "writec");
  mutinit(&writem, pthread_mutexattr_default, "writem");

  if ((nhop_dist = (int *)malloc(TTL_MAX_DEF*sizeof(int))) == NULL)
    error("nhop_dist malloc");
  bzero((char *)nhop_dist, TTL_MAX_DEF*sizeof(int));

#ifdef HAS_PTHREAD_ATTR
  if (pthread_attr_create(&attr) == -1)
     error("attr_create");
  if (pthread_attr_setinheritsched(&attr, PTHREAD_DEFAULT_SCHED) == -1)
     error("attr_setinheritsched");
  if (pthread_attr_setsched(&attr, SCHED_RR) == -1)
     error("attr_setsched");
  /* start up icmp receiver thread */
  if (pthread_create(&icmpthread, attr, 
			 (pthread_startroutine_t)ithread,
			 (pthread_addr_t)(long)nthreads) < 0)
	error("icmpthread_create");
#else
  /* start up icmp receiver thread */
  if (pthread_create(&icmpthread, NULL, 
			 (pthread_startroutine_t)ithread,
			 (pthread_addr_t)(long)nthreads) < 0)
	error("icmpthread_create");

#endif


  /* start up route threads */
  addrs_to_do = 1;

  if ((rthreads = (threadata_t *)malloc(nthreads*sizeof(threadata_t))) == NULL)
    error("rthreads malloc");

  for (i = 0; i < nthreads; i++)
    {
      threadarg_t *arg = &rthreads[i].args;
      arg->me = i;
      arg->wherefrom = wherefrom;
      arg->inf = inf;
      mutinit(&rthreads[i].dm, pthread_mutexattr_default, "dm");
      condinit(&rthreads[i].dc, pthread_condattr_default, "dc");
      rthreads[i].valid = 0;

#ifdef HAS_PTHREAD_ATTR
      if (pthread_create(&rthreads[i].thread, attr, 
			 (pthread_startroutine_t)rthread,
			 (pthread_addr_t)arg) < 0)
	error("rthread_create");
#else

      if (pthread_create(&rthreads[i].thread, NULL, 
			 (pthread_startroutine_t)rthread,
			 (pthread_addr_t)arg) < 0)
	error("rthread_create");
#endif
    }

  return;
}

/* 
 * When sorting input thread informs router threads that there's another to do 
 */

void 
rthread_inform()
{
  int me = -1;

  TRC(fprintf(stderr, "inform\n");)

  mlock(&routem, "inform routem");
  pool++;
  if (ww)
    condsig(&routec, "inform");
  munlock(&routem, "inform routem");
  
  return;
};

#define TTRC(x) x

/*
 * When not sorting input thread adds another address to waiting pool. 
 */

int  
add_addr(char *ostring, uint line)
{
  int me = -1;
  char *pp;
  uint16 port;
  uint32 addr;
  src_route_t route_try;
  int min_ttl, max_ttl;
  char astring[255];

  if (strlen(ostring) > 254) ostring[254] = 0;
  strcpy(astring, ostring);
  
  if (pp = strchr(astring, ':')) /* optional port No. */
    {
      *pp = '\0';		/* delimit route string */
      port = strtoul(++pp, &pp, 0) & 0xffff;
    }
  else
    {
      port = 80;
    }

  if (pp = strchr(pp ? pp : astring, '/')) /* at least min ttl specified */
    {
      *pp = '\0';		/* delimit route string */
      min_ttl = strtoul(++pp, &pp, 0) & 0xff;
      if (min_ttl == 0)
	min_ttl = con_def.ttl_min;	/* current default */
    }
  else
    {
      min_ttl = con_def.ttl_min;	/* current default */
    }

  if (pp && (pp = strchr(pp, '/'))) /* max ttl as well? */
    {
      max_ttl = strtoul(++pp, NULL, 0) & 0xff;
      if (max_ttl == 0)
	max_ttl = con_def.ttl_max;	/* current default */
    }
  else
    {
      max_ttl = con_def.ttl_max;	/* current default */
    }

  if (pp = strchr(astring, '=')) /* single probe */
    {
      *pp = '\0';		/* delimit route string */
      min_ttl = max_ttl = strtoul(++pp, NULL, 0) & 0xff;
//fprintf(stderr, "SINGLE %d\n",min_ttl);
      if (max_ttl == 0)
      {
	  min_ttl = con_def.ttl_min;	/* current default */
	  max_ttl = con_def.ttl_max;	/* current default */
      }
    }

  if (min_ttl < TTL_MIN_DEF || min_ttl > TTL_MAX 
      || max_ttl < min_ttl || max_ttl > TTL_MAX)
    {
      fprintf(stderr, "# problem with specified ttl line %u\n", line);
      return 0;
    }

  if (pp = strchr(pp ? pp : astring, '$')) /* if not NULL label specified */
    {
      *pp = '\0';
      pp++;
      if (strlen(pp) >= TR_LABEL_LEN -1)
	pp[TR_LABEL_LEN -1] = '\0';
    }

  if (!parseroute(&route_try, astring))
    return 0;

  if (route_try.num == 1)
    {
      addr = route_try.rt[0].addrint;
      route_try = con_def.route;
    }
  else
    {
      addr = route_try.rt[route_try.num-1].addrint;
      route_try.rt[route_try.num-1].addrint = 0U;
      route_try.num--;
    }

  mlock(&routem, "add_addr routem");
  while (pool == train_bufsz || all_addrs[addr_indx].in_use)
    {
      TTRC(fprintf(stderr, "# inthread block %d: %s (used %d/%d)\n", 
		   addr_indx, 
		   pool == train_bufsz ? "pool full"
		   : "slot in use", 
		   pool, train_bufsz);)

      inthread_waiting = 1;
      condwait(&routec, &routem, "add_addr routec");
    }
  inthread_waiting = 0;
  pool++;
  all_addrs[addr_indx].in_use = 0;
  all_addrs[addr_indx].dest.addrint = addr;
  all_addrs[addr_indx].port = port;
  all_addrs[addr_indx].con.ttl_max = max_ttl;
  all_addrs[addr_indx].con.ttl_min = min_ttl;
  all_addrs[addr_indx].con.route = route_try;
  if (pp)
    strcpy(all_addrs[addr_indx].label, pp);
  else
    all_addrs[addr_indx].label[0] = '\0';

  //fprintf(stderr, "LABEL %s\n",all_addrs[addr_indx].label);

  addr_indx = (++addr_indx) % train_bufsz;
  if (ww)
    condsig(&routec, "add_addr");
  munlock(&routem, "add_addr routem");
  
  return 1;
}  

void 
threads_finish(int nthreads)
{
  int i;
  long int val;
  int me = 0;

  addrs_to_do = 0;

  mlock(&routem, "finish routem");
  if (ww)
    condbroad(&routec, "finish");
  munlock(&routem, "finish routem");

  for (i = 0; i < nthreads; i++)
    {
      tjoin(rthreads[i].thread, &val, "rthread finish");
      if (verbose)
	fprintf(stderr, "# thread #%d terminated %s\n",  i, 
		val == 0 ? "normally" : "with error");
    }
  return;
}

/*
 * Loop for threads getting routes 
 */

int 
rthread(pthread_addr_t arg)
{
  int me = ((threadarg_t *)arg)->me;
  addr_train_t *train;
  threadata_t *tdata = &rthreads[me];

  tdata->wherefrom = *(((threadarg_t *)arg)->wherefrom);

  while (1)
    {
      mlock(&routem, "rthread routem");
      ww++;
      while (pool == 0)
	{
	  if (addrs_to_do)
	    {
	      if (inthread_waiting)
		condsig(&routec, "rthread");
	      condwait(&routec, &routem, "rthread routec");
	    }
	  else
	    {
	      ww--;
	      munlock(&routem, "rthread routem - finished");
	      done++;
	      goto out;
	    }
	}     
      train = &all_addrs[nextaddrt_indx];
      train->in_use = 1;
      nextaddrt_indx = (++nextaddrt_indx) % train_bufsz;
      pool--;
      ww--;
      munlock(&routem, "rthread routem");

      if( get == get_tnet )
      {
	  /* check for special `die' file */
	  char *die = "/tmp/jroute-die";
	  struct stat b;
	  
	  if( stat( die, &b ) == 0 )
	  {
	      /* Time to die. Let's just bail and hope the kernel clears 
		 up after us OK. */
	      exit(-1);
	  }
      }
      
      if (maxrate)		/* delay to limit bw generated */
	rthread_wait(me);       

      if( train->con.ttl_max == train->con.ttl_min)
	  getsingle(me, train, tdata);
      else
	  getroute(me, train, tdata);

      mlock(&routem, "rthread2 routem");
      train->in_use = 0;
      if (inthread_waiting && pool < train_bufsz/2)
	condsig(&routec, "rthread");
      munlock(&routem, "rthread2 routem");
    }
 
 out:
  return 0;
}

/*
 * Loop for thread receiving icmp packets
 */

int 
ithread(pthread_addr_t arg)
{
  int i;
  int me = -2;			/* distinct id for this'un */
  int on = 1;
  int ngot = 0;
  int rd;
  int nthreads = (long)arg & 0xffffffff;
  fd_set iset;
  char packet[ICMP_BUFSZ];	/* last inbound (icmp) packet */
  struct sockaddr_in from;	/* and where from */
  int fromlen = sizeof(from);


  struct icmp *icp;
  u_char type, code;
  int hlen;
  struct ip *ip;
  uchar ret_ttl;
  
  while (1)
    {
      mlock(&icmpm, "sthread icmpm loop");
      responses_outstanding -= ngot;
      while (!responses_outstanding)
	{
	  if (!done)
	    {
	      ithread_waiting++;
	      condwait(&icmpc, &icmpm, "ithread loop");
	    }
	  else
	    {
	      munlock(&icmpm, "ithread loop");
	      goto out;
	    }
	}
      ithread_waiting = 0;
      munlock(&icmpm, "sthread icmpm");
      ngot = 0;
      
      if  ((rd = recvfrom(s, packet, ICMP_BUFSZ, 0, 
		    (struct sockaddr *)&from, &fromlen)) < 1)
	error("ithread recvfrom");
      TRC(fprintf(stderr, ".");)

      icmp_recd ++;
      icmp_recdb += rd;
	
      ip = (struct ip *)packet;
      hlen = ip->ip_hl << 2;
      if (rd < hlen + ICMP_MINLEN) /* no good */
	continue;
      rd -= hlen;
      ret_ttl = ip->ip_ttl;
      icp = (struct icmp *)(packet + hlen);
      type = icp->icmp_type;
      code = icp->icmp_code;

#define TRCX(x)
TRCX(fprintf(stderr,"Got ICMP %d\n",type);)


      if ((type == ICMP_TIMXCEED && code == ICMP_TIMXCEED_INTRANS) ||
	  type == ICMP_UNREACH) 
	{
	  struct ip *hip;
	  struct tcphdr *tp;
	  
	  hip = &icp->icmp_ip;
	  hlen = hip->ip_hl << 2;
	  
	  tp = (struct tcphdr *)((u_char *)hip + hlen);

	  

	  /* XXX 12 is a magic number */
	  if (hlen + 12 <= rd &&
	      hip->ip_p == IPPROTO_TCP)
	    {
	      int thread = WHICH_RTHREAD(ntohs(tp->th_sport));
	      if (thread < nthreads)
		{
		  threadata_t *data = &rthreads[thread];
		  mlock(&data->dm, "ithread dm");
		  if (!data->valid 
		      //&& data->whereto.sin_addr.s_addr 
		      //== hip->ip_dst.s_addr 
		      && data->whereto.sin_port == tp->th_dport
		      && data->wherefrom.sin_port == tp->th_sport
		      )
		    {
		      /* got a valid response for this thread */
		      ngot++;
		      data->valid = type == ICMP_TIMXCEED ? 
			PROBE_TTL_EXCEEDED : code + 1;
		      data->icmp_from.addrin = from.sin_addr;
		      data->ret_ttl = ret_ttl;
		      data->tcplen = hip->ip_len - hlen;
		      if (data->whereto.sin_addr.s_addr != hip->ip_dst.s_addr)
			data->valid += PROBE_TEST_RELAXED;

                      /* is there a SRC route option here ? */
		      
		      data->icmp_src_route.num = 0;

		      if ( hlen > 5 )
		      {
			  uchar * optr = ((uchar *) hip) + 20;
			  uchar pad  = *optr;
			  uchar lsr  = *(optr+1);
			  uchar len  = *(optr+2);
			  uchar ptr  = *(optr+3);
			  int i,j;
#define XTRC(x)  
			  XTRC(fprintf(stderr, 
				      "ICMP: got an IP option: pad %d, opt %d, len %d, ptr %d\n", pad, lsr, len, ptr);)
			      
			  data->icmp_src_route.ptr = (ptr-4)/4;

			  for( i = 4, j=0; i < (len+1); i+=4, j++ )
			  {
			      uint32 a = * (uint32 *)(optr+i);
			      XTRC(fprintf(stderr,"ICMP: %d = %s\n",
					  j, 
					  inet_ntoa( *((struct in_addr *) 
						       (&a)))
					  );)
			      data->icmp_src_route.rt[j].addrint = a;
			      data->icmp_src_route.num++;
			  }
			      
			      
		      }


		      condsig(&data->dc, "ithread dc");
		      TRC(fprintf(stderr, "->%d\n", thread);)
		    } 
		  munlock(&data->dm, "ithread dm");
		}
	    }
	}
    } /* end while{1} */
 out:
  return 0;
}

void 
write_lock(int me)
{
  mlock(&writem, "writelock writem");
  write_waiting++;
  while (writing == 1)
    condwait(&writec, &writem, "writelock writec");

  writing++;
  munlock(&writem, "writelock writem");

  return;
}

void 
write_unlock(int me)
{
  mlock(&writem, "write_unlock writem");
  writing = 0;
  if (--write_waiting)
    condsig(&writec, "write_unlock writec");

   munlock(&writem, "write_unlock writem");

  return;
}

void 
rthread_wait(int me)
{
  static pthread_mutex_t m;
  static pthread_cond_t c;
  static int started = 0;
  timespec_t abs;

  double sent;
  double st;
  double wt;
  double ip;
  double fp;

  if (!started)
    {
      started ++;
      mutinit(&m, pthread_mutexattr_default, "rthread_wait");
      condinit(&c, pthread_condattr_default, "rthread_wait");
    }
  
  st = start.tv_sec + start.tv_usec/1000000;

  if (maxrate > 0)		/* limiting on bw */
    {
      sent = (double)((totb*8)/1000); /* working in Kb/s */
      wt = (sent/maxrate) + st;
      fp = modf(wt, &ip);
    }
  else				/* limiting on lsr pkts/s */
    {
      wt = st - (srsent/maxrate);	/* maxrate is negative */
      fp = modf(wt, &ip);
    }

  abs.tv_sec = (long)ip;
  abs.tv_nsec = (long)(fp*1000000000);
  
  mlock(&m, "rthread_wait");
  cond_timedwait(&c, &m, &abs, "rthread_wait");
  munlock(&m, "rthread_wait");
  
  
  return;
}

void 
update_globals(int me, addr_train_t *t)
{
  uint32 status = t->status;
  mlock(&globem, "update globemem");
  
  if (status & GOT_THERE)
    nreached++;
  else
    nttlexpired++;
  if (status & NO_AGREEMENT)
    nnotagreed++;
  if (status & PROHIB_UNREACH)
    nprohibited++;
  if (status & UNREACHABLE)
    nunreachable++;

  nhop_dist[t->nhops-1]++;

  /* 
   * catch dubious len returns from routers - assume max likely tcp opts
   * size is 8 - mss, wscale and padding 
   */

  if (t->tcplen > 28)
    t->tcplen = 28;
  if (t->tcplen < 20)
    t->tcplen = 20;

  probes_sent += (t->sent + t->srsent);
  probes_sentb += ((t->sent + t->srsent)*(t->iphlen+t->tcplen));

  srsent += t->srsent;
  syn_recd += t->srcd;

  /* 
   * return syns don't include any outgoing ip opts 
   * this will probably be an over-estimate as probe tcp part is likely 
   * to contain more tcp opts than returning syn
   */
  syn_recdb += (t->srcd*(t->tcplen + sizeof(struct ip)));

  /* 
   * close is called on every socket after the probe is sent - for probes 
   * not reaching the final destination (i.e. sockets not in the connected 
   * state) this does not result in a FIN being sent in most TCP 
   * implementations (exception solaris?) so we assume our traffic generated 
   * by closes to be one FIN and one ACK packet per incoming SYN
   *
   * also allow here for our ACKs to received SYNs 
   */

  /* TODO fix this for solaris - will be a lot more FINs and RSTs flying */

  facks_sent += (t->srcd*3);	/* 1 ACK for their SYN 
				   + 1 FIN + 1 ACK for their FIN */
  /* will contain any ip but no tcp opts */
  facks_sentb += (t->srcd*3*(t->iphlen+ sizeof(struct tcphdr)));

  /*
   * we get one FIN and probably one ACK (to our FIN) for each SYN returned 
   * - no ip/tcp opts in either
   */

  facks_recd += (t->srcd*2);
  facks_recdb += (t->srcd*2*(sizeof(struct ip) + sizeof(struct tcphdr)));

  if (maxrate)			/* need running total */
    totb = probes_sentb + facks_sentb + icmp_recdb + syn_recdb + facks_recdb 
      + res_sentb;

  naddr_done++;

  munlock(&globem, "update globemem");

  return;
}

void 
report(struct timeval *start, struct timeval *finish)
{
  int i;
  FILE *f = stdout;
  float fperiod = 0.0;
  int tot_in, tot_out, tot;
  long tot_inb, tot_outb, totb;

  if (start->tv_usec > finish->tv_usec)
    {
      start->tv_sec++;
      finish->tv_usec += 1000000;
    }
  fperiod += (finish->tv_sec - start->tv_sec);
  fperiod += ((finish->tv_usec - start->tv_usec)/1000000);
  /* avoid course clock granularity problems for short runs */ 
  if (fperiod == 0.0)
    fperiod = 0.001;

  fprintf(f, "\n# %d addresses\n", naddr_done);
  if (naddr && naddr_done != naddr)
    fprintf(f, "# *** input file claime %d to do ***\n", naddr);
  fprintf(f, "#\t%d reached\n", nreached);
  fprintf(f, "#\t%d not reached\n", nttlexpired);
  fprintf(f, "#\t%d unreachable\n", nunreachable);
  fprintf(f, "#\t\t(of which %d prohibited)\n", nprohibited);
  fprintf(f, "#\t%d no agreed outcome \n\n", nnotagreed);

  fprintf(f, "#\thop distribution:\n\n");
  fprintf(f, "#\thops    hosts\n");
  for (i = 0; i < TTL_MAX_DEF; i++)
    if (nhop_dist[i])
      fprintf(f, "#\t%2d%10d\n", i + 1, nhop_dist[i]);

  fprintf(f, "\n# period %.4fs.\n", fperiod);

  tot_out = probes_sent + facks_sent + res_sent;
  tot_outb = probes_sentb + facks_sentb + res_sentb;
  tot_in = icmp_recd + syn_recd + facks_recd;
  tot_inb = icmp_recdb + syn_recdb + facks_recdb;
  tot = tot_in + tot_out;
  totb = tot_inb + tot_outb;
  

  fprintf(f, "\n# Traffic generated:-\n\n");
  fprintf(f, "# %-15s%8s%10s%10s%8s%10s\n\n", "type", "pkts.", "bytes", 
	  "ave.size", "pkts/s", "Kb/s");
  
  if (probes_sent)
    fprintf(f, "# %-15s%8d%10ld%10d%8.0f%10.2f\n", "probes sent", 
	    probes_sent, probes_sentb, probes_sentb/probes_sent, 
	    probes_sent/fperiod,  (probes_sentb*8)/(fperiod*1000));

  if (srsent)
    fprintf(f, "# (%d loose source routed)\n", srsent);
  
  if (facks_sent)
    fprintf(f, "# %-15s%8d%10ld%10d%8.0f%10.2f\n", "FINS/ACKS sent", 
	    facks_sent, facks_sentb, facks_sentb/facks_sent, 
	    facks_sent/fperiod,  (facks_sentb*8)/(fperiod*1000));
  
  if (res_sent)
    fprintf(f, "# %-15s%8d%10ld%10d%8.0f%10.2f\n", "results sent", 
	    res_sent, res_sentb, res_sentb/res_sent, 
	    res_sent/fperiod,  (res_sentb*8)/(fperiod*1000));
  
  if (icmp_recd)
    fprintf(f, "# %-15s%8d%10ld%10d%8.0f%10.2f\n", "icmp responses", 
	    icmp_recd, icmp_recdb, icmp_recdb/icmp_recd, 
	    icmp_recd/fperiod,  (icmp_recdb*8)/(fperiod*1000));
  
  if (syn_recd)
    fprintf(f, "# %-15s%8d%10ld%10d%8.0f%10.2f\n", "syns received", 
	    syn_recd, syn_recdb, syn_recdb/syn_recd, 
	    syn_recd/fperiod,  (syn_recdb*8)/(fperiod*1000));
  
  if (facks_recd)
    fprintf(f, "# %-15s%8d%10ld%10d%8.0f%10.2f\n\n", "FINS/ACKS rec'd", 
	    facks_recd, facks_recdb, facks_recdb/facks_recd, 
	    facks_recd/fperiod,  (facks_recdb*8)/(fperiod*1000));
  
  if (tot_out)
    fprintf(f, "# %-15s%8d%10ld%10d%8.0f%10.2f\n", "total out", 
	    tot_out, tot_outb, tot_outb/tot_out, 
	    tot_out/fperiod,  (tot_outb*8)/(fperiod*1000));
  
  if (tot_in)
    fprintf(f, "# %-15s%8d%10ld%10d%8.0f%10.2f\n\n", "total in", 
	    tot_in, tot_inb, tot_inb/tot_in, 
	    tot_in/fperiod,  (tot_inb*8)/(fperiod*1000));
  
  if (tot)
    fprintf(f, "# %-15s%8d%10ld%10d%8.0f%10.2f\n\n", "total traffic", 
	    tot, totb, totb/tot, 
	    tot/fperiod,  (totb*8)/(fperiod*1000));
  
  fprintf(f, "\n");

  return;
}
    
  
  
  
